/*
 * Copyright 1992 The University of Newcastle upon Tyne
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation for any purpose other than its commercial exploitation is
 * hereby granted without fee, provided that the above copyright notice
 * appear in all copies and that both that copyright notice and this
 * permission notice appear in supporting documentation, and that the name of
 * The University of Newcastle upon Tyne not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission. The University of Newcastle upon Tyne makes no
 * representations about the suitability of this software for any purpose. It
 * is provided "as is" without express or implied warranty.
 * 
 * THE UNIVERSITY OF NEWCASTLE UPON TYNE DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF NEWCASTLE UPON TYNE BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR
 * IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 * 
 * Author:  Gerry Tomlinson (gerry.tomlinson@newcastle.ac.uk) Computing
 * Laboratory, University of Newcastle upon Tyne, UK
 */

/*
 * search.c
 * 
 * routines to search through a string
 * 
 */

#include <ctype.h>
#include <X11/Intrinsic.h>
#include "regexp.h"

static Boolean regexperr = FALSE;
char *regexperrmess;

/*
 * linesearch
 * 
 * search for line  & return NULL for failure or beginning of line for success
 * 
 */
char *
linesearch(s, line)
char *s;
int line;
{
	char *lptr = s;
	int n = 1;

	if (!s || line <= 0)
		return NULL;

	while (*s && n < line)
		if (*s++ == '\n')
			if (*s) {
				n++;
				lptr = s;
			}
	return line == n ? lptr : NULL;
}

/*
 * lastlinesearch
 * 
 * search for beginning of last line given start + size
 * 
 */

char *
lastlinesearch(start, size)
char *start;
int size;
{
	char *e;

	if (!start)
		return NULL;

	e = start + size;

	if (*(e - 1) == '\n')
		e--;
	while (--e > start)
		if (*e == '\n')
			return e + 1;

	return start;
}

void
regerror(s)
char *s;
{
	regexperrmess = s;
	regexperr = TRUE;
}

/*
 * ciregexec
 * 
 * simple-minded regexec with caseinsensitive option
 */
int
ciregexec(prog, string, caseinsens)
regexp *prog;
char *string;
Boolean caseinsens;
{
	static char *n = NULL;
	char *p, *s;
	int retcode;

	if (caseinsens) {
		if (!n)
			n = XtNewString(string);
		else {
			n = XtRealloc(n, (unsigned) strlen(string) + 1);
			n[strlen(string)] = '\0';
		}
		p = n;
		s = string;
		while (*s) {
			*p = *s;
			*s = isupper(*s) ? tolower(*s) : *s;
			s++;
			p++;
		}

		retcode = regexec(prog, string);

		p = n;
		s = string;
		while (*s)
			*s++ = *p++;

		return retcode;

	} else
		return regexec(prog, string);
}

/*
 * regsearch
 * 
 * regular express search routine
 * 
 * given a null terminated string (data) finds the longest match between
 * newlines, starting from start
 * 
 * forward/backwards search , wrap at eof/no wrap at eof (for backwards search,
 * start == data => eof)
 * 
 * repeats search of previous pattern
 * 
 * note only dependency upon X are type Boolean, XtFree & XtNewString
 */
int
regsearch(data, start, exp, forward, wrap, caseinsens, startmatch, endmatch)
char *data;
char *start;
char *exp;
Boolean forward, wrap, caseinsens;
char **startmatch, **endmatch;
{
	static regexp *compreg;
	static char *prevexp;
	static Boolean prevforward;
	static char *frontpos, *backpos;	/* of successful match */
	Boolean sameexp = FALSE;/* true when repeating pattern */
	Boolean reversing = FALSE;	/* true  if reversing direction with
					 * same pattern without moving caret */
	char c, *s, *e, *bof, *eof;

	int length;

	regexperrmess = NULL;

	if (data == NULL)
		return (FALSE);

	length = strlen(data);

	bof = data;
	eof = bof + length;


	if (*exp == '\0')
		exp = prevexp;

	if (prevexp && strcmp(exp, prevexp) == 0)
		sameexp = TRUE;

	XtFree(prevexp);
	prevexp = XtNewString(exp);

	if ((forward != prevforward) && sameexp &&
		((forward && start == frontpos) || (!forward && start == backpos)))
		reversing = TRUE;

	prevforward = forward;

	regexperr = FALSE;
	if (caseinsens && exp) {
		s = e = exp;
		while (*s) {
			*s++ = isupper(*e) ? tolower(*e) : *e;
			e++;
		}
	}
	XtFree((char *)compreg);
	compreg = regcomp(exp);
	if (regexperr)
		return (FALSE);



	if (forward) {

		/* search forward from start to eof */

		if (reversing)
			start = backpos;

		s = start;

		/* start on next line to match ^ if not at ^ */
		if (exp[0] == '^' && s != bof && (*(s - 1) != '\n')) {
			while (s < bof + length && *s++ != '\n');
			if (s == bof + length)
				s++;
		}
		e = s;

		while (s < bof + length) {
			while (*e != '\n' && *e)
				e++;
			c = *e;
			*e = '\0';
			if (ciregexec(compreg, s, caseinsens)) {
				*e = c;
				*startmatch = frontpos = compreg->startp[0];
				*endmatch = backpos = compreg->endp[0];
				return (TRUE);
			}
			*e = c;
			s = ++e;
		}

		if (wrap) {	/* search forward from beginning of file
				 * (bof) to start */

			s = bof;
			e = s;

			while (s <= start) {
				while (*e != '\n' && *e)
					e++;
				c = *e;
				*e = '\0';
				if (ciregexec(compreg, s, caseinsens)) {
					*e = c;
					*startmatch = frontpos = compreg->startp[0];
					*endmatch = backpos = compreg->endp[0];
					return (TRUE);
				}
				*e = c;
				s = ++e;
			}

		}
		return (FALSE);	/* failed forward search */
	}
	/* search back from start to beginning of file */

	if (reversing)
		start = frontpos;
	else if (start == data)
		start = bof + length;

	s = start;

	/* start on previous line to match $ if not at $ */
	if (exp[strlen(exp) - 1] == '$' && *s && *s != '\n') {
		while (s > bof && *s != '\n')
			s--;
		if (s == bof)
			s--;
	}
	e = s;
	while (s >= bof) {
		c = *s;
		*s = '\0';
		while (*e != '\n' && e >= bof)
			e--;
		if (ciregexec(compreg, e + 1, caseinsens)) {
			frontpos = compreg->startp[0];
			backpos = compreg->endp[0];
			if (exp[0] != '^' && exp[strlen(exp) - 1] != '$') {	/* keep looking on this
										 * line */
				while (ciregexec(compreg, frontpos + 1, caseinsens) &&
					((compreg->endp[0] - compreg->startp[0]) >= backpos - frontpos)) {
					frontpos = compreg->startp[0];
					backpos = compreg->endp[0];
				}
			}
			*s = c;
			*startmatch = frontpos;
			*endmatch = backpos;
			return (TRUE);
		}
		*s = c;
		s = e;
	}

	if (wrap) {		/* search back from eof to caret */

		s = e = eof - 1;
		while (s >= start) {
			c = *s;
			*s = '\0';
			while (*e != '\n' && e >= start)
				e--;
			if (ciregexec(compreg, e + 1, caseinsens)) {
				frontpos = compreg->startp[0];
				backpos = compreg->endp[0];
				if (exp[0] != '^' && exp[strlen(exp) - 1] != '$') {	/* keep looking on this
											 * line */
					while (ciregexec(compreg, frontpos + 1, caseinsens) &&
						((compreg->endp[0] - compreg->startp[0]) >= backpos - frontpos)) {
						frontpos = compreg->startp[0];
						backpos = compreg->endp[0];
					}
				}
				*s = c;
				*startmatch = frontpos;
				*endmatch = backpos;
				return (TRUE);
			}
			*s = c;
			s = e;
		}
	}
	return (FALSE);		/* failed backward search */

}
