/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   xmc.c								*
*									*
*	X Multiplexor control protocol support routines.		*
*									*
************************************************************************/
#include <sys/time.h>
#include <sys/types.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#include <X11/X.h>
#include <X11/Xproto.h>

#include <xmc.h>
#include <xmcp.h>

#include "xmx.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "zb.h"
#include "res.h"
#include "incl/xmc.pvt.h"
#include "fd.h"

#define DENY_BUSY	"Server busy"
#define DENY_VERSION	"Protocol version mismatch"
#define DENY_AUTH	"Client is not authorized to connect to Server"

static buffer_t *xbp;

static xmc_t *conns;	/* xmc connections */

/************************************************************************
*									*
*   xmc_init								*
*									*
************************************************************************/
void
xmc_init
   VOID
{
   xbp = buf_new(B_STATIC);
}

/************************************************************************
*									*
*   xmc_alloc								*
*									*
*	Allocate an xmc structure.					*
*									*
************************************************************************/
xmc_t *
xmc_alloc
   AL((fd, ap))
   DB int fd
   DD hostaddr_t *ap
   DE
{
   xmc_t *xp;

   if (MALLOC(xp, xmc_t *, sizeof(xmc_t)))
      return 0;

   xp->fd = fd;
   xp->qp = queue_new(Q_XMCCLIENT, (char *)xp);
   xp->accepted = 0;
   xp->swap = 0;
   xp->seqno = 0;
   xp->base = 0;
   xp->eventmask = 0;
   xp->addr.family = ap->family;
   if (xp->addr.length = ap->length) {
      if (MALLOC(xp->addr.address, char *, ap->length))
         return 0;
      bcopy(ap->address, xp->addr.address, ap->length);
   }
   else
      xp->addr.address = 0;

   xp->next = conns;
   conns = xp;

   return xp;
}

void
xmc_free
   AL((xp))
   DB xmc_t *xp
   DE
{
   register xmc_t *txp;

   window_xmc_client_death(xp);		/* remove input masks */

   if (xp->qp)
      queue_free(xp->qp);
   if (xp->addr.address) {
      free(xp->addr.address);
      xp->addr.length = 0;
      xp->addr.address = 0;
   }
   if (xp == conns)
      conns = xp->next;
   else {
      for (txp=conns; txp; txp=txp->next)
         if (xp == txp->next) {
            txp->next = xp->next;
            break;
         }
      if (txp == 0)
         warn("xmc_free: connection structure not found...that's bad\n");
   }
   free(xp);
}

/************************************************************************
*									*
*   xmc_connect								*
*									*
************************************************************************/
int
xmc_connect
   AL((xp, bp))
   DB xmc_t *xp
   DD buffer_t *bp
   DE
{
   register int len, rv = -1;
   u16_t namelen, datalen;
   register chunk_t *chp;
   register tp_t *tpp;
   xmcClientConn *p;
   char *msg;
   static u32_t base;

   if (buf_active(bp) < sz_xmcClientConn)
      return 0;		/* not an error, just wait for more */

   p = (xmcClientConn *)buf_curdata(bp);
   if (p->byteOrder != endian) {
      CPSWAP2(p->nBytesAuthName, namelen);
      CPSWAP2(p->nBytesAuthData, datalen);
   }
   else {
      namelen = p->nBytesAuthName;
      datalen = p->nBytesAuthData;
   }
   len = sz_xmcClientConn + namelen + PAD(namelen) + datalen + PAD(datalen);

   if (buf_active(bp) < len)
      return 0;		/* not an error, just wait for more */

   chp = buf_split(bp, len);
   p = (xmcClientConn *)buf_data(chp);

   if (xp->swap = (p->byteOrder == endian) ? 0 : 1) {
      SWAP2(p->majorVersion);	/* encapsulate in a routine? swap (TODO?) */
      SWAP2(p->minorVersion);
      p->nBytesAuthName = namelen;
      p->nBytesAuthData = datalen;
   }
   if (xp->accepted == 0)
      if (	access_get() == 0 ||
		access_check(	xp->addr.family,
				xp->addr.length,
				xp->addr.address) ||
		auth_check_local(p->nBytesAuthName,
				(char *)(p + 1),
				p->nBytesAuthData,
				(char *)(p + 1) +
					p->nBytesAuthName +
					PAD(p->nBytesAuthName))) {

         if (	p->majorVersion == XMC_MAJOR_VERSION &&
		p->minorVersion <= XMC_MINOR_VERSION) {
            xp->accepted = 1;
            xp->base = (++base << RID_MASKBITS) & RID_BASE_MASK,
            tpp = tptr_default();
            xmcp_ServerConn(	xbp,
				(base << RID_MASKBITS) & RID_BASE_MASK,
				RID_MASK,
				config_mode,
				tpp->id);
            rv = 0;
         }
         else
            msg = DENY_VERSION;
      }
      else
         msg = DENY_AUTH;
   else
      msg = DENY_BUSY;

   if (rv)	/* connection denial */
      xmcp_ServerDeny(xbp, strlen(msg)+1, msg);

   buf_clear(chp);

   if (buf_active(xbp)) {
      chp = buf_split(xbp, 0);
      queue_add(xp->qp, chp);
      buf_clear(chp);
   }
   return rv;
}

/************************************************************************
*									*
*   xmc_close								*
*									*
************************************************************************/
void
xmc_close
   AL((xp))
   DB xmc_t *xp
   DE
{
   main_close_conn(xp->fd);
   xmc_free(xp);
}

/************************************************************************
*									*
*   xmc_reset								*
*									*
*	Perform a server reset.  Sever all clients and destroy their	*
*	resources.  Reset all internal state.  If recalculating the	*
*	virtual configuration, clear it.  Then start a "client zero"	*
*	dialog with the servers leading to an initial server state.	*
*									*
************************************************************************/
void
xmc_reset
   AL((xp, level))
   DB xmc_t *xp
   DD int level
   DE
{
   register int i, n;
   register chunk_t *chp;
   int reconfig = 0;

   if (level != R_SOFT) {
      access_reset(opt.dpy, auth_on());	/* reset access list */

      for (i=0; i<=MAXFD; i++)		/* sever clients */
         if (cv[i].type == CLIENT)
            main_close_conn(i);

      client_reset();			/* clears all clients */

      sel_reset();

      /* queue requests to destroy all client resources */

      zb_dest(ZD_ALLSERV, 0);
      inp_reset();
      window_reset();
      pixmap_reset();
      gc_reset();
      cursor_reset();
      font_reset();
      color_reset();

      zb_queue(0);

      hash_reset(vmap);			/* clear hash table */

      util_id_reset();

      if (level == R_RECONFIG) {
#ifdef TODO
         cliz_unsetup();	/* destroy server resources on servers */
#endif
         queue_flush_servers();		/* <- could be elsewhere...TODO */
         sres_reset();	/* nuke server resources */
         vconf_init();	/* nuke virtual configuration */
      }
   }
   /*
   ** could do a sanity check here or above to see if we are
   ** already in a reset. (TODO)
   */
   for (i=0; i<num_serv; i++)
      if (servers[i]->state == S_NAMED)
         if (xmc_add(servers[i]))
            server_free(servers[i--]);	/* slides array left */

#ifdef NOT_RIGHT_TODO
   if ( !cliz_resetting() && level == R_RECONFIG) {
      vconf_reset();	/* need to handle failure TODO */
      cliz_setup1();
   }
#endif
}

/************************************************************************
*									*
*   xmc_process								*
*									*
************************************************************************/
void
xmc_process
   AL((xp, bp))
   DB xmc_t *xp
   DD buffer_t *bp
   DE
{
   register int i;
   register char *dp;
   register xmcReq *rp;
   u16_t length;
   register server_t *sp;
   register tp_t *tpp;
   register chunk_t *chp, *nchp, *rchp;

   nchp = buf_chunk(bp);

   for (; buf_chunksize(nchp) >= sz_xmcReq; buf_clear(chp)) {
      rp = (xmcReq *)buf_data(nchp);
      if (xp->swap) {
         CPSWAP2(rp->length, length);
      }
      else
         length = rp->length;

      if (buf_chunksize(nchp) < length)
         break;

      chp = buf_split(bp, length);
      nchp = buf_chunk(bp);

      if (xp->swap)
         swapreq(rp);

      xp->seqno++;

      D_CALL1(D_PROTOx, dprxmc_req, rp);

      switch (rp->code) {
         case XMC_Register:
            {  register xmcRegisterReq *p = (xmcRegisterReq *)rp;
               int len[4];
               char *str[4];

               len[0] = p->nBytesAddress;
               len[1] = p->nBytesName;
               len[2] = p->nBytesURL;
               len[3] = p->nBytesDesc;
               if (get_strings((char *)(p+1), 4, len, str))
                  xmcp_Error(xbp, xp->seqno, ErrAlloc, XMC_Register, 0);
               else
                  xmc_register(p->regID, p->family, p->length, str[0],
					p->port, str[1], str[2], str[3]);
            }
            break;
         case XMC_Unregister:
            {  register xmcUnregisterReq *p = (xmcUnregisterReq *)rp;

               xmc_unregister(p->id);
            }
            break;
         case XMC_SetAuth:
            {  register xmcSetAuthReq *p = (xmcSetAuthReq *)rp;
               int len[2];
               char *str[2];

               len[0] = p->nBytesName;
               len[1] = p->nBytesData;
               if (get_strings((char *)(p+1), 2, len, str))
                  xmcp_Error(xbp, xp->seqno, ErrAlloc, XMC_SetAuth, 0);
               else
                  xmc_set_auth(str[0], str[1], p->pMask);
            }
            break;
         case XMC_GetAuth:
            {  register xmcGetAuthReq *p = (xmcGetAuthReq *)rp;
               int len[2];
               char *str[2];

               len[0] = p->nBytesName;
               len[1] = p->nBytesData;
               if (get_strings((char *)(p+1), 2, len, str))
                  xmcp_Error(xbp, xp->seqno, ErrAlloc, XMC_GetAuth, 0);
               else
                  xmc_get_auth(xp, str[0], str[1]);
            }
            break;
         case XMC_AddDisplay:
            {  register xmcAddDisplayReq *p = (xmcAddDisplayReq *)rp;
               int len[5];
               char *str[5];
               geom_t *geomp = 0;

               len[0] = p->nBytesAddress;
               len[1] = p->nBytesGeom;
               len[2] = p->nBytesTag;
               len[3] = p->nBytesName;
               len[4] = p->nBytesData;
               if (get_strings((char *)(p+1), 5, len, str))
                  xmcp_Error(xbp, xp->seqno, ErrAlloc, XMC_AddDisplay, 0);
               else {
                  if (len[1] && (geomp = util_parse_geom(str[1])) == 0)
                     xmcp_Error(xbp, xp->seqno, ErrGeometry, XMC_AddDisplay, 0);
                  else if ((sp = server_alloc(p->dispID,
					str[2],
					p->nBytesTag,
					p->family,
					p->nBytesAddress,
					str[0],
					p->displayNumber,
					p->screen,
					p->mode,
					p->window,
					geomp)) == 0)
                     xmcp_Error(xbp, xp->seqno, ErrAlloc, XMC_AddDisplay, 0);
                  else {
                     if (len[3])
                        sp->cookie = cookie_new(len[3], str[3], len[4], str[4]);

                     if (xmc_add(sp)) {
                        xmcp_Error(xbp, xp->seqno, ErrConnect, XMC_AddDisplay,
									sp->id);
                        server_free(sp);
                     }
                  }
                  free_strings(5, str);
               }
            }
            break;
         case XMC_SetDisplayTag:
            {  register xmcSetDisplayTagReq *p = (xmcSetDisplayTagReq *)rp;
               int len;
               char *str;

               len = p->nBytesTag;
               if (get_strings((char *)(p+1), 1, &len, &str))
                  xmcp_Error(xbp, xp->seqno, ErrAlloc, XMC_SetDisplayTag, 0);
               else if (sp = server_find(p->dispID))
                  xmc_set_tag(xp, sp, str);
               else {
                  xmcp_Error(xbp,
				xp->seqno,
				ErrDisplay,
				XMC_SetDisplayTag,
				p->dispID);
                  free_strings(1, &str);
               }
            }
            break;
         case XMC_QueryDisplay:
            {  register xmcQueryDisplayReq *p = (xmcQueryDisplayReq *)rp;
               if (sp = server_find(p->id))
                  xmc_query(xp, sp);
               else
                  xmcp_Error(	xbp,
				xp->seqno,
				ErrDisplay,
				XMC_QueryDisplay,
				p->id);
            }
            break;
         case XMC_ListDisplays:
            {  register xmcListDisplaysReq *p = (xmcListDisplaysReq *)rp;
               xmc_list(xp);
            }
            break;
         case XMC_ListDisplaysWithInfo:
            {  register xmcListDisplaysWithInfoReq *p =
					(xmcListDisplaysWithInfoReq *)rp;
               xmc_list_with_info(xp);
            }
            break;
         case XMC_DropDisplay:
            {  register xmcDropDisplayReq *p = (xmcDropDisplayReq *)rp;
               if (sp = server_find(p->id))
                  xmc_drop(sp);
               else
                  xmcp_Error(	xbp,
				xp->seqno,
				ErrDisplay,
				XMC_DropDisplay,
				p->id);
            }
            break;
         case XMC_Sync:
            {  register xmcSyncReq *p = (xmcSyncReq *)rp;
               xmc_sync(xp);
            }
            break;
         case XMC_SetConfig:
            {  register xmcSetConfigReq *p = (xmcSetConfigReq *)rp;
               xmc_set_config(	p->length,
				p->width,
				p->nDepthTypes,
				0,
				p->nExtensions,
				0);
            }
            break;
         case XMC_GetConfig:
            {  register xmcGetConfigReq *p = (xmcGetConfigReq *)rp;
               xmc_get_config(xp);
            }
            break;
         case XMC_SetConfigMode:
            {  register xmcSetConfigModeReq *p = (xmcSetConfigModeReq *)rp;
               xmc_set_config_mode(p->mode);
            }
            break;
         case XMC_ChangeInputMode:
            {  register xmcChangeInputModeReq *p = (xmcChangeInputModeReq *)rp;
               if (sp = server_find(p->dispID))
                  xmc_change_mode(sp, p->mode);
               else
                  xmcp_Error(	xbp,
				xp->seqno,
				ErrDisplay,
				XMC_ChangeInputMode,
				p->dispID);
            }
            break;
         case XMC_SetEventMask:
            {  register xmcSetEventMaskReq *p = (xmcSetEventMaskReq *)rp;
               xmc_set_mask(xp, p->mask);
            }
            break;
         case XMC_GetEventMask:
            {  register xmcGetEventMaskReq *p = (xmcGetEventMaskReq *)rp;
               xmc_get_mask(xp);
            }
            break;
         case XMC_SetXEventMask:
            {  register xmcSetXEventMaskReq *p = (xmcSetXEventMaskReq *)rp;
               xmc_set_xmask(xp, p->window, p->mask);
            }
            break;
         case XMC_GetXEventMask:
            {  register xmcGetXEventMaskReq *p = (xmcGetXEventMaskReq *)rp;
               xmc_get_xmask(xp, p->id);
            }
            break;
         case XMC_GrabPointer:
            {  register xmcGrabPointerReq *p = (xmcGrabPointerReq *)rp;
               if (sp = server_find(p->id))
                  xmc_grab_pointer(xp, sp);
               else
                  xmcp_Error(	xbp,
				xp->seqno,
				ErrDisplay,
				XMC_GrabPointer,
				p->id);
            }
            break;
         case XMC_UngrabPointer:
            {  register xmcUngrabPointerReq *p = (xmcUngrabPointerReq *)rp;
               if (sp = server_find(p->id))
                  xmc_ungrab_pointer(xp, sp);
               else
                  xmcp_Error(	xbp,
				xp->seqno,
				ErrDisplay,
				XMC_UngrabPointer,
				p->id);
            }
            break;
         case XMC_GrabKeyboard:
            {  register xmcGrabKeyboardReq *p = (xmcGrabKeyboardReq *)rp;
               if (sp = server_find(p->id))
                  xmc_grab_keyboard(xp, sp);
               else
                  xmcp_Error(	xbp,
				xp->seqno,
				ErrDisplay,
				XMC_GrabKeyboard,
				p->id);
            }
            break;
         case XMC_UngrabKeyboard:
            {  register xmcUngrabKeyboardReq *p = (xmcUngrabKeyboardReq *)rp;
               if (sp = server_find(p->id))
                  xmc_ungrab_keyboard(xp, sp);
               else
                  xmcp_Error(	xbp,
				xp->seqno,
				ErrDisplay,
				XMC_UngrabKeyboard,
				p->id);
            }
            break;
         case XMC_ShareSelections:
            {  register xmcShareSelectionsReq *p = (xmcShareSelectionsReq *)rp;
               if (sp = server_find(p->id))
                  xmc_share_selections(xp, sp);
               else
                  xmcp_Error(	xbp,
				xp->seqno,
				ErrDisplay,
				XMC_ShareSelections,
				p->id);
            }
            break;
         case XMC_UnshareSelections:
            {  register xmcUnshareSelectionsReq *p =
						(xmcUnshareSelectionsReq *)rp;
               if (sp = server_find(p->id))
                  xmc_unshare_selections(xp, sp);
               else
                  xmcp_Error(	xbp,
				xp->seqno,
				ErrDisplay,
				XMC_UnshareSelections,
				p->id);
            }
            break;
         case XMC_CreateTptr:
            {  register xmcCreateTptrReq *p = (xmcCreateTptrReq *)rp;

               xmc_create_tptr(	p->tpID, p->mask,
				p->srcID, p->mskID,
				p->hotx, p->hoty,
				p->foreRed, p->foreGreen, p->foreBlue,
				p->backRed, p->backGreen, p->backBlue);
            }
            break;
         case XMC_CreateGlyphTptr:
            {  register xmcCreateGlyphTptrReq *p =
						(xmcCreateGlyphTptrReq *)rp;
               xmc_create_glyph_tptr(p->tpID, p->mask,
				p->srcID, p->mskID,
				p->srcChar, p->mskChar,
				p->foreRed, p->foreGreen, p->foreBlue,
				p->backRed, p->backGreen, p->backBlue);
            }
            break;
         case XMC_DestroyTptr:
            {  register xmcDestroyTptrReq *p = (xmcDestroyTptrReq *)rp;
               if (tpp = tptr_find(p->id))
                  tptr_destroy(tpp);
               else
                  xmcp_Error(	xbp,
				xp->seqno,
				ErrTelepointer,
				XMC_DestroyTptr,
				p->id);
            }
            break;
         case XMC_AssignTptr:
            {  register xmcAssignTptrReq *p = (xmcAssignTptrReq *)rp;
               if (sp = server_find(p->dispID))
                  if (p->tpID == 0)
                     tptr_set(sp, 0);
                  else if (tpp = tptr_find(p->tpID))
                     tptr_set(sp, tpp);
                  else
                     xmcp_Error(xbp,
				xp->seqno,
				ErrTelepointer,
				XMC_AssignTptr,
				p->tpID);
               else
                  xmcp_Error(	xbp,
				xp->seqno,
				ErrDisplay,
				XMC_AssignTptr,
				p->dispID);
            }
            break;
         case XMC_HideTptr:
            {  register xmcHideTptrReq *p = (xmcHideTptrReq *)rp;
               if (tpp = tptr_find(p->id))
                  tptr_hide(tpp);
               else
                  xmcp_Error(	xbp,
				xp->seqno,
				ErrTelepointer,
				XMC_HideTptr,
				p->id);
            }
            break;
         case XMC_ShowTptr:
            {  register xmcShowTptrReq *p = (xmcShowTptrReq *)rp;
               if (tpp = tptr_find(p->id))
                  tptr_show(tpp);
               else
                  xmcp_Error(	xbp,
				xp->seqno,
				ErrTelepointer,
				XMC_ShowTptr,
				p->id);
            }
            break;
      }
      if (buf_active(xbp)) {
         rchp = buf_split(xbp, 0);
         if (xp->swap)
            swapreply((xmcReply *)buf_data(rchp), rp->code);
         queue_add(xp->qp, rchp);
         buf_clear(rchp);
      }
   }
}

/************************************************************************
*									*
*   xmc_register							*
*   xmc_unregister							*
*									*
************************************************************************/
void
xmc_register
   AL((regID, family, length, addr, port, name, url, desc))
   DB u32_t regID
   DD u8_t family
   DD u16_t length
   DD char *addr
   DD u16_t port
   DD char *name
   DD char *url
   DD char *desc
   DE
{
   warn("xmc_register: not implemented\n");
}

void
xmc_unregister
   AL((regID))
   DB u32_t regID
   DE
{
   warn("xmc_unregister: not implemented\n");
}

/************************************************************************
*									*
*   xmc_set_auth							*
*   xmc_get_auth							*
*									*
************************************************************************/
void
xmc_set_auth
   AL((name, data, pmask))
   DB char *name
   DD char *data
   DD u32_t pmask
   DE
{
   warn("xmc_set_auth: not implemented\n");

   if (name)
      free(name);
   if (data)
      free(data);
}

void
xmc_get_auth
   AL((xp, name, data))
   DB xmc_t *xp
   DD char *name
   DD char *data
   DE
{
   warn("xmc_get_auth: not implemented\n");

   if (name)
      free(name);
   if (data)
      free(data);
}

/************************************************************************
*									*
*   xmc_add								*
*									*
************************************************************************/
int
xmc_add
   AL((sp))
   DB server_t *sp
   DE
{
   if (sp->state != S_NAMED)
      return 0;		/* server already added */

   if (cliz_add_server(sp))
      return -1;

   main_open_conn(sp->fd, SERVER, sp);

   return 0;
}

/************************************************************************
*									*
*   xmc_set_tag								*
*									*
************************************************************************/
void
xmc_set_tag
   AL((xp, sp, tag))
   DB xmc_t *xp
   DD server_t *sp
   DD char *tag
   DE
{
   warn("xmc_set_tag: not implemented\n");

   if (tag)
      free(tag);
}

/************************************************************************
*									*
*   xmc_query								*
*									*
************************************************************************/
void
xmc_query
   AL((xp, sp))
   DB xmc_t *xp
   DD server_t *sp
   DE
{
   xmcp_QueryDisplayReply(	xbp,
				xp->seqno,
				sp->screen,
				sp->display,
				sp->addr.family,
				sp->addr.length,
				sp->addr.address,
				sp->mode,
				sp->tag,
				0,	/* pointer */
				sp->smap.root.realshell);
}

/************************************************************************
*									*
*   xmc_list								*
*									*
************************************************************************/
void
xmc_list
   AL((xp))
   DB xmc_t *xp
   DE
{
   register int i;
   register list_t *lstp;

   lstp = list_new();
   for (i=0; i<num_serv; i++) {
      list_put(lstp, servers[i]->id);
   }
   xmcp_ListDisplaysReply(xbp, xp->seqno, num_serv, lstp);
   list_free(lstp);
}

/************************************************************************
*									*
*   xmc_list_with_info							*
*									*
************************************************************************/
void
xmc_list_with_info
   AL((xp))
   DB xmc_t *xp
   DE
{
   register int i;

   if (num_serv)
      for (i=0; i<num_serv; i++)
         xmcp_ListDisplaysWithInfoReply(xbp,
					xp->seqno,
					servers[i]->id,
					servers[i]->screen,
					servers[i]->display,
					servers[i]->addr.family,
					servers[i]->addr.length,
					servers[i]->addr.address,
					servers[i]->mode,
					servers[i]->tag,
					0,	/* pointer */
					servers[i]->smap.root.realshell,
					num_serv - i);
   else
      xmcp_ListDisplaysWithInfoReply(xbp, xp->seqno,
					0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
}

/************************************************************************
*									*
*   xmc_drop								*
*									*
************************************************************************/
void
xmc_drop
   AL((sp))
   DB server_t *sp
   DE
{
   register xmc_t *xp;

   /*
   **  ensure that nothing pending is sent to this server
   */
   zb_server_death(sp);

   switch (sp->state) {
      case S_NAMED:
         break;
      case S_READY:
         xmc_display_event(DisplayOutEvent, sp->id);
         tptr_set(sp, 0);
         main_close_conn(sp->fd);
         break;
      default:
         xmc_display_event(DisplayRefusedEvent, sp->id);
         main_close_conn(sp->fd);
         break;
   }
   reply_server_death(sp);
   server_free(sp);
}

/************************************************************************
*									*
*   xmc_sync								*
*									*
*	Ensure that all servers have processed all X requests, then	*
*	send the XMC client a reply.					*
*									*
************************************************************************/
void
xmc_sync
   AL((xp))
   DB xmc_t *xp
   DE
{
   zb_dest(ZD_ALLSERV, 0);
   proto_GetInputFocus();
   rx_begin_repeat(zrxqp);
   rx_queue(zrxqp, X_GetInputFocus, 0, zb_seqno(), 0);
   rx_end_repeat(zrxqp);
 
   df_put_p((void *)xp);
   df_put_i((int)xp->seqno);
   cx_push(zrxqp, sync_cont);
 
   fd_reset(-1);
}

static void
sync_cont
   AL((dfp))
   DB df_t *dfp
   DE
{
   register chunk_t *chp;
   register xmc_t *xp = (xmc_t *)df_get_p(dfp);
   register u16_t seqno = (u16_t)df_get_i(dfp);

   fd_unreset();

   xmcp_SyncReply(xbp, seqno);

   chp = buf_split(xbp, 0);
   if (xp->swap)
      swapreply((xmcReply *)buf_data(chp), XMC_Sync);
   queue_add(xp->qp, chp);
   buf_clear(chp);
}

/************************************************************************
*									*
*   xmc_set_config							*
*									*
************************************************************************/
void
xmc_set_config
   AL((length, width, ndepthtypes, depthtypes, nexts, exts))
   DB u16_t length
   DD u16_t width
   DD int ndepthtypes
   DD char *depthtypes
   DD int nexts
   DD char **exts
   DE
{
   warn("xmc_set_config: not implemented\n");
}

/************************************************************************
*									*
*   xmc_get_config							*
*									*
************************************************************************/
void
xmc_get_config
   AL((xp))
   DB xmc_t *xp
   DE
{
   warn("xmc_get_config: not implemented\n");
}

/************************************************************************
*									*
*   xmc_set_config_mode							*
*									*
************************************************************************/
void
xmc_set_config_mode
   AL((mode))
   DB int mode
   DE
{
   if (mode != config_mode) {
      config_mode = mode;
      if (config_mode == ConfigAllow)
         cliz_vconfig_cont(0);
   }
}

/************************************************************************
*									*
*   xmc_change_mode							*
*									*
************************************************************************/
void
xmc_change_mode
   AL((sp, mode))
   DB server_t *sp
   DD int mode
   DE
{
   register chunk_t *chp;

   if (sp->mode != mode) {
      /*
      ** if gaining or losing the floor, update telepointer
      */
      if (mode == Floor)
         tptr_ref(sp);
      else if (sp->mode == Floor)
         tptr_deref(sp);

      sp->mode = mode;
      /*
      ** reset all window event masks to effect the mode change
      */
      zb_dest(ZD_SERV, sp);
      window_touch_all_emasks();
      /*
      ** generate an XMC event
      */
      xmc_inputmode_event(sp->id, mode);
   }
}

/************************************************************************
*									*
*   xmc_set_mask							*
*									*
************************************************************************/
void
xmc_set_mask
   AL((xp, mask))
   DB xmc_t *xp
   DD u16_t mask
   DE
{
   xp->eventmask = mask;
}

/************************************************************************
*									*
*   xmc_get_mask							*
*									*
*	Collect all regular event definitions, and send them to the	*
*	xmc client.							*
*									*
************************************************************************/
void
xmc_get_mask
   AL((xp))
   DB xmc_t *xp
   DE
{
   xmcp_GetEventMaskReply(xbp, xp->seqno, xp->eventmask);
}

/************************************************************************
*									*
*   xmc_set_xmask							*
*									*
************************************************************************/
void
xmc_set_xmask
   AL((xp, window, xmask))
   DB xmc_t *xp
   DD rid_t window
   DD u16_t xmask
   DE
{
   register int i;
   register window_t *wp;

   if ((wp = (window_t *)hash_data(vmap, window)) == 0) {
      xmc_error(xp, ErrId, XMC_SetXEventMask, window);
      return;
   }
   /*
   **  The four X event masks match the corresponding key and button
   **  event masks in X protocol.  If they differed, we'd translate
   **  them here.
   */
   /* should really generate a bad value error here if mask is bogus */
   xmask &= KeyPressMask|KeyReleaseMask|ButtonPressMask|ButtonReleaseMask;

   imask_put(wp->xmcmask, (char *)xp, (mask_t)xmask);
   MkCompMask(wp);
   zb_dest(ZD_SEAT, 0);
   window_touch_emask(wp);
}

/************************************************************************
*									*
*   xmc_get_xmask							*
*									*
*	Send to a client its X event mask for a given window.		*
*									*
************************************************************************/
void
xmc_get_xmask
   AL((xp, window))
   DB xmc_t *xp
   DD rid_t window
   DE
{
   register int i;
   register window_t *wp;

   if ((wp = (window_t *)hash_data(vmap, window)) == 0) {
      xmc_error(xp, ErrId, XMC_GetXEventMask, window);
      return;
   }
   /*
   **  The four X event masks match the corresponding key and button
   **  event masks in X protocol.  If they differed, we'd translate
   **  them here.
   */
   xmcp_GetXEventMaskReply(	xbp,
				xp->seqno,
				(u16_t)imask_get(wp->xmcmask, (char *)xp));
}

/************************************************************************
*									*
*   xmc_grab_pointer							*
*   xmc_ungrab_pointer							*
*									*
************************************************************************/
void
xmc_grab_pointer
   AL((xp, sp))
   DB xmc_t *xp
   DD server_t *sp
   DE
{
   warn("xmc_grab_pointer: not implemented\n");
}

void
xmc_ungrab_pointer
   AL((xp, sp))
   DB xmc_t *xp
   DD server_t *sp
   DE
{
   warn("xmc_ungrab_pointer: not implemented\n");
}

/************************************************************************
*									*
*   xmc_grab_keyboard							*
*   xmc_ungrab_keyboard							*
*									*
************************************************************************/
void
xmc_grab_keyboard
   AL((xp, sp))
   DB xmc_t *xp
   DD server_t *sp
   DE
{
   warn("xmc_grab_keyboard: not implemented\n");
}

void
xmc_ungrab_keyboard
   AL((xp, sp))
   DB xmc_t *xp
   DD server_t *sp
   DE
{
   warn("xmc_grab_keyboard: not implemented\n");
}

void
xmc_grab_succeeded
   AL((xp, sp))
   DB xmc_t *xp
   DD server_t *sp
   DE
{
   warn("xmc_grab_succeeded: not implemented\n");
}

void
xmc_grab_failed
   AL((xp, sp))
   DB xmc_t *xp
   DD server_t *sp
   DE
{
   warn("xmc_grab_failed: not implemented\n");
}

/************************************************************************
*									*
*   xmc_share_selections						*
*   xmc_unshare_selections						*
*									*
************************************************************************/
void
xmc_share_selections
   AL((xp, sp))
   DB xmc_t *xp
   DD server_t *sp
   DE
{
   warn("xmc_share_selections: not implemented\n");
}

void
xmc_unshare_selections
   AL((xp, sp))
   DB xmc_t *xp
   DD server_t *sp
   DE
{
   warn("xmc_unshare_selections: not implemented\n");
}

/************************************************************************
*									*
*   xmc_create_tptr							*
*   xmc_create_glyph_tptr						*
*									*
************************************************************************/
void
xmc_create_tptr
   AL((tpid, mask, srcid, mskid, x, y, fr, fg, fb, br, bg, bb))
   DB u32_t tpid
   DD u32_t mask
   DD rid_t srcid
   DD rid_t mskid
   DD s16_t x
   DD s16_t y
   DD u16_t fr
   DD u16_t fg
   DD u16_t fb
   DD u16_t br
   DD u16_t bg
   DD u16_t bb
   DE
{
   warn("xmc_create_tptr: not implemented\n");
}

void
xmc_create_glyph_tptr
   AL((tpid, mask, srcid, mskid, sch, mch, fr, fg, fb, br, bg, bb))
   DB u32_t tpid
   DD u32_t mask
   DD rid_t srcid
   DD rid_t mskid
   DD s16_t sch
   DD s16_t mch
   DD u16_t fr
   DD u16_t fg
   DD u16_t fb
   DD u16_t br
   DD u16_t bg
   DD u16_t bb
   DE
{
   warn("xmc_create_glyph_tptr: not implemented\n");
}

/************************************************************************
*									*
*   xmc_xevent								*
*									*
*	Send x event to each xmc client that has selected for X		*
*	events.								*
*									*
*	Note: this routine needs to be rewritten to accommodate		*
*	window-granularity event handling.				*
*									*
************************************************************************/
void
xmc_xevent
   AL((sp, wp, type, detail, time, event, child, rootX, rootY, eventX, eventY,
									state))
   DB server_t *sp
   DD window_t *wp
   DD u8_t type
   DD u8_t detail
   DD timestamp_t time
   DD rid_t event
   DD rid_t child
   DD s16_t rootX
   DD s16_t rootY
   DD s16_t eventX
   DD s16_t eventY
   DD u16_t state
   DE
{
   register u8_t code;
   rid_t realchild = wp->cid;
   mask_t j, mask;
   register chunk_t *chp;
   xmc_t *xp;

   switch (type) {
      case KeyPress:
         code = SeatKeyPressEvent;
         mask = KeyPressMask;
         break;
      case KeyRelease:
         code = SeatKeyReleaseEvent;
         mask = KeyReleaseMask;
         break;
      case ButtonPress:
         code = SeatButtonPressEvent;
         mask = ButtonPressMask;
         break;
      case ButtonRelease:
         code = SeatButtonReleaseEvent;
         mask = ButtonReleaseMask;
         break;
      default:
         return;
   }
   for (;;)
      if (wp->xmcmask->mask & mask)
         break;
      else if (wp->parent) {
         eventX = XOFFSET(wp);
         eventY = YOFFSET(wp);
         wp = wp->parent;
      }
      else
         return;		/* no window wants this event */

   for (	j=imask_first(wp->xmcmask, mask, (char **)&xp);
		j;
		j=imask_next((char **)&xp)) {
      xmcp_KeyButtonEvent(xbp, code, xp->seqno, sp->id, detail, time, wp->cid,
			realchild, rootX, rootY, eventX, eventY, state);
      chp = buf_split(xbp, 0);
      queue_add(xp->qp, chp);
      buf_clear(chp);
   }
}

void
xmc_display_event
   AL((code, id))
   DB u8_t code
   DD rid_t id
   DE
{
   register chunk_t *chp;
   register xmc_t *xp;

   for (xp=conns; xp; xp=xp->next)
      if (xp->eventmask & DisplayMask) {
         xmcp_DisplayIdEvent(xbp, code, xp->seqno, id);
         chp = buf_split(xbp, 0);
         queue_add(xp->qp, chp);
         buf_clear(chp);
      }
}

void
xmc_inputmode_event
   AL((id, mode))
   DB rid_t id
   DD u8_t mode
   DE
{
   switch (mode) {
      case View:
         xmc_display_event(ModeViewEvent, id);
         break;
      case Seat:
         xmc_display_event(ModeSeatEvent, id);
         break;
      case Floor:
         xmc_display_event(ModeFloorEvent, id);
         break;
   }
}

void
xmc_error
   AL((xp, errcode, opcode, data))
   DB xmc_t *xp
   DD u8_t errcode
   DD u8_t opcode
   DD u32_t data
   DE
{
   register chunk_t *chp;

   if (xp) {
      xmcp_Error(xbp, xp->seqno, errcode, opcode, data);
      chp = buf_split(xbp, 0);
      queue_add(xp->qp, chp);
      buf_clear(chp);
   }
   else
      warn("xmc_error: internal error\n");	/* could be better */
}

/*
**  get_strings
**
**	Make copies of strings located contiguously in memory.  Lenp
**	is an array of lengths.  An array of newly allocated and null-
**	terminated strings is placed in strp.
*/
static int
get_strings
   AL((cp, n, lenp, strp))
   DB char *cp
   DD int n
   DD int *lenp
   DD char **strp
   DE
{
   register int i, j;

   for (i=0; i<n; i++) {
      if (lenp[i]) {
         if (MALLOC(strp[i], char *, lenp[i] + 1)) {
            for (j=0; j<i; j++)
               if (strp[j])
                  free(strp[j]);
            return -1;
         }
         bcopy(cp, strp[i], lenp[i]);
         strp[i][lenp[i]] = '\0';
         cp += lenp[i];
      }
      else
         strp[i] = 0;
   }
   return 0;
}

/*
**  free_strings
**
**	Free the strings allocated by the previous procedure.
*/
static int
free_strings
   AL((n, strp))
   DB int n
   DD char **strp
   DE
{
   register int i;

   for (i=0; i<n; i++)
      if (strp[i])
         free(strp[i]);
}

static void
swapreq
   AL((rp))
   DB xmcReq *rp
   DE
{
   SWAP2(rp->length);

   switch (rp->code) {
      case XMC_Register:
         {  register xmcRegisterReq *p = (xmcRegisterReq *)rp;
            SWAP4(p->regID);
            SWAP2(p->port);
            SWAP2(p->nBytesAddress);
            SWAP2(p->nBytesName);
            SWAP2(p->nBytesURL);
            SWAP2(p->nBytesDesc);
         }
         break;
      case XMC_Unregister:
         {  register xmcUnregisterReq *p = (xmcUnregisterReq *)rp;
            SWAP4(p->id);
         }
         break;
      case XMC_SetAuth:
         {  register xmcSetAuthReq *p = (xmcSetAuthReq *)rp;
            SWAP4(p->pMask);
            SWAP2(p->nBytesName);
            SWAP2(p->nBytesData);
         }
         break;
      case XMC_GetAuth:
         {  register xmcGetAuthReq *p = (xmcGetAuthReq *)rp;
            SWAP2(p->nBytesName);
            SWAP2(p->nBytesData);
         }
         break;
      case XMC_AddDisplay:
         {  register xmcAddDisplayReq *p = (xmcAddDisplayReq *)rp;
            SWAP4(p->dispID);
            SWAP2(p->screen);
            SWAP2(p->nBytesAddress);
            SWAP2(p->displayNumber);
            SWAP4(p->window);
            SWAP2(p->nBytesGeom);
            SWAP2(p->nBytesTag);
            SWAP2(p->nBytesName);
            SWAP2(p->nBytesData);
            SWAP4(p->tpID);
         }
         break;
      case XMC_SetDisplayTag:
         {  register xmcSetDisplayTagReq *p = (xmcSetDisplayTagReq *)rp;
            SWAP4(p->dispID);
            SWAP2(p->nBytesTag);
         }
         break;
      case XMC_QueryDisplay:
         {  register xmcQueryDisplayReq *p = (xmcQueryDisplayReq *)rp;
            SWAP4(p->id);
         }
         break;
      case XMC_ListDisplays:
         break;
      case XMC_ListDisplaysWithInfo:
         break;
      case XMC_DropDisplay:
         {  register xmcDropDisplayReq *p = (xmcDropDisplayReq *)rp;
            SWAP4(p->id);
         }
         break;
      case XMC_Sync:
         break;
      case XMC_SetConfig:
         {  register xmcSetConfigReq *p = (xmcSetConfigReq *)rp;
            SWAP2(p->width);
            SWAP2(p->height);
            SWAP2(p->nDepthTypes);
            SWAP2(p->nExtensions);
         }
         break;
      case XMC_GetConfig:
         break;
      case XMC_SetConfigMode:
         break;
      case XMC_ChangeInputMode:
         {  register xmcChangeInputModeReq *p = (xmcChangeInputModeReq *)rp;
            SWAP4(p->dispID);
         }
         break;
      case XMC_SetEventMask:
         {  register xmcSetEventMaskReq *p = (xmcSetEventMaskReq *)rp;
            SWAP4(p->mask);
         }
         break;
      case XMC_GetEventMask:
         break;
      case XMC_SetXEventMask:
         {  register xmcSetXEventMaskReq *p = (xmcSetXEventMaskReq *)rp;
            SWAP4(p->window);
            SWAP4(p->mask);
         }
         break;
      case XMC_GetXEventMask:
         {  register xmcGetXEventMaskReq *p = (xmcGetXEventMaskReq *)rp;
            SWAP4(p->id);
         }
         break;
      case XMC_GrabPointer:
         {  register xmcGrabPointerReq *p = (xmcGrabPointerReq *)rp;
            SWAP4(p->id);
         }
         break;
      case XMC_UngrabPointer:
         {  register xmcUngrabPointerReq *p = (xmcUngrabPointerReq *)rp;
            SWAP4(p->id);
         }
         break;
      case XMC_GrabKeyboard:
         {  register xmcGrabKeyboardReq *p = (xmcGrabKeyboardReq *)rp;
            SWAP4(p->id);
         }
         break;
      case XMC_UngrabKeyboard:
         {  register xmcUngrabKeyboardReq *p = (xmcUngrabKeyboardReq *)rp;
            SWAP4(p->id);
         }
         break;
      case XMC_ShareSelections:
         {  register xmcShareSelectionsReq *p = (xmcShareSelectionsReq *)rp;
            SWAP4(p->id);
         }
         break;
      case XMC_UnshareSelections:
         {  register xmcUnshareSelectionsReq *p = (xmcUnshareSelectionsReq *)rp;
            SWAP4(p->id);
         }
         break;
      case XMC_CreateTptr:
         {  register xmcCreateTptrReq *p = (xmcCreateTptrReq *)rp;
            SWAP4(p->tpID);
            SWAP4(p->srcID);
            SWAP4(p->mskID);
            SWAP2(p->hotx);
            SWAP2(p->hoty);
            SWAP2(p->foreRed);
            SWAP2(p->foreGreen);
            SWAP2(p->foreBlue);
            SWAP2(p->backRed);
            SWAP2(p->backGreen);
            SWAP2(p->backBlue);
         }
         break;
      case XMC_CreateGlyphTptr:
         {  register xmcCreateGlyphTptrReq *p=(xmcCreateGlyphTptrReq *)rp;
            SWAP4(p->tpID);
            SWAP4(p->srcID);
            SWAP4(p->mskID);
            SWAP2(p->srcChar);
            SWAP2(p->mskChar);
            SWAP2(p->foreRed);
            SWAP2(p->foreGreen);
            SWAP2(p->foreBlue);
            SWAP2(p->backRed);
            SWAP2(p->backGreen);
            SWAP2(p->backBlue);
         }
         break;
      case XMC_DestroyTptr:
         {  register xmcDestroyTptrReq *p = (xmcDestroyTptrReq *)rp;
            SWAP4(p->id);
         }
         break;
      case XMC_AssignTptr:
         {  register xmcAssignTptrReq *p = (xmcAssignTptrReq *)rp;
            SWAP4(p->dispID);
            SWAP4(p->tpID);
         }
         break;
      case XMC_HideTptr:
         {  register xmcHideTptrReq *p = (xmcHideTptrReq *)rp;
            SWAP4(p->id);
         }
         break;
      case XMC_ShowTptr:
         {  register xmcShowTptrReq *p = (xmcShowTptrReq *)rp;
            SWAP4(p->id);
         }
         break;
      default:
         warn("xmc.c/swapreq: unknown XMC request type %d\n", rp->code);
         break;
   }
}

static void
swapreply
   AL((rp, code))
   DB xmcReply *rp
   DD u8_t code
   DE
{
   register int i;
   register u16_t *sp;
   register u32_t *lp;

   SWAP2(rp->seqNo);
   SWAP4(rp->length);
   switch (code) {
      case XMC_GetAuth:
         {  register xmcGetAuthReply *r = (xmcGetAuthReply *)rp;
            SWAP4(r->pMask);
         }
         break;
      case XMC_QueryDisplay:
         {  register xmcQueryDisplayReply *r = (xmcQueryDisplayReply *)rp;
            SWAP2(r->nBytesAddress);
            SWAP2(r->displayNumber);
            SWAP2(r->nBytesTag);
            SWAP4(r->tpID);
            SWAP4(r->window);
            SWAP2(r->screen);
         }
         break;
      case XMC_ListDisplays:
         {  register xmcListDisplaysReply *r = (xmcListDisplaysReply *)rp;
            lp = (u32_t *)(r + 1);
            for (i=0; i<r->nDisplays; i++)
               SWAP4(lp[i]);
            SWAP2(r->nDisplays);
         }
         break;
      case XMC_ListDisplaysWithInfo:
         {  register xmcListDisplaysWithInfoReply *r =
					(xmcListDisplaysWithInfoReply *)rp;
            SWAP4(r->dispID);
            SWAP2(r->nBytesAddress);
            SWAP2(r->displayNumber);
            SWAP2(r->nBytesTag);
            SWAP4(r->tpID);
            SWAP4(r->window);
            SWAP2(r->screen);
            SWAP2(r->count);
         }
         break;
      case XMC_Sync:
         break;
      case XMC_GetConfig:
         {  register xmcGetConfigReply *r = (xmcGetConfigReply *)rp;
            SWAP2(r->width);
            SWAP2(r->height);
            SWAP2(r->nDepthTypes);
            SWAP2(r->nExtensions);
         }
         break;
      case XMC_GetEventMask:
         {  register xmcGetEventMaskReply *r = (xmcGetEventMaskReply *)rp;
            SWAP4(r->mask);
         }
         break;
      case XMC_GetXEventMask:
         {  register xmcGetEventMaskReply *r = (xmcGetEventMaskReply *)rp;
            SWAP4(r->mask);
         }
         break;
      default:
         warn("xmc.c/swapreply: unknown XMC request type %d\n", code);
         break;
   }
}
