/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   util.c								*
*									*
*	Various and sundry.						*
*									*
************************************************************************/
#ifdef _AIX
#include <sys/types.h>
#endif
#include <sys/param.h>
#include <sys/socket.h>
#ifdef SVR4
#include <sys/utsname.h>
#endif
#include <signal.h>

#include <X11/X.h>
#include <X11/Xproto.h>

#include "xmx.h"
#include "incl/util.pvt.h"

static int dosig = 0;

/************************************************************************
*									*
*   util_init								*
*									*
************************************************************************/
#define Undef	0

void
util_init
   VOID
{
   win_defs[IWBackPixmap] =       None;
   win_defs[IWBackPixel] =        Undef;
   win_defs[IWBorderPixmap] =     CopyFromParent;
   win_defs[IWBorderPixel] =      Undef;
   win_defs[IWBitGravity] =       ForgetGravity;
   win_defs[IWWinGravity] =       NorthWestGravity;
   win_defs[IWBackingStore] =     NotUseful;
   win_defs[IWBackingPlanes] =    0xffffffff;
   win_defs[IWBackingPixel] =     0;
   win_defs[IWOverrideRedirect] = xFalse;
   win_defs[IWSaveUnder] =        0;
   win_defs[IWEventMask] =        0;
   win_defs[IWDontPropagate] =    xFalse;
   win_defs[IWColormap] =         CopyFromParent;
   win_defs[IWCursor] =           None;

   gc_defs[IGFunction] =          GXcopy;
   gc_defs[IGPlaneMask] =         0xffffffff;
   gc_defs[IGForeground] =        0;
   gc_defs[IGBackground] =        1;
   gc_defs[IGLineWidth] =         0;
   gc_defs[IGLineStyle] =         LineSolid;
   gc_defs[IGCapStyle] =          CapButt;
   gc_defs[IGJoinStyle] =         JoinMiter;
   gc_defs[IGFillStyle] =         FillSolid;
   gc_defs[IGFillRule] =          EvenOddRule;
   gc_defs[IGTile] =              0;      /* need to define this */
   gc_defs[IGStipple] =           0;      /* and this */
   gc_defs[IGTileStipXOrigin] =   0;
   gc_defs[IGTileStipYOrigin] =   0;
   gc_defs[IGFont] =              0;      /* and this */
   gc_defs[IGSubwindowMode] =     ClipByChildren;
   gc_defs[IGGraphicsExposures] = xTrue;
   gc_defs[IGClipXOrigin] =       0;
   gc_defs[IGClipYOrigin] =       0;
   gc_defs[IGClipMask] =          None;
   gc_defs[IGDashOffset] =        0;
   gc_defs[IGDashList] =          4;
   gc_defs[IGArcMode] =           ArcPieSlice;
}

/************************************************************************
*									*
*   util_setsig								*
*									*
*	Setup any signal processing.					*
*									*
************************************************************************/
void
util_setsig
   VOID
{
#ifdef NEED_PROTO
   void (*fptr)(int);
#else
   void (*fptr)();
#endif
#ifdef SA_RESTART
   struct sigaction act;
#endif

#ifdef SA_RESTART
   act.sa_handler = sig_auth_reset;
   sigemptyset(&act.sa_mask);
   act.sa_flags = SA_RESTART;
   (void) sigaction(SIGHUP, &act, 0);
#else
   (void) signal(SIGHUP, sig_auth_reset);
#endif

   (void) signal(SIGINT, sig_cleanup);
   (void) signal(SIGQUIT, sig_cleanup);
   (void) signal(SIGTERM, sig_cleanup);

   (void) signal(SIGPIPE, SIG_IGN);
   fptr = signal(SIGUSR1, SIG_IGN);
   if (fptr == SIG_IGN)
      dosig = 1;
   else if (
#ifdef SIG_HOLD
		fptr != SIG_HOLD &&
#endif
		 fptr != SIG_ERR)
      (void) signal(SIGUSR1, fptr);	/* restore signal handler */
}

static void
sig_auth_reset
   AL((sig))
   DB int sig
   DE
{
   auth_reset();
}

static void
sig_cleanup
   AL((sig))
   DB int sig
   DE
{
   socket_cleanup();
   exit(0);
}

/************************************************************************
*									*
*   util_ready								*
*									*
*	Print the ready message.					*
*									*
************************************************************************/
void
util_ready
   VOID
{
   int ppid;

   if (opt.quiet == 0)
      printf("XMX Version %d.%d%s (patchlevel %d), ready.\n",
			xmx_version_major,
			xmx_version_minor,
			xmx_release_status,
			xmx_patchlevel);

   if (dosig)
      if ((ppid = getppid()) > 0) {
         /*
         **  This is a stupid kludge.  Xmx sends SIGUSR1 before xinit is
         **  ready to catch it.  So, xinit misses it and waits 15 secs
         **  before proceeding - ugh!  It's a bug in xinit, but we want
         **  to work with what's out there, so a short delay here helps
         **  considerably.
         */
         sleep(1);
         kill(ppid, SIGUSR1);
      }
}

/************************************************************************
*									*
*   util_occludes							*
*									*
*	Returns 1 if top occludes bot, -1 if bot occludes top or	*
*	0 if neither occludes the other.				*
*									*
*	If bot is zero, returns 1 if top occludes any sibling, -1 if	*
*	any sibling occludes top or 0 if neither case is true.		*
*									*
*	If top and bot are both supplied, they must be siblings.	*
*	If they are not, the return value is undefined.			*
*									*
************************************************************************/
int
util_occludes
   AL((top, bot))
   DB window_t *top	/* must be non-null */
   DD window_t *bot	/* null means any */
   DE
{
   window_t *tp;

   if (top->mapped)
      if (bot) {		/* does top occlude bot ? */
         if (bot->mapped && OVERLAP(top, bot)) {
            for (tp=top; tp && tp!=bot; tp=tp->nextsib);
            return tp ? 1 : -1;
         }
      }
      else {			/* is top occluded by any sibling ? */
         for (tp=top->prevsib; tp; tp=tp->prevsib)
            if (tp->mapped && OVERLAP(top,tp))
               return -1;
				/* or does top occlude any sibling ? */
         for (tp=top->nextsib; tp; tp=tp->nextsib)
            if (tp->mapped && OVERLAP(top,tp))
               return 1;
      }
   return 0;
}

/************************************************************************
*									*
*   util_fam_utox							*
*									*
************************************************************************/
u16_t
util_fam_utox
   AL((family))
   DB u16_t family
   DE
{
   switch (family) {
#ifdef AF_UNIX
      case AF_UNIX:	return FamilyLocal;
#endif
      case AF_INET:	return FamilyInternet;
#ifdef AF_CHAOS
      case AF_CHAOS:	return FamilyChaos;
#endif
#ifdef AF_DECnet
      case AF_DECnet:	return FamilyDECnet;
#endif
      default:
         warn("util_fam_utox: unknown address family [%d]\n", family);
         return FamilyWild;	/* not a great error value */
   }
}

/************************************************************************
*									*
*   util_fam_xtou							*
*									*
************************************************************************/
u16_t
util_fam_xtou
   AL((family))
   DB u16_t family
   DE
{
   switch (family) {
#ifdef AF_UNIX
      case FamilyLocal:		return AF_UNIX;
#endif
      case FamilyInternet:	return AF_INET;
#ifdef AF_CHAOS
      case FamilyChaos:		return AF_CHAOS;
#endif
#ifdef AF_DECnet
      case FamilyDECnet:	return AF_DECnet;
#endif
      default:
         warn("util_fam_xtou: unknown address family [%d]\n", family);
         return 0;	/* not a good error value */
   }
}

/************************************************************************
*									*
*   util_match								*
*									*
*	Match a string against a hostname glob pattern.  Returns	*
*	non-zero if it matches, zero if it doesn't.			*
*									*
*	Handles csh-style wildcards *, ?, [] and \.			*
*									*
*	Note: heinously coded.						*
*									*
************************************************************************/
int
util_match
   AL((pp, sp))
   DB char *pp
   DD char *sp
   DE
{
   register int i;
   char *pat[32];	/* up to 31 asterisks may appear in pattern */
   char *str[32];

   pat[i=0] = (char *)0;
   while (*sp && *pp) {
      switch (*pp) {
         case '*':
            for (pp++; *pp == '*'; pp++);
            if (*pp == '\0')
               return 1;
            if (++i == 32)
               return err(0, "util_match: too many wildcards - no match\n");
            pat[i] = pp;
            str[i] = sp;
            break;
         case '?':
            sp++;
            pp++;
            break;
         case '[':
            for (pp++; *pp && *pp != ']'; pp++) {
               if (	*pp == '-' &&		/* match against a range */
			*(pp-1) != '[' &&
			*(pp+1) != ']' &&
                  	*sp >= *(pp-1) &&
			*sp <= *(pp+1))
                  pp++;
               else if (*pp != *sp)
                  continue;
               sp++;
               break;
            }
            if (*pp && *pp != ']')		/* matched */
               while (*pp && *pp++ != ']');
            else {
               if (*sp == 0 && i)
                  sp = str[i--];		/* backtrack */
               pp = pat[i];			/* skip */
            }
            break;
         case '\\':
            pp++;	/* fall through */
         default:
            if (*pp++ != *sp++) {
               if (*sp == 0 && i)
                  sp = str[i--];		/* backtrack */
               pp = pat[i];			/* skip */
            }
      }
      if (pp == 0)
         return 0;				/* nothing left to try */
   }
   while (*pp == '*')		/* eat trailing asterisks */
      pp++;
   return (*sp == 0 && *pp == 0);
}

/************************************************************************
*									*
*   util_parse_display							*
*									*
*	Pull apart a display string, returning the hostname portion	*
*	and numeric address family, display and screen.			*
*									*
*	This routine relaxes the string format requirements.  Any	*
*	or all of the string may be omitted; missing values are set	*
*	to the defaults, below.  Only a malformed string will return	*
*	a -1 error.  Zero is returned if successful.			*
*									*
************************************************************************/
int
util_parse_display
   AL((string, host, family, display, screen, window))
   DB char *string
   DD char **host
   DD u16_t *family
   DD u16_t *display
   DD u16_t *screen
   DD rid_t *window
   DE
{
   register int i, len;
   register char *cp, *dp;
   char *ep;
   static char buf[MAXHOSTNAMELEN+1];

   buf[0] = '\0';			/* defaults */
   *host = buf;
   *family = FamilyLocal;
   *display = 0;
   *screen = 0;
   *window = 0;

   if (string) {
      if (cp = (char *)index(string, ':')) {
         len = (int)(cp - string);

         if (*++cp == ':') {		/* :: is decnet */
            *family = FamilyDECnet;
            cp++;
         }
         else
            *family = FamilyInternet;

         dp = cp;
         for (i=0; isdigit(*cp); i++, cp++)
            buf[i] = *cp;
         if (i) {
            buf[i] = '\0';
            *display = atoi(dp);	/* display number */
         }
         if (*cp) {
            if (*cp++ != '.')
               return err(-1, "malformed display string '%s'\n", string);
            dp = cp;
            for (i=0; isdigit(*cp); i++, cp++)
               buf[i] = *cp;
            if (i) {
               buf[i] = '\0';
               *screen = atoi(dp);	/* screen number */
            }
            if (*cp) {
               if (*cp++ != '.')
                  return err(-1, "malformed display string '%s'\n", string);
               dp = cp;
               for (i=0; isalnum(*cp); i++, cp++)
                  buf[i] = *cp;
               if (i) {
                  buf[i] = '\0';	/* window id */
                  if ((*window = strtol(dp, &ep, 0)) == 0)
                     return err(-1, "malformed display string '%s'\n", string);
                  else if (*ep)	/* garbage at end of string */
                     return err(-1, "malformed display string '%s'\n", string);
               }
               if (*cp)
                  return err(-1, "malformed display string '%s'\n", string);
            }
         }
      }
      else
         len = strlen(string);

      if (len == 0 || (len == 4 && bcmp(string, "unix", len) == 0)) {
         buf[0] = '\0';
         *family = FamilyLocal;
      }
      else {
         if (len > MAXHOSTNAMELEN)
            return err(-1, "%s: hostname too long!\n", string);
         bcopy(string, buf, len);
         buf[len] = '\0';
      }
   }
   return 0;
}

/************************************************************************
*									*
*   util_parse_geom							*
*									*
*	Parse a geometry string.  Handles the standard X geometry	*
*	string, with some extensions:					*
*		- dimensions may be expressed relatively, as a		*
*		  fraction of screensize.				*
*		- if the height dimension is omitted, it is copied	*
*		  from the supplied width.				*
*		- if the y position is omitted, it is copied from	*
*		  the x position.					*
*		- the x or y position integer component may be		*
*		  by 'c' or 'C' to indicate that the window should	*
*		  be centered on the screen.				*
*									*
*	Some examples:							*
*		100x200+50+75	- standard style string			*
*		1/2+c		- quarter screensize and centered	*
*		1/1x100+0	- a 100-pixel high strip across the top	*
*									*
*	Default is:							*
*		23/40		- 1/3 screensize, no position		*
*									*
************************************************************************/
geom_t *
util_parse_geom
   AL((str))
   DB char *str
   DE
{
   u16_t what;
   u16_t w, h, x, y;
   double wr, hr;
   char *cp, *sp;
   register geom_t *geomp;

   if (str == 0)
      return 0;

   what = 0;

   if (*(cp=str) == '=')
      cp++;

   if (isdigit(*cp)) {
      sp = cp;
      if (parse_dimen(&cp, G_ABSWIDTH, &what, &w, &wr))
         return 0;

      if (*cp == 'x') {
         cp++;
         if (isdigit(*cp))
            if (parse_dimen(&cp, G_ABSHEIGHT, &what, &h, &hr))
               return 0;
      }
      else
         if (parse_dimen(&sp, G_ABSHEIGHT, &what, &h, &hr))
            return 0;
   }
   else		/* default dimensions (1/3 screen area) */
      wr = hr = 23.0 / 40.0;

   if (*cp == '+' || *cp == '-') {
      what |= G_SETPOS;		/* user supplied position */

      sp = cp;
      if (parse_pos(&cp, G_CHORIZ, G_FROMRIGHT, &what, &x))
         return 0;

      if (*cp == '+' || *cp == '-') {
         if (parse_pos(&cp, G_CVERT, G_FROMBOTTOM, &what, &y))
            return 0;
      }
      else
         if (parse_pos(&sp, G_CVERT, G_FROMBOTTOM, &what, &y))
            return 0;
   }
   else
      x = y = 0;

   if (*cp)
      return 0;

   if (MALLOC(geomp, geom_t *, sizeof(geom_t)))
      return 0;

   geomp->what = what;
   if (what & G_ABSWIDTH)
      geomp->w = w;
   else
      geomp->wr = wr;
   if (what & G_ABSHEIGHT)
      geomp->h = h;
   else
      geomp->hr = hr;
   geomp->x = x;
   geomp->y = y;

   return geomp;
}

static int
parse_num
   AL((cpp))
   DB char **cpp	/* modified upon return */
   DE
{
   register char *cp, *bp;
   char buf[32];

   cp = *cpp;
   for (bp=buf; isdigit(*cp);)
      *bp++ = *cp++;
   *bp = '\0';
   *cpp = cp;

   return atoi(buf);
}

static int
parse_dimen
   AL((cpp, absmask, mp, wp, wrp))
   DB char **cpp
   DD u16_t absmask
   DD u16_t *mp
   DD u16_t *wp
   DD double *wrp
   DE
{
   int num, den;

   num = parse_num(cpp);
   if (**cpp == '/')
      if (isdigit(*++(*cpp))) {
         den = parse_num(cpp);
         *wrp = (double)num / (double)den;
      }
      else
         return -1;
   else {
      *wp = (u16_t)num;
      *mp |= absmask;
   }
   return 0;
}

static int
parse_pos
   AL((cpp, ctrmask, negmask, mp, posp))
   DB char **cpp
   DD u16_t ctrmask
   DD u16_t negmask
   DD u16_t *mp
   DD u16_t *posp
   DE
{
   char *cp;

   cp = *cpp;
   switch (*cp++) {
      case '+':
         break;
      case '-':
         *mp |= negmask;
         break;
      default:
         return -1;
   }
   switch (*cp) {
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
         *posp = (u16_t)parse_num(&cp);
         break;
      case 'c':
      case 'C':
         *mp |= ctrmask;
         cp++;
         break;
   }
   *cpp = cp;

   return 0;
}

/************************************************************************
*									*
*   util_adjsize							*
*									*
************************************************************************/
void
util_adjsize
   AL((gp, w, h, wmm, hmm, wp, hp, wmmp, hmmp))
   DB geom_t *gp
   DD u16_t w		/* screen pixWidth */
   DD u16_t h		/* screen pixHeight */
   DD u16_t wmm		/* screen mmWidth */
   DD u16_t hmm		/* screen mmHeight */
   DD u16_t *wp		/* return values */
   DD u16_t *hp
   DD u16_t *wmmp	/* optional */
   DD u16_t *hmmp	/* optional */
   DE
{
   register u16_t nw, nh;

   if (gp->what & G_ABSWIDTH)
      nw = gp->w;
   else
      nw = (u16_t)(gp->wr * (double)w);

   if (gp->what & G_ABSHEIGHT)
      nh = gp->h;
   else
      nh = (u16_t)(gp->hr * (double)h);

   *wp = nw;
   *hp = nh;
   if (wmmp)
      *wmmp = (u16_t)(((u32_t)wmm * (u32_t)nw) / (u32_t)w);
   if (hmmp)
      *hmmp = (u16_t)(((u32_t)hmm * (u32_t)nh) / (u32_t)h);
}

/************************************************************************
*									*
*   util_adjloc								*
*									*
************************************************************************/
void
util_adjloc
   AL((gp, ow, oh, nw, nh, xp, yp))
   DB geom_t *gp
   DD u16_t ow
   DD u16_t oh
   DD u16_t nw
   DD u16_t nh
   DD s16_t *xp
   DD s16_t *yp
   DE
{
   register s16_t nx, ny;

   if (gp->what & G_SETPOS) {

      if (gp->what & G_CHORIZ)
         nx = ((s16_t)ow - (s16_t)nw) / 2;
      else if (gp->what & G_FROMRIGHT)
         nx = (s16_t)ow - (s16_t)nw - gp->x;
      else
         nx = gp->x;

      if (gp->what & G_CVERT)
         ny = ((s16_t)oh - (s16_t)nh) / 2;
      else if (gp->what & G_FROMBOTTOM)
         ny = (s16_t)oh - (s16_t)nh - gp->y;
      else
         ny = gp->y;
   }
   else {
      nx = 0;
      ny = 0;
   }
   *xp = nx;
   *yp = ny;
}

int
util_bnull
   AL((cp,n))
   DB char *cp
   DD int n
   DE
{
   register int i;

   n /= 4;
   for (i=0; i<n; i++)
      if (((int *)cp)[i])
         break;

   return (i == n);
}

static rid_t client_id = 3;
static rid_t server_id = 3;

rid_t
util_new_client_id
   VOID
{
   return client_id++;
}

rid_t
util_new_server_id
   VOID
{
   return (server_id++ | RID_SERVER_MASK);
}

void
util_id_reset
   VOID
{
   client_id = 3;
   server_id = 3;
}
