/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   tptr.c								*
*									*
*									*
************************************************************************/
#include <sys/time.h>
#include <sys/types.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/X.h>
#include <X11/Xproto.h>
#include <xmc.h>
#include "xmx.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "zb.h"
#include "res.h"
#include "fd.h"
#include "incl/tptr.pvt.h"

static tp_t *deftpp;		/* default telepointer */
static tp_t *tptrs;

static tpim_t *rtpimp;		/* root window telepointer image */

/************************************************************************
*									*
*   tptr_default							*
*									*
************************************************************************/
tp_t *
tptr_default
   VOID
{
   return deftpp;
}

/************************************************************************
*									*
*   tptr_default_image							*
*									*
************************************************************************/
tpim_t *
tptr_default_image
   VOID
{
   return rtpimp;
}

/************************************************************************
*									*
*   tptr_init								*
*									*
*	Allocate default telepointer and initialize root window		*
*	telepointer image.						*
*									*
************************************************************************/
void
tptr_init
   VOID
{
   if (rtpimp == 0)
      if (rtpimp = new_tpim(1))
         root_tptr_dimen(	&rtpimp->hotx, &rtpimp->hoty,
				&rtpimp->width, &rtpimp->height);
      else
         quit(-1, "tptr_init: can't allocate root tptr image - fatal\n");

   if (deftpp == 0) {
      if ((deftpp = new_tp(0, opt.notp ? 0 : 1)) == 0)
         quit(-1, "tptr_init: can't allocate default telepointer\n");

      deftpp->tpimp = rtpimp;
      deftpp->next = deftpp->prev = 0;	/* not in tptr list */
   }
}

/************************************************************************
*									*
*   tptr_create								*
*									*
*	Create a new telepointer.					*
*									*
************************************************************************/
tp_t *
tptr_create
   AL((mask, spxp, mpxp, hotx, hoty, fr, fg, fb, br, bg, bb))
   DB u8_t mask
   DD pixmap_t *spxp
   DD pixmap_t *mpxp
   DD s16_t hotx
   DD s16_t hoty
   DD u16_t fr
   DD u16_t fg
   DD u16_t fb
   DD u16_t br
   DD u16_t bg
   DD u16_t bb
   DE
{
   tp_t *tpp;
   chunk_t *chp;

   if ((tpp = new_tp(mask, 1)) == 0) {
      free(tpp);
      return 0;
   }
   warn("tptr_create: assigning a bogus id - not implemented\n");
   tpp->id = 0;

   /* make sure source and image pixmaps are same size! */

   if (mask & TP_IMAGE) {
      if ((tpp->tpimp=tptr_create_image(spxp, mpxp, hotx, hoty, &chp)) == 0 ){
         free_tp(tpp);
         return 0;
      }
      zb_dest(ZD_ALLSERV, 0);
      zb_queue_chunk(chp);
      tptr_draw_tp(tpp);
   }
   else {
      tpp->tpimp = rtpimp;
      zb_dest(ZD_ALLSERV, 0);
      tptr_draw_tp(tpp);
   }
   link_tp(tpp);
   return tpp;
}

/************************************************************************
*									*
*   tptr_create_glyph							*
*									*
*	Create a new glyph telepointer.					*
*									*
************************************************************************/
tp_t *
tptr_create_glyph
   AL((mask, sfp, mfp, sch, mch, fr, fg, fb, br, bg, bb))
   DB u8_t mask
   DD font_t *sfp
   DD font_t *mfp
   DD u16_t sch
   DD u16_t mch
   DD u16_t fr
   DD u16_t fg
   DD u16_t fb
   DD u16_t br
   DD u16_t bg
   DD u16_t bb
   DE
{
   tp_t *tpp;
   chunk_t *chp;

   if ((tpp = new_tp(mask, 1)) == 0) {
      free(tpp);
      return 0;
   }
   warn("tptr_create: assigning a bogus id - not implemented\n");
   tpp->id = 0;

   if (mask & TP_IMAGE) {
      if ((tpp->tpimp = tptr_create_glyph_image(sfp, mfp, sch, mch,
								&chp)) == 0) {
         free_tp(tpp);
         return 0;
      }
      zb_dest(ZD_QSERV, 0);
      zb_queue_chunk(chp);
      /*
      **  can't draw tptr until glyph image is done
      */
      df_put_p((void *)tpp);
      cx_push(zrxqp, tptr_create_cont);
   }
   else {
      tpp->tpimp = rtpimp;
      zb_dest(ZD_ALLSERV, 0);
      tptr_draw_tp(tpp);
   }
   link_tp(tpp);
   return tpp;
}

void
tptr_create_cont
   AL((dfp))
   DB df_t *dfp
   DE
{
   tp_t *tpp = (tp_t *)df_get_p(dfp);

   zb_dest(ZD_ALLSERV, 0);
   tptr_draw_tp(tpp);
}

/************************************************************************
*									*
*   tptr_draw_tp							*
*									*
************************************************************************/
void
tptr_draw_tp
   AL((tpp))
   DB tp_t *tpp
   DE
{
   register mask_t mask;
   u32_t values[4];

   /*
   **  this is a slight hack, but the root window may not have been
   **  defined when the tpp structure was created
   */
   tpp->x = vscreen.wp->dwb.width / 2;
   tpp->y = vscreen.wp->dwb.height / 2;

   mask = CWBackPixmap | CWBackingStore | CWSaveUnder | CWEventMask;
   values[0] = tpp->tpimp->spmid;
   values[1] = xTrue;
   values[2] = xTrue;
   values[3] = ExposureMask;

   proto_CreateWindow(	vscreen.wp->dwb.depth,
			tpp->wid,
			vscreen.shellroot,
			tpp->x - tpp->tpimp->hotx,
			tpp->y - tpp->tpimp->hoty,
			tpp->tpimp->width,
			tpp->tpimp->height,
			0,
			InputOutput,
			vscreen.wp->vp->cid,
			mask,
			values,
			vscreen.wp->mp->mappixels);
}

/************************************************************************
*									*
*   tptr_map_tp								*
*									*
************************************************************************/
void
tptr_map_tp
   AL((tpp))
   DB tp_t *tpp
   DE
{
   if (tpp->show && tpp->refs)
      proto_MapWindow(tpp->wid);
}

/************************************************************************
*									*
*   tptr_create_image							*
*									*
************************************************************************/
tpim_t *
tptr_create_image
   AL((spxp, mpxp, x, y, chpp))
   DB pixmap_t *spxp
   DD pixmap_t *mpxp		/* can be null */
   DD u16_t x			/* hot spot */
   DD u16_t y
   DD chunk_t **chpp		/* queue to all */
   DE
{
   register int i, j;
   register tpim_t *tpimp;

   if ((tpimp = new_tpim(mpxp!=0)) == 0)
      return 0;

   /*
   **	create source and mask pixmaps on every screen
   */
   tpimp->hotx = x;
   tpimp->hoty = y;
   tpimp->width = spxp->dwb.width;	/* this better match mpxp */
   tpimp->height = spxp->dwb.height;

   zb_dest(ZD_NONE, 0);		/* clear buffer */

   tptr_draw_pixmaps(tpimp);

   proto_CopyArea(	spxp->cid,
			tpimp->sbmid,
			vscreen.dv[0].gc,
			0, 0, 0, 0,
			tpimp->width,
			tpimp->height);
   /*
   **	on the screen where the user's pixmap is, copy the image
   **	(there's no need to PutImage that one)
   */
   proto_CopyPlane(	spxp->cid,
			tpimp->spmid,
			vscreen.dp->gc,
			0, 0, 0, 0,
			tpimp->width,
			tpimp->height,
			0x1);
   if (mpxp) {
      proto_CopyArea(	spxp->cid,
			tpimp->sbmid,
			vscreen.dv[0].gc,
			0, 0, 0, 0,
			tpimp->width,
			tpimp->height);
      proto_CopyPlane(	mpxp->cid,
			tpimp->mpmid,
			vscreen.dp->gc,
			0, 0, 0, 0,
			tpimp->width,
			tpimp->height,
			0x1);
   }
   /*
   **	queue to all servers
   */
   *chpp = zb_split();

   return tpimp;
}

/************************************************************************
*									*
*   tptr_create_glyph_image						*
*									*
*	Create new glyph telepointer images.				*
*									*
************************************************************************/
tpim_t *
tptr_create_glyph_image
   AL((sfp, mfp, sch, mch, chpp))
   DB font_t *sfp
   DD font_t *mfp		/* can be null */
   DD u16_t sch		/* these are server-byte-order shorts */
   DD u16_t mch
   DD chunk_t **chpp
   DE
{
   register tpim_t *tpimp;

   if ((tpimp = new_tpim(mfp!=0)) == 0)
      return 0;

   zb_dest(ZD_NONE, 0);

   /*
   **  CreateGlyphCursor takes a STRING16, but QueryTextExtents takes
   **  a CHAR2B.  On little-endian machines, we must swap to move from
   **  one to the other.
   */
   if (endian == 'l') {
      SWAP2(sch);
      SWAP2(mch);
   }
   /*
   **  we only need the source size, we assume the mask, if any, matches
   */
   proto_QueryTextExtents(sfp->cid, 1, &sch);
   df_put_p((void *)tpimp);
   df_put_p((void *)sfp);
   df_put_p((void *)mfp);
   df_put_i((int)sch);
   df_put_i((int)mch);
   rx_queue(zrxqp, X_QueryTextExtents, 0, zb_seqno(), query_text_reply);
   *chpp = zb_split();

   /*
   **  This is a hack.  It is bad to do this since we don't know for
   **  sure that the chunk is going to the query server (from this
   **  module's view).  If it doesn't we hang forever.  The problem
   **  is that we mustn't process other protocol before this completes,
   **  esp mouse/tptr processing, which may reference resources that
   **  the routine below creates.
   */
   fd_pending(qservp->fd);

   return tpimp;
}

static void
query_text_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register s16_t xoff, yoff;
   register xQueryTextExtentsReply *rp=(xQueryTextExtentsReply *)buf_data(chp);
   register tpim_t *tpimp = (tpim_t *)df_get_p(dfp);
   register font_t *sfp = (font_t *)df_get_p(dfp);
   register font_t *mfp = (font_t *)df_get_p(dfp);
   u16_t sch = (u16_t)df_get_i(dfp);
   u16_t mch = (u16_t)df_get_i(dfp);
   u32_t vals[8];

   /*
   **  Hack.  See above.
   */
   fd_unpending();

   if (rp->type == X_Error) {
      warn("tptr.c/query_text_reply: request failed\n");
      dprx_error((xError *)rp);
      return;
   }
   xoff = rp->drawDirection == FontLeftToRight ?
					-rp->overallLeft :
					rp->overallRight;
   yoff = rp->overallAscent;

   tpimp->hotx = xoff;
   tpimp->hoty = yoff;

   tpimp->width = (u16_t)(rp->overallWidth < 0 ?
				- rp->overallWidth :
				rp->overallWidth);
   tpimp->height = rp->overallAscent + rp->overallDescent;
   if (tpimp->width == 0 || tpimp->height == 0) {
      /*
      **  this error is not handled very gracefully - in particular,
      **  the pixmaps not created below will eventually cause protocol
      **  errors - TODO.
      */
      warn("tptr.c/query_text_reply: cursor glyph has zero dimensions\n");
      return;
   }
   zb_dest(ZD_ALLSERV, 0);

   tptr_draw_pixmaps(tpimp);
   /*
   **  draw into bitmaps
   */
   vals[0] = 0;
   proto_ChangeGC(vscreen.dv[0].gc, GCForeground, vals, 0);
   proto_PolyFillRectangle(tpimp->sbmid, vscreen.dv[0].gc, 0, 0,
					tpimp->width, tpimp->height);
   vals[0] = 1;
   vals[1] = sfp->cid;
   proto_ChangeGC(vscreen.dv[0].gc, GCForeground | GCFont, vals, 0);
   proto_ImageText16(tpimp->sbmid, vscreen.dv[0].gc, xoff, yoff, 1, &sch);
   if (tpimp->mbmid) {
      vals[0] = 0;
      proto_ChangeGC(vscreen.dv[0].gc, GCForeground, vals, 0);
      proto_PolyFillRectangle(tpimp->mbmid, vscreen.dv[0].gc, 0, 0,
					tpimp->width, tpimp->height);
      vals[1] = mfp->cid;
      proto_ChangeGC(vscreen.dv[0].gc, GCForeground | GCFont, vals, 0);
      proto_ImageText16(tpimp->mbmid, vscreen.dv[0].gc, xoff, yoff, 1, &mch);
   }
   /*
   **  draw into background pixmaps
   */
   proto_ChangeGC(vscreen.dp->gc, GCForeground, &vscreen.whitePixel,
						vscreen.wp->mp->mappixels);
   proto_PolyFillRectangle(tpimp->spmid, vscreen.dp->gc, 0, 0,
					tpimp->width, tpimp->height);
   vals[0] = vscreen.blackPixel;
   vals[1] = sfp->cid;
   proto_ChangeGC(vscreen.dp->gc, GCForeground | GCFont, vals,
						vscreen.wp->mp->mappixels);
   proto_ImageText16(tpimp->spmid, vscreen.dp->gc, xoff, yoff, 1, &sch);
   if (tpimp->mpmid) {
      proto_ChangeGC(vscreen.dp->gc, GCForeground, &vscreen.whitePixel,
						vscreen.wp->mp->mappixels);
      proto_PolyFillRectangle(tpimp->mpmid, vscreen.dp->gc, 0, 0,
					tpimp->width, tpimp->height);
      vals[1] = mfp->cid;
      proto_ChangeGC(vscreen.dp->gc, GCForeground | GCFont, vals,
						vscreen.wp->mp->mappixels);
      proto_ImageText16(tpimp->mpmid, vscreen.dp->gc, xoff, yoff, 1, &mch);
   }
   zb_queue(0);
}

/************************************************************************
*									*
*   tptr_change								*
*									*
*	Change a tptr.							*
*									*
************************************************************************/
void
tptr_change
   VOID
{
}

/************************************************************************
*									*
*   tptr_set								*
*									*
*	Associate a tptr with a server.					*
*									*
************************************************************************/
void
tptr_set
   AL((sp, tpp))
   DB server_t *sp
   DD tp_t *tpp
   DE
{
   if (sp->tpp) {
      if (sp->mode == Floor)
         tptr_deref(sp);
      if (sp->tpp->sp == sp)
         sp->tpp->sp = 0;
   }
   sp->tpp = tpp;

   if (tpp && sp->mode == Floor)
      tptr_ref(sp);
}

/************************************************************************
*									*
*   tptr_show								*
*									*
************************************************************************/
void
tptr_show
   AL((tpp))
   DB tp_t *tpp
   DE
{
   if (tpp->show == 0) {
      tpp->show = 1;

      if (tpp->refs) {
         zb_dest(ZD_ALLBUTONE, tpp->sp);
         proto_MapWindow(tpp->wid);
         zb_queue(0);
      }
   }
}

/************************************************************************
*									*
*   tptr_hide								*
*									*
************************************************************************/
void
tptr_hide
   AL((tpp))
   DB tp_t *tpp
   DE
{
   if (tpp->show) {
      tpp->show = 0;

      if (tpp->refs) {
         zb_dest(ZD_ALLBUTONE, tpp->sp);
         proto_UnmapWindow(tpp->wid);
         zb_queue(0);
      }
   }
}

/************************************************************************
*									*
*   tptr_move								*
*									*
*	This should only be called on shown telepointers, as no		*
*	check is done here.						*
*									*
************************************************************************/
/*
**	locate telepointer image associated with window
*/
#define tpim_of_window(_tpimp, wp)\
   for (_tpimp=0; _tpimp == 0;)\
      if ((wp)->cursp)\
         _tpimp = (wp)->cursp->tpimp;\
      else if ((wp)->parent)\
         wp = (wp)->parent;\
      else\
         _tpimp = rtpimp

void
tptr_move
   AL((tpp, sp, wp, x, y))
   DB tp_t *tpp
   DD server_t *sp
   DD window_t *wp
   DD s16_t x
   DD s16_t y
   DE
{
   register tpim_t *tpimp;
   mask_t mask;
   u32_t values[4];

   if (tpp->x != x || tpp->y != y) {
      if (sp != tpp->sp) {
         zb_dest(ZD_SERV, sp);
         proto_UnmapWindow(tpp->wid);
      }
      zb_dest(ZD_ALLBUTONE, sp);

      mask = 0;
      if (tpp->wp != wp) {
         tpp->wp = wp;
         if ((tpp->mask & TP_IMAGE) == 0) {
            tpim_of_window(tpimp, wp);
            tpp->tpimp = tpimp;

            zb_dest(ZD_ALLSERV, 0);	/* overrides zb_dest above */
            proto_ChangeWindowAttributes(tpp->wid,
					CWBackPixmap,
					&tpimp->spmid,
					vscreen.wp->mp->mappixels);
            mask |= CWWidth | CWHeight;
            values[2] = tpimp->width;
            values[3] = tpimp->height;
         }
      }
      mask |= CWX | CWY;
      values[0] = (u32_t)x - tpp->tpimp->hotx;
      values[1] = (u32_t)y - tpp->tpimp->hoty;

      proto_ConfigureWindow(tpp->wid, mask, values);

      if (sp != tpp->sp) {
         if (tpp->sp) {
            zb_dest(ZD_SERV, tpp->sp);
            proto_MapWindow(tpp->wid);
         }
         tpp->sp = sp;
      }
      tpp->x = x;
      tpp->y = y;
   }
}

/************************************************************************
*									*
*   tptr_ref								*
*									*
*	The server will begin providing input to its telepointer.  It	*
*	must not already be doing that.					*
*									*
************************************************************************/
void
tptr_ref
   AL((sp))
   DB server_t *sp
   DE
{
   if (sp->tpp) {
      if (sp->tpp->refs == 0) {
         if (sp->tpp->show) {
            zb_dest(ZD_ALLBUTONE, sp);
            proto_MapWindow(sp->tpp->wid);
            zb_queue(0);
         }
         sp->tpp->sp = sp;
      }
      sp->tpp->refs++;
   }
}

/************************************************************************
*									*
*   tptr_deref								*
*									*
*	The server will no longer provide input to its telepointer,	*
*	though it will continue to be associated with it.		*
*									*
************************************************************************/
void
tptr_deref
   AL((sp))
   DB server_t *sp
   DE
{
   if (sp->tpp) {
      if (sp->tpp->refs == 1) {
         if (sp->tpp->show) {
            zb_dest(ZD_ALLBUTONE, sp->tpp->sp);	/* okay if null */
            proto_UnmapWindow(sp->tpp->wid);
            zb_queue(0);
         }
         sp->tpp->sp = 0;
      }
      sp->tpp->refs--;
   }
}

/************************************************************************
*									*
*   tptr_destroy							*
*									*
*	Destroy a telepointer.  The default telepointer may not be	*
*	destroyed.							*
*									*
************************************************************************/
void
tptr_destroy
   AL((tpp))
   DB tp_t *tpp
   DE
{
   register int i;

   if (tpp == deftpp)	/* nope */
      return;

   for (i=0; i<num_serv; i++)		/* kill refs */
      if (servers[i]->tpp == tpp)
         servers[i]->tpp = 0;

   /* destroy all resources here - TODO */

   unlink_tp(tpp);
   free_tp(tpp);
}

/************************************************************************
*									*
*   tptr_ketchup							*
*									*
*	Assumes all tptr images have been created already.		*
*									*
************************************************************************/
void
tptr_ketchup
   VOID
{
   register tp_t *tpp;

   tptr_draw_tp(deftpp);
   if (num_serv > 1)
      tptr_map_tp(deftpp);

   for (tpp=tptrs; tpp; tpp=tpp->next) {
      tptr_draw_tp(tpp);
      if (tpp->sp)
         tptr_map_tp(tpp);
   }
}

/************************************************************************
*									*
*   tptr_pixmap_ketchup							*
*									*
************************************************************************/
int
tptr_pixmap_ketchup
   VOID
{
   register int i;
   register tp_t *tpp;

   for (tpp=tptrs; tpp; tpp=tpp->next) {

      if ((tpp->mask & TP_IMAGE) == 0)
         continue;

      tptr_draw_pixmaps(tpp->tpimp);
   }
}

/************************************************************************
*									*
*   tptr_get_images							*
*									*
*	Generate GetImage requests for telepointer pixmap images.	*
*	Sets up xmx<->server dialog.  This routine is called when a	*
*	server is added.						*
*									*
************************************************************************/
void
tptr_get_images
   AL((dest))
   DB int dest
   DE
{
   register tp_t *tpp;

   for (tpp=tptrs; tpp; tpp=tpp->next)
      if (tpp->mask & TP_IMAGE) {
         proto_GetImage(	XYPixmap,
				tpp->tpimp->sbmid,
				0, 0,
				tpp->tpimp->width, tpp->tpimp->height, 1);
         df_put_i((int)dest);
         df_put_p((void *)tpp->tpimp);
         df_put_i(0);			/* zero means source image */
         rx_queue(zrxqp, X_GetImage, 0, zb_seqno(), tptr_image_reply);

         if (tpp->tpimp->mbmid) {
            proto_GetImage(	XYPixmap,
				tpp->tpimp->mbmid,
				0, 0,
				tpp->tpimp->width, tpp->tpimp->height, 1);
            df_put_i((int)dest);
            df_put_p((void *)tpp->tpimp);
            df_put_i(1);		/* one means mask image */
            rx_queue(zrxqp, X_GetImage, 0, zb_seqno(), tptr_image_reply);
         }
      }
}

/************************************************************************
*									*
*   tptr_image_reply							*
*									*
*	An incoming GetImage reply is transformed into an outgoing	*
*	PutImage.							*
*									*
*	Note that unlike pixmap_image_reply, this routine always	*
*	transmits bitmaps (cursor images are always depth 1), and it	*
*	makes no provision for large images, since cursors are never	*
*	that large (we hope).						*
*									*
************************************************************************/
void
tptr_image_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register int dest = df_get_i(dfp);
   register tpim_t *tpimp = (tpim_t *)df_get_p(dfp);
   register int domask = (u16_t)df_get_i(dfp);
   register xGetImageReply *rp = (xGetImageReply *)buf_data(chp);
   register pp_t *pp;

   if (rp->type == X_Error) {
      warn("tptr_image_reply: request failed\n");
      dprx_error((xError *)rp);
      return;
   }
   zb_dest(dest, 0);

   proto_PutImage(	XYBitmap,
			domask ? tpimp->mpmid : tpimp->spmid,
			vscreen.dp->gc, 0, 0,
			tpimp->width, tpimp->height,
			0, 1);
   /*
   **  this is a tag, for matching up a chunk that will follow later
   **  with this outgoing chunk.
   */
   pp = pp_image(rp->length * 4, sp->smap.iordv[1], tpimp->width, 1);
   pp_assign(sp->pp, pp);		/* set up pp for incoming data */
   zb_queue(pp);

   df_put_i(dest);
   rx_push(zrxqp, X_GetImage, 0, seqno, pixmap_image_reply_data);
}

/************************************************************************
*									*
*   tptr_find								*
*									*
*	This needs to be a hash table. (TODO)				*
*									*
************************************************************************/
tp_t *
tptr_find
   AL((id))
   DB rid_t id
   DE
{
   tp_t *tpp;

   if (deftpp->id == id)
      tpp = deftpp;
   else
      for (tpp=tptrs; tpp; tpp=tpp->next)
         if (tpp->id == id)
            break;

   return tpp;
}

/************************************************************************
*									*
*   tptr_translate							*
*									*
*	Translate a pointer event on a telepointer window into an	*
*	equivalent event on the window below it.			*
*									*
************************************************************************/
window_t *
tptr_translate
   AL((tpp, x, y, xp, yp))
   DB tp_t *tpp
   DD s16_t x
   DD s16_t y
   DD s16_t *xp
   DD s16_t *yp
   DE
{
   register s16_t rootx, rooty;
   window_t *wp;

   rootx = tpp->x - tpp->tpimp->hotx + x;
   rooty = tpp->y - tpp->tpimp->hoty + y;

   inp_root_translate(rootx, rooty, &wp, xp, yp);

   return wp;
}

/************************************************************************
*									*
*   tptr_draw_pixmaps							*
*									*
************************************************************************/
void
tptr_draw_pixmaps
   AL((tpimp))
   DB tpim_t *tpimp
   DE
{
   proto_CreatePixmap(		tpimp->sbmid,
				vscreen.shellroot,
				tpimp->width,
				tpimp->height,
				1);
   proto_CreatePixmap(		tpimp->spmid,
				vscreen.shellroot,
				tpimp->width,
				tpimp->height,
				vscreen.wp->dwb.depth);
   if (tpimp->mbmid) {
      proto_CreatePixmap(	tpimp->mbmid,
				vscreen.shellroot,
				tpimp->width,
				tpimp->height,
				1);
      proto_CreatePixmap(	tpimp->mpmid,
				vscreen.shellroot,
				tpimp->width,
				tpimp->height,
				vscreen.wp->dwb.depth);
   }
}

/*
**   new_tp
**
**	Allocate telepointer internal structure
*/
static tp_t *
new_tp
   AL((mask, show))
   DB u8_t mask
   DD u8_t show
   DE
{
   register tp_t *tpp;

   if (MALLOC(tpp, tp_t *, sizeof(tp_t)))
      return 0;

   tpp->id = util_new_server_id();	/* not right!  TODO */
   tpp->mask = mask;
   tpp->show = show;
   tpp->refs = 0;
   tpp->x = 0;
   tpp->y = 0;
   tpp->wp = 0;
   tpp->sp = 0;

   tpp->wid = util_new_client_id();
   tpp->dwb.type = W_TPTR;
   tpp->dwb.res.client = 0;	/* all telepointers are client zero */
   hash_add(vmap, tpp->wid, tpp->wid, (char *)tpp);

   /* default colors */
   tpp->fg = vscreen.blackPixel;
   tpp->bg = vscreen.whitePixel;

   return tpp;
}

/*
**   free_tp
**
**	Free telepointer structure and mark rid's for removal
*/
static void
free_tp
   AL((tpp))
   DB tp_t *tpp
   DE
{
   pixel_t cols[2];

   if (tpp->mask & TP_COLOR) {
      cols[0] = tpp->fg;
      cols[1] = tpp->bg;
      if (cmap_free_cells(vscreen.wp->mp->cmap, 0, 2, cols, 0))
         warn("free_tp: cannot free tptr colors\n");
      hash_mark(vmap, tpp->wid);
   }
   if (tpp->tpimp)
      if (tpp->mask & TP_IMAGE)
         free_tpim(tpp->tpimp);
      else
         ; /* dereference tpimp... TODO */

   free(tpp);
}

static void
link_tp
   AL((tpp))
   DB tp_t *tpp
   DE
{
   tpp->next = tptrs;
   tpp->prev = 0;
   tptrs = tpp;
   if (tpp->next)
      tpp->next->prev = tpp;
}

static void
unlink_tp
   AL((tpp))
   DB tp_t *tpp
   DE
{
   if (tpp->prev)
      tpp->prev->next = tpp->next;
   else if (tptrs == tpp)
      tptrs = tpp->next;
   if (tpp->next)
      tpp->next->prev = tpp->prev;
}

/*
**   new_tpim
**
**	Allocate telepointer image structure and associated rid's
*/
static tpim_t *
new_tpim
   AL((domask))
   DB int domask
   DE
{
   tpim_t *tpimp;

   if (MALLOC(tpimp, tpim_t *, sizeof(tpim_t)))
      return 0;

   tpimp->sbmid = util_new_client_id();
   hash_add(vmap, tpimp->sbmid, tpimp->sbmid, (char *)0);
   tpimp->spmid = util_new_client_id();
   hash_add(vmap, tpimp->spmid, tpimp->spmid, (char *)0);

   if (domask) {
      tpimp->mbmid = util_new_client_id();
      hash_add(vmap, tpimp->mbmid, tpimp->mbmid, (char *)0);
      tpimp->mpmid = util_new_client_id();
      hash_add(vmap, tpimp->mpmid, tpimp->mpmid, (char *)0);
   }
   else {
      tpimp->mbmid = 0;
      tpimp->mpmid = 0;
   }
   /*
   **	This is bogus for glyph cursors - fix it TODO
   */
   tpimp->refs = 0;

   return tpimp;
}

/*
**   free_tpim
**
**	Free image source structure and mark rid's for removal
*/
static void
free_tpim
   AL((tpimp))
   DB tpim_t *tpimp
   DE
{
   hash_mark(vmap, tpimp->sbmid);
   hash_mark(vmap, tpimp->spmid);
   if (tpimp->mbmid) {
      hash_mark(vmap, tpimp->mbmid);
      hash_mark(vmap, tpimp->mpmid);
   }
   free (tpimp);
}
