/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   reply.c								*
*									*
************************************************************************/
#include <sys/time.h>
#include <sys/types.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/Xproto.h>
#include <X11/X.h>
#include <xmcp.h>
#include "xmx.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "zb.h"
#include "rs.h"
#include "es.h"
#include "res.h"
#include "ptc.h"
#include "sm.h"
#include "kpm.h"
#include "incl/reply.pvt.h"
#include "fd.h"

/************************************************************************
*									*
*   reply_process							*
*									*
************************************************************************/
#define CHUNK(n)\
   chp = buf_split(bp, (n))

#define UNMAP(id)\
   (id) = (((id) & sp->smap.base) == sp->smap.base ?\
      hash_unmap(vmap, ((id) & ~sp->smap.base) >> sp->smap.shift) :\
      hash_unmap(sp->smap.mp, (id)))

void
reply_process
   AL((sp, bp))
   DB server_t *sp
   DD buffer_t *bp
   DE
{
   register uint_t n, len;
   register u8_t type;
   u8_t clinum;
   u16_t seqno;
   register int i, seq;
   register int clearit;
   register chunk_t *chp;
   register client_t *cp;
   register window_t *wp;
   register drawable_t *dp;
   register xConnSetupPrefix *prefix;
   register xGenericReply *rp;
   register xError *ep;
   register xEvent *ev;
   register pp_t *pp;
   register rxq_t *rxqp;
   register rx_t *rxp;

   switch (sp->state) {
      case S_INPROGRESS:
         warn("reply_process: server '%s' in S_INPROGRESS!\n", sp->tag);
         return;
      case S_POKED:
         chp = buf_chunk(bp);
         if (buf_chunksize(chp) >= sz_xConnSetupPrefix) {
            prefix = (xConnSetupPrefix *)buf_data(chp);
            len = sz_xConnSetupPrefix + prefix->length * 4;
            if (buf_chunksize(chp) >= len) {
               chp = buf_split(bp, len);
               if (rxp = rx_get(zrxqp, 0))
                  rx_continue(zrxqp, rxp, sp, 0, 0, chp);
               else {
                  warn("reply_process: lost setup context!\n");
                  rx_print(zrxqp);
               }
               buf_clear(chp);
            }
         }
         return;  /* even if there is something more, we're not ready */
      default:
         break;
   }
   while (buf_active(bp)) {
      if (sp->pp) {	/* we've got an unfinished reply to deal with */
         if (chp = pp_process(bp, &sp->pp)) {
            sm_get_last(sp->smp, &clinum, &seqno);
            cp = client_ptr(clinum);
            rxqp = rx_q(cp);
            if (rxp = rx_get(rxqp, seqno))
               rx_continue(rxqp, rxp, sp, cp, seqno, chp);
            else {
               warn("reply_process: lost partial context, seqno [%d/%d]\n",
							clinum, seqno);
               rx_print(rxqp);
            }
            buf_clear(chp);
         }
         else		/* not enough data */
            break;

         continue;
      }/* else... */

      if (buf_active(bp) < sz_xGenericReply)
         break;					/* done */

      rp = (xGenericReply *)buf_curdata(bp);
      if (rp->type == KeymapNotify)	/* these have no seqno in them */
         sm_get_last(sp->smp, &clinum, &seqno);
      else
         if (sm_get(sp->smp, rp->sequenceNumber, &clinum, &seqno) < 0) {
            warn("reply_process: server [%d/%s] lost sequence number [%d] %s\n",
		sp->fd, sp->tag, rp->sequenceNumber, dprx_event_str(rp->type));
#ifdef HANGS_IT_BUT_NOT_FATAL
            sm_print(sp->smp);
#endif
         }
      switch (rp->type) {
         case X_Error:
            ep = (xError *)rp;
            cp = client_ptr(clinum);

            if (gets_reply(ep->majorCode, ep->minorCode)) {
               rxqp = rx_q(cp);
               if (rxp = rx_get(rxqp, seqno)) {
                  if (chp = unmap_error(bp, sp, seqno)) {
                     rx_continue(rxqp, rxp, sp, cp, seqno, chp);
                     buf_clear(chp);
                  }
               }
               else {
                  warn("reply_process: lost error context, seqno [%d/%d]\n",
							clinum, seqno);
                  rx_print(rxqp);
                  if (chp = unmap_error(bp, sp, seqno))
                     buf_clear(chp);
                  else
                     goto breakloop;
               }
            }
            else if (cp) {
               /*
               **  non-reply client-bound protocol error
               */
               DEBUG1(D_NETWORK, "<error for [%d]>\n", cp ? cp->fd : 0);
               D_CALL1(D_PROTO3, dprx_error, ep);
               chp = unmap_error(bp, sp, seqno);

               /* filter multiple errors */
               if (cp->oseqno != seqno)
                  queue_add(cp->qp, chp);

               buf_clear(chp);
            }
            else {
               /*
               **  non-reply client zero protocol error
               */
               warn("PROTOCOL ERROR: server [%d] [%s]\n", sp->fd, sp->tag);
               dprx_error((xError *)rp);
               chp = unmap_error(bp, sp, seqno);
               buf_clear(chp);
            }
            break;
         case X_Reply:
            cp = client_ptr(clinum);
            rxqp = rx_q(cp);
            if (rxp = rx_get(rxqp, seqno)) {
               if (chp = unmap_reply(bp,sp,rx_major(rxp),rx_minor(rxp),seqno)){
                  rx_continue(rxqp, rxp, sp, cp, seqno, chp);
                  buf_clear(chp);
               }
               else
                  goto breakloop;
            }
            else {
               warn("reply_process: lost reply context, seqno [%d/%d]\n",
							clinum, seqno);
               rx_print(rxqp);
               if (chp = unmap_reply(bp, sp, 0, 0, seqno))
                  buf_clear(chp);
               else
                  goto breakloop;
            }
            break;
         default: /* EVENTS */
            ev = (xEvent *)rp;
            D_CALL1(D_PROTO3, dprx_event, ev);
            es_mark(bp);

            switch (type = (ev->u.u.type & 0x7f)) {
               case MotionNotify:	/* skip all but last MotionNotify */
                  if (	buf_active(bp) > sz_xEvent &&
			(ev[1].u.u.type & 0x7f) == MotionNotify) {
                     CHUNK(sz_xEvent);
                     break;
                  }
                  /* else fall through */
               case KeyPress:
               case KeyRelease:
               case ButtonPress:
               case ButtonRelease:
                  UNMAP(ev->u.keyButtonPointer.root);
                  UNMAP(ev->u.keyButtonPointer.event);
                  if (ev->u.keyButtonPointer.child)
                     UNMAP(ev->u.keyButtonPointer.child);
                  CHUNK(sz_xEvent);
                  /*
                  **  We want to know when dp==0, don't just ignore it. TODO
                  */
                  dp = (drawable_t *)hash_data(vmap,
						ev->u.keyButtonPointer.event);
                  if (dp) {
                     switch (dp->type) {
                        case W_TPTR:
			   dp = (drawable_t *)tptr_translate((tp_t *)dp,
					ev->u.keyButtonPointer.eventX,
					ev->u.keyButtonPointer.eventY,
					&ev->u.keyButtonPointer.eventX,
					&ev->u.keyButtonPointer.eventY);
                           /* fall through */
                        case W_WINDOW:
                           inp_device(sp, (window_t *)dp, chp, type);
                           break;
                     }
                  }
                  break;
               case EnterNotify:
                  UNMAP(ev->u.enterLeave.root);
                  UNMAP(ev->u.enterLeave.event);
                  if (ev->u.enterLeave.child)
                     UNMAP(ev->u.enterLeave.child);
                  CHUNK(sz_xEvent);
                  break;
               case LeaveNotify:
                  UNMAP(ev->u.enterLeave.root);
                  UNMAP(ev->u.enterLeave.event);
                  if (ev->u.enterLeave.child)
                     UNMAP(ev->u.enterLeave.child);
                  CHUNK(sz_xEvent);
                  break;
               case FocusIn:
                  UNMAP(ev->u.focus.window);
                  CHUNK(sz_xEvent);
                  break;
               case FocusOut:
                  UNMAP(ev->u.focus.window);
                  CHUNK(sz_xEvent);
                  break;
               case KeymapNotify:
                  CHUNK(sz_xEvent);
                  break;
               case Expose:
                  if (ev->u.expose.count) {
                     n = buf_active(bp) / sz_xEvent;
                     n = MIN(n, ev->u.expose.count+1);
                  }
                  else
                     n = 1;
                  CHUNK(n*sz_xEvent);
                  for (i=0; i<n; i++)
                     UNMAP(ev[i].u.expose.window);
                  if (dp = (drawable_t *)hash_data(vmap,ev->u.expose.window))
                     switch (dp->type) {
                        case W_WINDOW:
                           wp = (window_t *)dp;
                           if (wp->parent == 0)	/* virtual root window */
                              cliz_root_expose(sp, chp, n);
                           else {
                              es_set(chp, n);
                              expos_queue(wp, chp, n);
                           }
                           break;
                        case W_TPTR:
                           cliz_root_expose(sp, chp, n);
                     }
                  break;
               case GraphicsExpose:
                  UNMAP(ev->u.graphicsExposure.drawable);
                  CHUNK(sz_xEvent);
                  if (clinum && client_ptr(clinum))
                     event_sendto(chp, client_ptr(clinum));
                  break;
               case NoExpose:
                  UNMAP(ev->u.noExposure.drawable);
                  CHUNK(sz_xEvent);
                  if (clinum && client_ptr(clinum))
                     event_sendto(chp, client_ptr(clinum));
                  break;
               case VisibilityNotify:
                  UNMAP(ev->u.visibility.window);
                  CHUNK(sz_xEvent);
                  wp = (window_t *)hash_data(vmap,ev->u.visibility.window);
                  if (wp == 0) {
                     DEBUG1(D_REPLY, "reply_process: VisibilityNotify bad window! [%x]\n",
						ev->u.visibility.window);
                  }
                  else
                     event_send(chp, 1, wp, VisibilityChangeMask, 0);
                  break;
               case CreateNotify:
                  UNMAP(ev->u.createNotify.parent);
                  UNMAP(ev->u.createNotify.window);
                  CHUNK(sz_xEvent);
                  break;
               case DestroyNotify:
                  UNMAP(ev->u.destroyNotify.event);
                  UNMAP(ev->u.destroyNotify.window);
                  CHUNK(sz_xEvent);
                  dp = (drawable_t *)hash_data(vmap,ev->u.destroyNotify.window);
                  if (dp == 0) {
                     DEBUG1(D_REPLY,
			"reply_process: DestroyNotify bad window! [%x]\n",
						ev->u.destroyNotify.window);
                  }
                  else if (dp->type == W_SHELL)
                     xmc_drop(sp);

                  /* else ignore it */
                  break;
               case UnmapNotify:
                  UNMAP(ev->u.unmapNotify.event);
                  UNMAP(ev->u.unmapNotify.window);
                  CHUNK(sz_xEvent);
                  break;
               case MapNotify:
                  UNMAP(ev->u.mapNotify.event);
                  UNMAP(ev->u.mapNotify.window);
                  CHUNK(sz_xEvent);
                  break;
               case MapRequest:
                  UNMAP(ev->u.mapRequest.parent);
                  UNMAP(ev->u.mapRequest.window);
                  CHUNK(sz_xEvent);
                  break;
               case ReparentNotify:
                  UNMAP(ev->u.reparent.event);
                  UNMAP(ev->u.reparent.window);
                  UNMAP(ev->u.reparent.parent);
                  CHUNK(sz_xEvent);
                  break;
               case ConfigureNotify:
                  UNMAP(ev->u.configureNotify.event);
                  UNMAP(ev->u.configureNotify.window);
                  if (ev->u.configureNotify.aboveSibling)
                     UNMAP(ev->u.configureNotify.aboveSibling);
                  CHUNK(sz_xEvent);
                  dp = (drawable_t *)hash_data(	vmap,
						ev->u.configureNotify.window);
                  if (dp->type == W_SHELL) {
		     sp->smap.root.ok = 0;
                     sp->smap.root.width = ev->u.configureNotify.width;
                     sp->smap.root.height = ev->u.configureNotify.height;
                  }
                  break;
               case ConfigureRequest:
                  UNMAP(ev->u.configureRequest.parent);
                  UNMAP(ev->u.configureRequest.window);
                  if (ev->u.configureRequest.sibling)
                     UNMAP(ev->u.configureRequest.sibling);
                  CHUNK(sz_xEvent);
                  break;
               case GravityNotify:
                  UNMAP(ev->u.gravity.event);
                  UNMAP(ev->u.gravity.window);
                  CHUNK(sz_xEvent);
                  break;
               case ResizeRequest:
                  UNMAP(ev->u.resizeRequest.window);
                  CHUNK(sz_xEvent);
                  break;
               case CirculateNotify:
                  UNMAP(ev->u.circulate.event);
                  UNMAP(ev->u.circulate.window);
                  UNMAP(ev->u.circulate.parent);
                  CHUNK(sz_xEvent);
                  break;
               case CirculateRequest:
                  UNMAP(ev->u.property.window);
                  CHUNK(sz_xEvent);
                  break;
               case PropertyNotify:
                  UNMAP(ev->u.property.window);
                  CHUNK(sz_xEvent);
                  break;
               case SelectionClear:
                  UNMAP(ev->u.selectionClear.window);
                  CHUNK(sz_xEvent);
                  break;
               case SelectionRequest:
                  UNMAP(ev->u.selectionRequest.owner);
                  UNMAP(ev->u.selectionRequest.requestor);
                  CHUNK(sz_xEvent);
                  break;
               case SelectionNotify:
                  UNMAP(ev->u.selectionNotify.requestor);
                  CHUNK(sz_xEvent);
                  break;
               case ColormapNotify:
                  UNMAP(ev->u.colormap.window);
                  if (ev->u.colormap.colormap)
                     UNMAP(ev->u.colormap.colormap);
                  CHUNK(sz_xEvent);
                  break;
               case ClientMessage:
                  UNMAP(ev->u.clientMessage.window);
                  CHUNK(sz_xEvent);
                  break;
               case MappingNotify:
                  CHUNK(sz_xEvent);
                  zb_dest(ZD_SERV, sp);
                  switch (ev->u.mappingNotify.request) {
                     case MappingKeyboard:
                        kpm_update_keys(ev->u.mappingNotify.firstKeyCode,
					ev->u.mappingNotify.count);
                     case MappingModifier:
                        kpm_update_mods();
                     case MappingPointer:
                        kpm_update_buttons();
                  }
                  break;
               default:
                  chp = 0;	/* this is fatal */
                  quit(-1, "reply_process: unknown event - fatal error\n");
                  break;
            }
            buf_clear(chp);
            break;
      }
   }
   breakloop:
   return;
}

#undef CHUNK

/************************************************************************
*									*
*   reply_server_death							*
*									*
*	Manufacture replies to pending requests for a server that	*
*	has terminated unexpectedly.					*
*									*
************************************************************************/
void
reply_server_death
   AL((sp))
   DB server_t *sp
   DE
{
   register u16_t sseqno, first, next;
   register rxq_t *rxqp;
   register rx_t *rxp;
   u8_t clinum;
   u16_t cseqno, rseqno;

   if (sp->state == S_POKED) {
      server_state(sp, S_ZOMBIE);
      /*
      **  If a connection failed after being deferred because it is
      **  non-blocking, there is a pending reply context from the
      **  client setup block (initial "poke").  Since it was never
      **  actually sent (because the connection was never made) there
      **  will be no response, so we would hang.
      **
      **  Clear that context now.
      **
      **  Theoretically, this results in a gentle drop of the server
      **  in question as other operations continue.
      */
      if (rxp = rx_get(zrxqp, 0))
         rx_continue(zrxqp, rxp, sp, 0, 0, 0);
   }
   else {
      server_state(sp, S_ZOMBIE);
      first = sm_last_reply(sp->smp) + 1;
      next = sm_next_seqno(sp->smp);

      for (sseqno=first; sseqno!=next; sseqno++)
         if (sm_get(sp->smp, sseqno, &clinum, &cseqno) < 0) {
            warn("reply_server_death: server [%d/%s] lost seq number [%d]\n",
						sp->fd, sp->tag, sseqno);
         }
         else {	/* success */
            rxqp = rx_q(client_ptr(clinum));
            if (rx_next_seqno(rxqp, &rseqno)) {
               if (rseqno == cseqno)
                  if (rxp = rx_get(rxqp, cseqno))
                     rx_continue(rxqp, rxp, sp, 0, cseqno, 0);
            }
         }
   }
}

/*
**
**  unmap_reply
**
**	Unmap the data in a reply or error, and split it into a chunk
**	from its buffer.  Returns a chunk with the unmapped reply in
**	it, or zero if not enough data was there.
**
*/
#define CHUNK(n)\
   chp = buf_split(bp, (n))

static chunk_t *
unmap_reply
   AL((bp, sp, major, minor, seqno))
   DB buffer_t *bp
   DD server_t *sp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DE
{
   register int i, len;
   register u32_t *lp;
   register chunk_t *chp;
   register xGenericReply *rp = (xGenericReply *)buf_curdata(bp);

   /*
   **  it's assumed that there is at least sz_xGenericReply bytes
   */
   len = sz_xGenericReply;
   if (rp->length) {
      len += rp->length * 4;
      if (buf_active(bp) < len)
         return 0;
   }
   switch (major) {
      case 0:
         CHUNK(len);
         break;
      case X_GetWindowAttributes:
         {  register xGetWindowAttributesReply *p =
					(xGetWindowAttributesReply *)rp;
            UNMAP(p->visualID);
            UNMAP(p->colormap);
            CHUNK(len);
         }
         break;
      case X_GetGeometry:
         {  register xGetGeometryReply *p = (xGetGeometryReply *)rp;
            UNMAP(p->root);
            CHUNK(len);
         }
         break;
      case X_QueryTree:
         {  register xQueryTreeReply *p = (xQueryTreeReply *)rp;
            UNMAP(p->root);
            UNMAP(p->parent);
            CHUNK(len);
         }
         break;
      case X_InternAtom:
         {  register xInternAtomReply *p = (xInternAtomReply *)rp;
            CHUNK(len);
         }
         break;
      case X_GetAtomName:
         {  register xGetAtomNameReply *p = (xGetAtomNameReply *)rp;
            CHUNK(len);
         }
         break;
      case X_GetProperty:
         {  register xGetPropertyReply *p = (xGetPropertyReply *)rp;
            CHUNK(len);
         }
         break;
      case X_ListProperties:
         {  register xListPropertiesReply *p =
						(xListPropertiesReply *)rp;
            CHUNK(len);
         }
         break;
      case X_GetSelectionOwner:
         {  register xGetSelectionOwnerReply *p =
						(xGetSelectionOwnerReply *)rp;
            UNMAP(p->owner);
            CHUNK(len);
         }
         break;
      case X_GrabPointer:
         {  register xGrabPointerReply *p = (xGrabPointerReply *)rp;
            CHUNK(len);
         }
         break;
      case X_GrabKeyboard:
         {  register xGrabKeyboardReply *p =
						(xGrabKeyboardReply *)rp;
            CHUNK(len);
         }
         break;
      case X_QueryPointer:
         {  register xQueryPointerReply *p =
						(xQueryPointerReply *)rp;
            UNMAP(p->root);
            UNMAP(p->child);
            CHUNK(len);
         }
         break;
      case X_GetMotionEvents:
         {  register xGetMotionEventsReply *p =
						(xGetMotionEventsReply *)rp;
            /* need a TRUNC_TIMECOORD() here */
            CHUNK(len);
         }
         break;
      case X_TranslateCoords:
         {  register xTranslateCoordsReply *p =
						(xTranslateCoordsReply *)rp;
            UNMAP(p->child);
            CHUNK(len);
         }
         break;
      case X_GetInputFocus:
         {  register xGetInputFocusReply *p =
						(xGetInputFocusReply *)rp;
            UNMAP(p->focus);
            CHUNK(len);
         }
         break;
      case X_QueryKeymap:
         {  register xQueryKeymapReply *p = (xQueryKeymapReply *)rp;
            CHUNK(len);
         }
         break;
      case X_QueryFont:
         {  register xQueryFontReply *p = (xQueryFontReply *)rp;
            /* need a TRUNC_FONTINFO() here */
            CHUNK(len);
         }
         break;
      case X_QueryTextExtents:
         {  register xQueryTextExtentsReply *p =
						(xQueryTextExtentsReply *)rp;
            CHUNK(len);
         }
         break;
      case X_ListFonts:
         {  register xListFontsReply *p = (xListFontsReply *)rp;
            /* could TRUNC_BYTES() here */
            CHUNK(len);
         }
         break;
      case X_ListFontsWithInfo:
         {  register xListFontsWithInfoReply *p =
						(xListFontsWithInfoReply *)rp;
            /* need a TRUNC_FONTINFO() here (but eat name first) */
            CHUNK(len);
         }
         break;
      case X_GetFontPath:
         {  register xGetFontPathReply *p =
						(xGetFontPathReply *)rp;
            CHUNK(len);
         }
         break;
      case X_GetImage:
         {  register xGetImageReply *p = (xGetImageReply *)rp;

            UNMAP(p->visual);
            CHUNK(sz_xGetImageReply);
         }
         break;
      case X_ListInstalledColormaps:
         {  register xListInstalledColormapsReply *p =
					(xListInstalledColormapsReply *)rp;
            lp = (u32_t *)(p+1);
            for (i=0; i<(int)p->nColormaps; i++)
               UNMAP(lp[i]);
            CHUNK(len);
         }
         break;
      case X_AllocColor:
         {  register xAllocColorReply *p = (xAllocColorReply *)rp;
            CHUNK(len);
         }
         break;
      case X_AllocNamedColor:
         {  register xAllocNamedColorReply *p =
						(xAllocNamedColorReply *)rp;
            CHUNK(len);
         }
         break;
      case X_AllocColorCells:
         {  register xAllocColorCellsReply *p =
						(xAllocColorCellsReply *)rp;
            CHUNK(len);
         }
         break;
      case X_AllocColorPlanes:
         {  register xAllocColorPlanesReply *p =
						(xAllocColorPlanesReply *)rp;
            CHUNK(len);
         }
         break;
      case X_QueryColors:
         {  register xQueryColorsReply *p =
						(xQueryColorsReply *)rp;
            CHUNK(len);
         }
         break;
      case X_LookupColor:
         {  register xLookupColorReply *p =
						(xLookupColorReply *)rp;
            CHUNK(len);
         }
         break;
      case X_QueryBestSize:
         {  register xQueryBestSizeReply *p =
						(xQueryBestSizeReply *)rp;
            CHUNK(len);
         }
         break;
      case X_QueryExtension:
         {  register xQueryExtensionReply *p =
						(xQueryExtensionReply *)rp;
            CHUNK(len);
         }
         break;
      case X_ListExtensions:
         {  register xListExtensionsReply *p =
						(xListExtensionsReply *)rp;
            CHUNK(len);
         }
         break;
      case X_GetKeyboardMapping:
         {  register xGetKeyboardMappingReply *p =
						(xGetKeyboardMappingReply *)rp;
            CHUNK(len);
         }
         break;
      case X_GetKeyboardControl:
         {  register xGetKeyboardControlReply *p =
						(xGetKeyboardControlReply *)rp;
            CHUNK(len);
         }
         break;
      case X_GetPointerControl:
         {  register xGetPointerControlReply *p =
						(xGetPointerControlReply *)rp;
            CHUNK(len);
         }
         break;
      case X_GetScreenSaver:
         {  register xGetScreenSaverReply *p =
						(xGetScreenSaverReply *)rp;
            CHUNK(len);
         }
         break;
      case X_ListHosts:
         {  register xListHostsReply *p = (xListHostsReply *)rp;
            CHUNK(len);
         }
         break;
      case X_SetPointerMapping:
         {  register xSetPointerMappingReply *p =
						(xSetPointerMappingReply *)rp;
            CHUNK(len);
         }
         break;
      case X_GetPointerMapping:
         {  register xGetPointerMappingReply *p =
						(xGetPointerMappingReply *)rp;
            CHUNK(len);
         }
         break;
      case X_SetModifierMapping:
         {  register xSetModifierMappingReply *p =
						(xSetModifierMappingReply *)rp;
            CHUNK(len);
         }
         break;
      case X_GetModifierMapping:
         {  register xGetModifierMappingReply *p =
						(xGetModifierMappingReply *)rp;
            CHUNK(len);
         }
         break;
      default:
         if (major > X_NoOperation)
            chp = ext_unmap_reply(bp, sp, major, minor, seqno);
         else {
            warn("reply_unmap: bad reply type [%d] [%s]\n",
					major, dprx_reqType_str(major));
            CHUNK(len);
         }
         break;
   }
   if (chp) {
      rp->sequenceNumber = seqno;
      rs_mark(chp, major, minor);
   }
   return chp;
}

static chunk_t *
unmap_error
   AL((bp, sp, seqno))
   DB buffer_t *bp
   DD server_t *sp
   DD u16_t seqno
   DE
{
   register xError *ep;
   register chunk_t *chp;

   chp = buf_split(bp, sz_xError);
   ep = (xError *)buf_data(chp);

   ep->sequenceNumber = seqno;

   switch (ep->errorCode) {
      case BadWindow:
      case BadPixmap:
      case BadCursor:
      case BadFont:
      case BadDrawable:
      case BadGC:
      case BadIDChoice:
         UNMAP(ep->resourceID);
         break;
   }
   chp->type = P_ERROR;
   return chp;
}

static int
gets_reply
   AL((major, minor))
   DB u8_t major
   DD u16_t minor
   DE
{
   if (major < 128)	/* extensions occupy opcodes 128-255 */
      switch (major) {
         case X_GetWindowAttributes:
         case X_GetGeometry:
         case X_QueryTree:
         case X_InternAtom:
         case X_GetAtomName:
         case X_GetProperty:
         case X_ListProperties:
         case X_GetSelectionOwner:
         case X_GrabPointer:
         case X_GrabKeyboard:
         case X_QueryPointer:
         case X_GetMotionEvents:
         case X_TranslateCoords:
         case X_GetInputFocus:
         case X_QueryKeymap:
         case X_QueryFont:
         case X_QueryTextExtents:
         case X_ListFonts:
         case X_ListFontsWithInfo:
         case X_GetFontPath:
         case X_GetImage:
         case X_ListInstalledColormaps:
         case X_AllocColor:
         case X_AllocNamedColor:
         case X_AllocColorCells:
         case X_AllocColorPlanes:
         case X_QueryColors:
         case X_LookupColor:
         case X_QueryBestSize:
         case X_QueryExtension:
         case X_ListExtensions:
         case X_GetKeyboardMapping:
         case X_GetKeyboardControl:
         case X_GetPointerControl:
         case X_GetScreenSaver:
         case X_ListHosts:
         case X_SetPointerMapping:
         case X_GetPointerMapping:
         case X_SetModifierMapping:
         case X_GetModifierMapping:
            return 1;
      }
   else
      return ext_gets_reply(major, minor);

   return 0;
}
