/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   imask.c								*
*									*
*	Input masks.							*
*									*
************************************************************************/
#include "xmx.h"
#include "incl/imask.pvt.h"

/*
**	client/mask pair
*/
typedef struct _masknode_t {
   mask_t		mask;
   char *		cp;	/* client (x or xmc client) */
   struct _masknode_t	*next;
}masknode_xxx;

/************************************************************************
*									*
*   imask_alloc								*
*									*
************************************************************************/
inmask_t *
imask_alloc
   VOID
{
   inmask_t *imp;

   if (MALLOC(imp, inmask_t *, sizeof(inmask_t)))
      return (inmask_t *)0;

   imp->mask = 0;
   imp->mnp = 0;

   return imp;
}

/************************************************************************
*									*
*   imask_put								*
*									*
*	Create or replace the mask value of an masknode list element	*
*	for the given input mask data structure.  Also recalculates	*
*	the composite input mask.					*
*									*
************************************************************************/
int
imask_put
   AL((imp, cp, mask))
   DB inmask_t *imp
   DD char *cp		/* x or xmc client */
   DD mask_t mask	/* new mask */
   DE
{
   masknode_t *mp, *lp;

   lp = 0;
   for (mp=imp->mnp; mp; mp=mp->next)
      if (mp->cp == cp)
         break;
      else
         lp = mp;

   if (mp == 0) {
      if (MALLOC(mp, masknode_t *, sizeof(masknode_t)))
         return -1;

      mp->cp = cp;
      mp->next = 0;

      if (lp)
         lp->next = mp;
      else
         imp->mnp = mp;
   }
   mp->mask = mask;

   imp->mask = 0;
   for (mp=imp->mnp; mp; mp=mp->next)
      imp->mask |= mp->mask;

   return 0;
}

/************************************************************************
*									*
*   imask_get								*
*									*
*	Returns the mask value corresponding to the given client.	*
*									*
************************************************************************/
mask_t
imask_get
   AL((imp, cp))
   DB inmask_t *imp
   DD char *cp
   DE
{
   masknode_t *mp;

   for (mp=imp->mnp; mp; mp=mp->next)
      if (mp->cp == cp)
         return mp->mask;

   return 0;
}

/************************************************************************
*									*
*   imask_first								*
*   imask_next								*
*									*
*	Iterate through the clients that have selected input		*
*	corresponding to the given mask.				*
*									*
************************************************************************/
static masknode_t *mp_saved = 0;
static mask_t mask_saved = 0;

mask_t
imask_first
   AL((imp, msk, cpp))
   DB inmask_t *imp
   DD mask_t msk
   DD char **cpp
   DE
{
   register masknode_t *mp;
   register mask_t tm;

   for (mp=imp->mnp; mp; mp=mp->next)
      if (mp->mask & msk) {
         mask_saved = msk;		/* save for imask_next */
         tm = mp->mask;
         *cpp = mp->cp;
         mp_saved = mp->next;
         return tm;
      }
   return (mask_t)0;
}

mask_t
imask_next
   AL((cpp))
   DB char **cpp
   DE
{
   register masknode_t *mp;
   register mask_t tm;

   for (mp=mp_saved; mp; mp=mp->next)
      if (mp->mask & mask_saved) {
         tm = mp->mask;
         *cpp = mp->cp;
         mp_saved = mp->next;
         return tm;
      }

   return (mask_t)0;
}

/************************************************************************
*									*
*   imask_remove							*
*									*
*	Remove any masknode list element that exists for the given	*
*	client.  Recalculates the composite input mask.			*
*									*
*	Returns non-zero if the composite mask was changed.		*
*									*
************************************************************************/
int
imask_remove
   AL((imp, cp))
   DB inmask_t *imp
   DD char *cp
   DE
{
   register mask_t oldmask;
   masknode_t *mp, *lp;

   lp = 0;
   for (mp=imp->mnp; mp; mp=mp->next)
      if (mp->cp == cp) {
         if (lp)
            lp->next = mp->next;
         else
            imp->mnp = mp->next;
         if (mp == mp_saved)	/* if called from within an imask_next loop */
            mp_saved = mp->next;
         free(mp);
         oldmask = imp->mask;
         imp->mask = 0;
         for (mp=imp->mnp; mp; mp=mp->next)
            imp->mask |= mp->mask;
         return (oldmask != imp->mask);
      }
      else
         lp = mp;

   return 0;
}

/************************************************************************
*									*
*   imask_free								*
*									*
*	Remove and free the given input mask data structure.		*
*									*
************************************************************************/
void
imask_free
   AL((imp))
   DB inmask_t *imp
   DE
{
   register masknode_t *mp, *lp;

   for (mp=imp->mnp; lp=mp;) {
      mp = lp->next;
      if (lp == mp_saved)
         mp_saved = 0;	/* just in case */
      free(lp);
   }
   free(imp);
}
