/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   image.c								*
*									*
*	Image translation routines.					*
*									*
*	There are 36 possible bitmap orderings.  There are 6 possible	*
*	orderings for each Z format image of a particular depth.	*
*	Since we never deal with different depth images at the same	*
*	time, 36 orderings are sufficient to express all the possible	*
*	image translation pairings.					*
*									*
*	Images do not need to be translated for each server - some	*
*	servers share the same ordering.  So each image is kept as	*
*	part of an "image group" of images in various orderings, as	*
*	needed - so the translation is only done once for each		*
*	ordering.							*
*									*
*	The worst-case number of translations we have to make is	*
*	36 for bitmaps and 6 for Z images.				*
*									*
*	The way this works is that each ordering is registered with	*
*	image_order().  A new image is stored by a call to		*
*	image_source().  Then, image_swap() returns a chunk with	*
*	image data in any format needed - translated on demand.		*
*	A call to image_free dereferences a chunk from an image		*
*	group, freeing the group when the number of chunks goes to	*
*	zero.								*
*									*
************************************************************************/
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/X.h>
#include <X11/Xproto.h>
#include "xmx.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "image.h"
#include "incl/image.pvt.h"
#include "iswap.h"

/************************************************************************
*									*
*   image_xy_iord							*
*   image_z_iord							*
*									*
*	Return an opaque image order value for the given image		*
*	format parameters.						*
*									*
************************************************************************/
iord_t
image_xy_iord
   AL((bit_order, byte_order, scan_unit, scan_pad))
   DB u8_t bit_order
   DD u8_t byte_order
   DD u8_t scan_unit
   DD u8_t scan_pad
   DE
{
   return XYIORD(XYTYPE(bit_order, byte_order, scan_unit), scan_pad);
}

iord_t
image_z_iord
   AL((bpp, scan_pad))
   DB u8_t bpp
   DD u8_t scan_pad
   DE
{
   return ZIORD(ZTYPE(bpp), scan_pad);
}

/************************************************************************
*									*
*   image_length							*
*									*
*	Calculate the length of the data portion of a request or	*
*	reply for the given iord.					*
*									*
*	Returns a value in units of (byte*4).				*
*									*
************************************************************************/
int
image_length
   AL((iord, width, height, left_pad, depth))
   DB iord_t iord
   DD u16_t width
   DD u16_t height
   DD u8_t left_pad
   DD u8_t depth
   DE
{
   register uint_t Bps, len;
   register u8_t bpp, pad;

   width += left_pad;
   pad = IORD_PAD(iord);

   if (iord & ZBIT) {	/* ZPixmap */
      bpp = IORD_ZBPP(iord);
      Bps = BPS((uint_t)width, bpp, pad);
      len = (Bps * height)/4;
   }
   else {		/* XYBitmap or XYPixmap */
      Bps = BPS((uint_t)width, 1, pad);
      len = (Bps * height * depth)/4;
   }
   DEBUG2(D_IMAGE, "image_length: iord [%d] length [%d]\n", iord, len);

   return (int)len;
}

/************************************************************************
*									*
*   image_bite_size							*
*									*
*	Calculate the largest rectangular area of a given image size	*
*	that can be retrieved in a single GetImage request, in		*
*	xmx-natural bitmap format (always ZPixmap).			*
*									*
*	All X protocol requests and replies have a length field		*
*	expressed as an unsigned short.  That limits the size of the	*
*	data that can follow.  This is true of both PutImage requests	*
*	and GetImage replies.  Because PutImage requests include the	*
*	size of the header in the length (and replies do not), they	*
*	are the most constrained.  A PutImage header uses 6 words	*
*	of data, so the maximum image size expressible is		*
*	(biggest short value) - 6, or 65535 - 6 = 65529.  Note that 	*
*	that's 65529 4-byte words, or 262116 bytes.			*
*									*
************************************************************************/
void
image_bite_size
   AL((depth, widthp, heightp))
   DB u8_t depth
   DD u16_t *widthp
   DD u16_t *heightp
   DE
{
   register u16_t maxheight;
   register uint_t Wps;		/* 32 bit words per scanline */
   register u8_t bpp;

   /*
   **	pick conservative values for bpp and scan_pad
   */
   if (depth == 1)
      bpp = 1;
   else if (depth <= 8)
      bpp = 8;
   else
      bpp = 32;

   Wps = (BPS((uint_t)*widthp, bpp, 32) + 3) / 4;

   if (Wps <= 65529) {
      maxheight = 65529 / Wps;
      if (maxheight < *heightp)
         *heightp = maxheight;
   }
   else {
      *widthp /= (u16_t)((Wps + 65528)/ 65529);
      *heightp = 1;
   }
}

/************************************************************************
*									*
*   image_source							*
*									*
*	Sets the given chunk as the source data for a new image group.	*
*	The image should be in the format specified by iord.		*
*									*
************************************************************************/
chunk_t *
image_source
   AL((iord, width, depth, bp, n))
   DB iord_t iord
   DD uint_t width
   DD u8_t depth
   DD buffer_t *bp
   DD uint_t n
   DE
{
   register u8_t bpp, pad;
   register int i, imdex;
   register imlist_t *ilp;
   register chunk_t *chp;
   register uint_t Bps, scanlines;

   pad = IORD_PAD(iord);
   bpp = iord & ZBIT ? IORD_ZBPP(iord) : 1;

   Bps = BPS(width, bpp, pad);
   scanlines = n / Bps;

   if (scanlines) {
      imdex = IORD_IMDEX(iord);

      ilp = new_imlist();
      ilp->refs = 1;
      ilp->depth = depth;
      ilp->source = iord;
      ilp->width = width;
      ilp->scanlines = scanlines;
      ilp->bp = 0;

      for (i=0; i<IMDEX_SZ; i++)
         ilp->chp[i] = 0;

      chp = buf_split(bp, scanlines * Bps);
      ilp->chp[imdex] = chp;

      chp->type = P_IMAGE;
      chp->dptr = (void *)ilp;

      DEBUG4(D_IMAGE,"image_source: iord [%d] width [%d] depth [%d] n [%d]\n",
						iord, width, depth, n);
      DEBUG3(D_IMAGE, "\tBps [%d] scanlines [%d] new chunksize [%d]\n",
				Bps, ilp->scanlines, buf_chunksize(chp));
   }
   else
      chp = 0;

   return chp;
}

/************************************************************************
*									*
*   image_free								*
*									*
*	Free image group.  This should only be called from buf_clear.	*
*									*
************************************************************************/
void
image_free
   AL((chp))
   DB chunk_t *chp
   DE
{
   register int i;
   register imlist_t *ilp = (imlist_t *)chp->dptr;

   if (ilp->refs > 0) {
      DEBUG2(D_IMAGE, "image_free: [0%x] refs-- [%d]\n", ilp, ilp->refs);
      /*
      **  clear chunk pointer in image group structure
      */
      for (i=0; i<IMDEX_SZ; i++)
         if (ilp->chp[i] == chp) {
            ilp->chp[i] = 0;
            break;
         }
      /*
      **  if no more chunks, free it
      */
      if (--ilp->refs == 0)
         free_imlist(ilp);
   }
   else
      warn("image_free: trying to free freed image group\n");
}

/************************************************************************
*									*
*   image_swap								*
*									*
*	Return a buffer chunk containing the given image data in the	*
*	given format.  Data is translated on demand.			*
*									*
************************************************************************/
chunk_t *
image_swap
   AL((chp, iordv))
   DB chunk_t *chp
   DD iord_t *iordv	/* array of MAXDEPTH+1 */
   DE
{
   register int n, imdex, src_imdex;
   register iord_t iord;
   register u8_t pad, bpp;
   register imlist_t *ilp = (imlist_t *)chp->dptr;
   void (*iswapfunc)();

   iord = iordv[ilp->depth];
   imdex = IORD_IMDEX(iord);

   DEBUG2(D_IMAGE, "image_swap: [0x%x] iord [%d]\n", ilp, iord);

   if (ilp->chp[imdex] == 0) {
      if (ilp->bp == 0)
         ilp->bp = buf_new(B_FREEONWRITE);
					/* create a new buffer/chunk */
      pad = IORD_PAD(iord);
      if (iord & ZBIT) {
	 bpp = IORD_ZBPP(iord);
         iswapfunc = iswapZfunc[IORD_ZTYPE(iord)];
      }
      else {
         bpp = 1;
         iswapfunc = iswapXYfunc[IORD_XYTYPE(ilp->source)][IORD_XYTYPE(iord)];
      }
      n = ilp->scanlines * BPS(ilp->width, bpp, pad);

      buf_adjust(ilp->bp, n);
      buf_grow(ilp->bp, n);
      ilp->chp[imdex] = buf_split(ilp->bp, n);

      ilp->chp[imdex]->type = P_IMAGE;
      ilp->chp[imdex]->dptr = (void *)ilp;
      
      src_imdex = IORD_IMDEX(ilp->source);
      /*
      ** swap it
      */
      (*iswapfunc) (	buf_data(ilp->chp[src_imdex]),
			buf_data(ilp->chp[imdex]),
			IORD_PAD(ilp->source),
			pad,
			bpp,
			ilp->width,
			ilp->scanlines);
      ilp->refs++;
      /*
      **  Note that the following are useful as tags only, the data
      **  in them may not be valid for this new chunk.  However, once
      **  a pp has been assigned to a chunk, it is only used as a tag
      **  (or queue ordering purposes).
      */
      pp_assign(ilp->chp[imdex]->lpp, ilp->chp[src_imdex]->lpp);
      pp_assign(ilp->chp[imdex]->tpp, ilp->chp[src_imdex]->tpp);
   }
   return ilp->chp[imdex];
}

void
image_client_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register pp_t *pp;
   register int depth;
   register u8_t format = (u8_t)df_get_i(dfp);
   register u16_t width = (u16_t)df_get_i(dfp);
   register xGetImageReply *rp = (xGetImageReply *)buf_data(chp);

   if (rp->type == X_Error) {
      warn("image_client_reply: request failed\n");
      dprx_error((xError *)rp);
      return;
   }
   depth = format == ZPixmap ? rp->depth : 1;
   pp = pp_image(rp->length * 4, sp->smap.iordv[depth], width, rp->depth);
   pp_assign(sp->pp, pp);
   pp_assign(chp->tpp, pp);
   queue_add(cp->qp, chp);

   rx_push(cp->rxqp, X_GetImage, 0, seqno, queue_client_reply);
}

static imlist_t *freeilp;

static imlist_t *
new_imlist
   VOID
{
   register imlist_t *ilp;

   if (freeilp) {
      ilp = freeilp;
      freeilp = freeilp->next;
   }
   else if (MALLOC(ilp, imlist_t *, sizeof(imlist_t)))
      return 0;

   return ilp;
}

static void
free_imlist
   AL((ilp))
   DB imlist_t *ilp
   DE
{
   ilp->next = freeilp;
   freeilp = ilp;
}
