/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   hash.c								*
*									*
*	Hash table management routines.					*
*									*
*	Since the things we are storing are generally sequential, the	*
*	hash function is simply id modulo table_size.  This probably	*
*	isn't optimal for client resource ids, as clients are		*
*	typically differentiated by high order bits of the id.  In	*
*	that case, clusters of N entries in the table will appear,	*
*	where N is the number of clients.				*
*									*
*	All tables are actually two tables, a map and an inverse map.	*
*									*
************************************************************************/
#include "xmx.h"
#include "df.h"

#define HASH(id,tabsize)	((id)%(tabsize))

/*
**	Hash table entry structure
*/
typedef struct _entry_t {
	short			mark;	/* if >0, marked for removal */
	u32_t			key;	/* client id/atom */
	u32_t			val;	/* server id/atom */
	char *			data;	/* optional data pointer */
	struct _entry_t *	next;
}entry_t;

#include "incl/hash.pvt.h"

/*	hashtab_t defined in xmx.h */

/************************************************************************
*									*
*   hash_new								*
*									*
************************************************************************/
hashtab_t *
hash_new
   AL((tabsize))
   DB int tabsize
   DE
{
   hashtab_t *tab;

   if (MALLOC(tab, hashtab_t *, sizeof(hashtab_t)))
      return 0;

   if (CALLOC(tab->map, entry_t **, tabsize, sizeof(entry_t))) {
      free(tab);
      return 0;
   }
   if (CALLOC(tab->unmap, entry_t **, tabsize, sizeof(entry_t))) {
      free(tab->map);
      free(tab);
      return 0;
   }
   tab->dirty = 0;
   tab->size = tabsize;

   return tab;
}

void
hash_free
   AL((tab))
   DB hashtab_t *tab
   DE
{
   if (tab->map)
      free(tab->map);
   if (tab->unmap)
      free(tab->unmap);
   free(tab);
}

/************************************************************************
*									*
*  hash_add_client_id							*
*									*
*	Add a new client id to the virtual server hash table.  The	*
*	virtual id is manufactured in this routine and returned by	*
*	it.  If the client id is found in the table, see if it is	*
*	marked.  If marked, then technically it is free, so just	*
*	unmark it and use the virtual id found.  Otherwise it's an	*
*	error (return zero).						*
*									*
*	Note that when an entry is unmarked it will be used for two	*
*	completely different resources, one of which has been		*
*	destroyed, but for which all protocol may not yet have been	*
*	seen.  That's okay, because a destroyed resource has no data,	*
*	and the mapping (or unmapping) is the same for both.		*
*									*
************************************************************************/
u32_t
hash_add_client_id
   AL((key, data))
   DB u32_t key
   DD char *data
   DE
{
   register entry_t *ep;
   register u32_t val;
   register keyhash, valhash;

   keyhash = HASH(key, vmap->size);

   for (ep=vmap->map[keyhash]; ep; ep=ep->next)
      if (ep->key == key)	/* found it - not good */
         if (ep->mark) {	/* marked? - better */
            ep->mark = 0;
            ep->data = data;
            val = ep->val;
            /*
            **  check reverse map for consistency
            */
            for (ep=vmap->unmap[HASH(val, vmap->size)]; ep; ep=ep->next)
               if (ep->key == val)
                  if (ep->mark) {
                     ep->mark = 0;
                     ep->data = data;
                     DEBUG4(D_HASH, "%s: key[0x%x] data[0x%x] -> val[0x%x]\n",
					"hash_add_client_id", key, data, val);
                     return val;	/* success */
                  }
                  else
                     return err(0, "hash_add_client_id: table messed up\n");
            /*
            **  reverse entry not found, uh oh
            */
            return err(0, "hash_add_client_id: table inconsistent\n");
         }
         else
            return err(0, "hash_add_client_id: id [0x%x] already exists\n",key);
   /*
   **  not found, just plug it in
   */
   val = util_new_client_id();
   valhash = HASH(val, vmap->size);

   if (MALLOC(ep, entry_t *, sizeof(entry_t)))
      return 0;
   ep->mark = 0;
   ep->key = key;
   ep->val = val;
   ep->data = data;
   ep->next = vmap->map[keyhash];
   vmap->map[keyhash] = ep;

   if (MALLOC(ep, entry_t *, sizeof(entry_t))) {
      ep = vmap->map[keyhash];
      vmap->map[keyhash] = ep->next;
      free(ep);
      return 0;
   }
   ep->mark = 0;
   ep->key = val;
   ep->val = key;
   ep->data = data;
   ep->next = vmap->unmap[valhash];
   vmap->unmap[valhash] = ep;

   DEBUG3(D_HASH, "hash_add_client_id: key[0x%x] data[0x%x] -> val[0x%x]\n",
							key, data, val);
   return val;
}

/************************************************************************
*									*
*   hash_add								*
*									*
*	Add map and unmap entries to the given table.			*
*									*
************************************************************************/
int
hash_add
   AL((tab, key, val, data))
   DB hashtab_t *tab
   DD u32_t key
   DD u32_t val
   DD char *data
   DE
{
   register entry_t *ep;
   register keyhash, valhash;

   DEBUG4(D_HASH, "hash_add: tab[%x] key[%x] val[%x] data[%x]\n",
							tab, key, val, data);
   keyhash = HASH(key, tab->size);
   for (ep=tab->map[keyhash]; ep; ep=ep->next)
      if (ep->key == key)
         return err(-1, "hash_add: id [0x%x] in map already exists%s\n",
					key, ep->mark ? " (marked)" : "");
   valhash = HASH(val, tab->size);
   for (ep=tab->unmap[valhash]; ep; ep=ep->next)
      if (ep->key == val)
         return err(-1, "hash_add: id [0x%x] in unmap already exists%s\n",
					val, ep->mark ? " (marked)" : "");
   if (MALLOC(ep, entry_t *, sizeof(entry_t)))
      return -1;
   ep->mark = 0;
   ep->key = key;
   ep->val = val;
   ep->data = data;
   ep->next = tab->map[keyhash];
   tab->map[keyhash] = ep;

   if (MALLOC(ep, entry_t *, sizeof(entry_t))) {
      ep = tab->map[keyhash];
      tab->map[keyhash] = ep->next;
      free(ep);
      return -1;
   }
   ep->mark = 0;
   ep->key = val;
   ep->val = key;
   ep->data = data;
   ep->next = tab->unmap[valhash];
   tab->unmap[valhash] = ep;

   return 0;
}

/************************************************************************
*									*
*   hash_remove								*
*									*
************************************************************************/
int
hash_remove
   AL((tab, key))
   DB hashtab_t *tab
   DD u32_t key
   DE
{
   u32_t val;

   DEBUG2(D_HASH, "hash_remove: tab[%x] key[%x]\n", tab, key);
   val = hash_map(tab, key);

   if (map_remove(tab->map, tab->size, key))
      return -1;

   if (map_remove(tab->unmap, tab->size, val))
      return err(-1, "hash_remove: table inconsistent\n");

   return 0;
}

/*
**   map_remove
**
**	Remove a table entry.
*/
static int
map_remove
   AL((map, size, key))
   DB entry_t **map
   DD int size
   DD u32_t key
   DE
{
   register entry_t *ep, **epp;

   for (epp = &map[HASH(key, size)]; *epp; epp = &(*epp)->next)
      if ((*epp)->key == key) {
         ep = (*epp)->next;
         free(*epp);
         *epp = ep;

         return 0;
      }
   return err(-1, "map_remove: id [%x] does not exist\n", key);
}

/************************************************************************
*									*
*   hash_map								*
*									*
*	Map key to value.						*
*									*
*	If a mapping does not exist, quietly returns the given key.	*
*	This is a useful if somewhat tricky behavior - you'd expect	*
*	we'd want an error if something couldn't map.  Well, a truly	*
*	bad value will cause an error eventually, and we leave in	*
*	a debugging statement for tracking that down.			*
*									*
************************************************************************/
u32_t
hash_map
   AL((tab, key))
   DB hashtab_t *tab
   DD u32_t key
   DE
{
   register entry_t *ep;

   for (ep=tab->map[HASH(key, tab->size)]; ep; ep=ep->next)
      if (ep->key == key)
         return ep->val;

   DEBUG1(D_HASH, "hash_map: can't map [0x%x]\n", key);
   return key;
}

/************************************************************************
*									*
*   hash_unmap								*
*									*
*	Map value to key.						*
*									*
************************************************************************/
u32_t
hash_unmap
   AL((tab, val))
   DB hashtab_t *tab
   DD u32_t val
   DE
{
   register entry_t *ep;

   for (ep=tab->unmap[HASH(val, tab->size)]; ep; ep=ep->next)
      if (ep->key == val)
         return ep->val;

   DEBUG1(D_HASH, "hash_unmap: can't unmap [0x%x]\n", val);
   return val;		/* see comment for hash_map, above */
}

/************************************************************************
*									*
*   hash_data								*
*									*
*	Return data associated with key.				*
*									*
************************************************************************/
char *
hash_data
   AL((tab, key))
   DB hashtab_t *tab
   DD u32_t key
   DE
{
   register entry_t *ep;

   for (ep=tab->map[HASH(key, tab->size)]; ep; ep=ep->next)
      if (ep->key == key)
         return ep->data;

   return 0;
}

/************************************************************************
*									*
*   hash_undata								*
*									*
*	Return data associated with val.				*
*									*
************************************************************************/
char *
hash_undata
   AL((tab, val))
   DB hashtab_t *tab
   DD u32_t val
   DE
{
   register entry_t *ep;

   for (ep=tab->unmap[HASH(val, tab->size)]; ep; ep=ep->next)
      if (ep->key == val)
         return ep->data;

   return 0;
}

/************************************************************************
*									*
*   hash_mark								*
*									*
*	Mark hash table entry.  Marked entries are removed by		*
*	hash_cleanup.  This is used to delay removal of entries at	*
*	client termination until hash table is no longer needed.	*
*									*
************************************************************************/
int
hash_mark
   AL((tab, key))
   DB hashtab_t *tab
   DD u32_t key
   DE
{
   register entry_t *ep;
   register u32_t val;

   for (ep=tab->map[HASH(key, tab->size)]; ep; ep=ep->next)
      if (ep->key == key) {
         ep->mark = 1;
         ep->data = 0;
         tab->dirty = 1;
         val = ep->val;
         for (ep=tab->unmap[HASH(val, tab->size)]; ep; ep=ep->next)
            if (ep->key == val) {
               ep->mark = 1;
               ep->data = 0;
               return 0;
            }
         return err(-1, "hash_mark: table inconsistent\n");
      }

   return -1;
}

/************************************************************************
*									*
*   hash_reset								*
*									*
*	Clear maps.							*
*									*
************************************************************************/
void
hash_reset
   AL((tab))
   DB hashtab_t *tab
   DE
{
   map_reset(tab->map, tab->size);
   map_reset(tab->unmap, tab->size);
}

/*
**   map_reset
**
**	Scan hash table for entries and remove them.
*/
static void
map_reset
   AL((map, size))
   DB entry_t **map
   DD int size
   DE
{
   register int i;
   register entry_t *ep, *lp;

   for (i=0; i<size; i++)
      if (map[i]) {
         for (lp=map[i]; ep=lp;) {
            lp = lp->next;
            free(ep);
         }
         map[i] = 0;
      }
}

/************************************************************************
*									*
*   hash_cleanup							*
*									*
*	Collect garbage.						*
*									*
************************************************************************/
void
hash_cleanup
   AL((tab))
   DB hashtab_t *tab
   DE
{
   DEBUG1(D_HASH, "hash_cleanup: [%x]\n", tab);
   if (tab->dirty) {
      DEBUG0(D_HASH, "\tbefore:\n");
      D_CALL1(D_HASH, hash_print, tab);
      map_cleanup(tab->map, tab->size);
      map_cleanup(tab->unmap, tab->size);
      DEBUG0(D_HASH, "\tafter:\n");
      D_CALL1(D_HASH, hash_print, tab);
      tab->dirty = 0;
   }
}

void
hash_cleanup_cont
   AL((dfp))
   DB df_t *dfp
   DE
{
   hash_cleanup(vmap);
}

/*
**   map_cleanup
**
**	Scan hash table for marked entries and remove them.
*/
static void
map_cleanup
   AL((map, size))
   DB entry_t **map
   DD int size
   DE
{
   register int i;
   register entry_t **epp, *ep;

   for (i=0; i<size; i++)
      for (epp = &map[i]; *epp;)
         if ((*epp)->mark) {
            ep = *epp;
            *epp = (*epp)->next;
            free(ep);
         }
         else
            epp = &(*epp)->next;
}

/************************************************************************
*									*
*   hash_print								*
*									*
*	Dump a hash table (in ascii) to stderr.				*
*	Format is one bucket per line, curly bracketed entries list	*
*	{key,val[,MARK][,data]}, where square bracketed items are	*
*	only output if present.						*
*									*
************************************************************************/
void
hash_print
   AL((tab))
   DB hashtab_t *tab
   DE
{
   register int i;
   register entry_t *ep;
   char dbuf[32];

   warn("HASH TABLE [0x%x]:\n", tab);
   warn("\t->\n");
   for (i=0; i<tab->size; i++)
      if (tab->map[i]) {
         warn("\t[%04d]", i);
         for (ep=tab->map[i]; ep; ep=ep->next) {
            if (ep->data)
               sprintf(dbuf, ",0x%x", ep->data);
            else
               strcpy(dbuf, "");
            warn(" {%x,%x%s%s}", ep->key, ep->val, ep->mark ? ",MARK":"", dbuf);
         }
         warn("\n");
      }
   warn("\t<-\n");

   for (i=0; i<tab->size; i++)
      if (tab->unmap[i]) {
         warn("\t[%04d]", i);
         for (ep=tab->unmap[i]; ep; ep=ep->next) {
            if (ep->data)
               sprintf(dbuf, ",0x%x", ep->data);
            else
               strcpy(dbuf, "");
            warn(" {%x,%x%s%s}", ep->key, ep->val, ep->mark ? ",MARK":"", dbuf);
         }
         warn("\n");
      }
}
