/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   gc.c								*
*									*
************************************************************************/
#include <X11/Xproto.h>
#include <X11/X.h>
#include "xmx.h"
#include "df.h"
#include "zb.h"
#include "res.h"
#include "incl/gc.pvt.h"

static gc_t *gcs;

/************************************************************************
*									*
*   gc_create								*
*									*
************************************************************************/
gc_t *
gc_create
   AL((cp, p, chpp))
   DB client_t *cp
   DD xCreateGCReq *p
   DD chunk_t **chpp
   DE
{
   register int i;
   register drawable_t *dp;
   register chunk_t *chp;
   register gc_t *gp;
   register pixmap_t *pxp;
   register font_t *fp;
   register u32_t *vp;
   register mask_t mask;
   u32_t values[IGCount];

   if ((dp = (drawable_t *)hash_data(vmap, p->drawable)) == 0) {
      proto_Error(cp, BadDrawable, p->drawable, 0, X_CreateGC);
      return 0;
   }
   if (MALLOC(gp, gc_t *, sizeof(gc_t))) {
      proto_Error(cp, BadAlloc, 0, 0, X_CreateGC);
      return 0;
   }
   if ((gp->vid = hash_add_client_id(p->gc, (char *)gp)) == 0) {
      free(gp);
      proto_Error(cp, BadIDChoice, p->gc, 0, X_CreateGC);
      return 0;
   }
   gp->res.client = cp;
   gp->cid = p->gc;
   gp->tilepxp = 0;
   gp->stipplepxp = 0;
   gp->clipmaskpxp = 0;
   gp->fp = 0;
   gp->depth = dp->depth;
   gp->refs = 1;

   gp->wp = dp->type == W_WINDOW ? (window_t *)dp : vscreen.wp;
   vp = (u32_t *)(p+1);
   for (i=0; i<IGCount; i++)  /* load attributes (or defaults) */
      if (p->mask & 1<<i) {
         gp->atts[i] = *vp;
         switch (1L<<i) {
            case GCTile:
               if (pxp = (pixmap_t *)hash_data(vmap, *vp))
                  pixmap_assign(&gp->tilepxp, pxp);
               else {
                  proto_Error(cp, BadPixmap, *vp, 0, X_CreateGC);
                  return 0;
               }
               break;
            case GCStipple:
               if (pxp = (pixmap_t *)hash_data(vmap, *vp))
                  pixmap_assign(&gp->stipplepxp, pxp);
               else {
                  proto_Error(cp, BadPixmap, *vp, 0, X_CreateGC);
                  return 0;
               }
               break;
            case GCClipMask:
               if (*vp != None)
                  if (pxp = (pixmap_t *)hash_data(vmap, *vp))
                     pixmap_assign(&gp->clipmaskpxp, pxp);
                  else {
                     proto_Error(cp, BadPixmap, *vp, 0, X_CreateGC);
                     return 0;
                  }
               break;
            case GCFont:
               if (fp = (font_t *)hash_data(vmap, *vp))
                  font_assign(&gp->fp, fp);
               else {
                  proto_Error(cp, BadFont, *vp, 0, X_CreateGC);
                  return 0;
               }
               break;
         }
         vp++;
      }
      else
         gp->atts[i] = gc_defs[i];

   gp->last = 0;
   gp->next = gcs;
   if (gp->next)
      gp->next->last = gp;
   gcs = gp;
   /*
   **  These must be set.  If they are not, set them in a separate
   **  request.
   */
   mask = ~p->mask & (GCForeground | GCBackground | GCGraphicsExposures);
   if (mask) {
      zb_dest(ZD_NONE, 0);

      i = 0;
      if (mask & GCForeground)
         values[i++] = gc_defs[IGForeground];
      if (mask & GCBackground)
         values[i++] = gc_defs[IGBackground];
      if (mask & GCGraphicsExposures)
         values[i++] = xTrue;

      proto_ChangeGC(gp->cid, mask, values, gp->wp->mp->mappixels);
      *chpp = zb_split();
   }
   cp->refs++;

   return gp;
}

/************************************************************************
*									*
*   gc_change								*
*									*
************************************************************************/
gc_t *
gc_change
   AL((cp, p))
   DB client_t *cp
   DD xChangeGCReq *p
   DE
{
   register int i;
   register gc_t *gp;
   register pixmap_t *pxp;
   register font_t *fp;
   register u32_t *vp;

   if ((gp = (gc_t *)hash_data(vmap, p->gc)) == 0) {
      proto_Error(cp, BadGC, p->gc, 0, X_ChangeGC);
      return 0;
   }
   vp = (u32_t *)(p+1);
   for (i=0; i<IGCount; i++)
      if (p->mask & 1<<i) {
         gp->atts[i] = *vp;
         switch (1L<<i) {
            case GCTile:
               if (pxp = (pixmap_t *)hash_data(vmap, *vp))
                  pixmap_assign(&gp->tilepxp, pxp);
               else {
                  proto_Error(cp, BadPixmap, *vp, 0, X_ChangeGC);
                  return 0;
               }
               break;
            case GCStipple:
               if (pxp = (pixmap_t *)hash_data(vmap, *vp))
                  pixmap_assign(&gp->stipplepxp, pxp);
               else {
                  proto_Error(cp, BadPixmap, *vp, 0, X_ChangeGC);
                  return 0;
               }
               break;
            case GCClipMask:
               if (*vp == None)
                  pixmap_assign(&gp->clipmaskpxp, 0);
               else if (pxp = (pixmap_t *)hash_data(vmap, *vp))
                  pixmap_assign(&gp->clipmaskpxp, pxp);
               else {
                  proto_Error(cp, BadPixmap, *vp, 0, X_ChangeGC);
                  return 0;
               }
               break;
            case GCFont:
               if (fp = (font_t *)hash_data(vmap, *vp))
                  font_assign(&gp->fp, fp);
               else {
                  proto_Error(cp, BadFont, *vp, 0, X_ChangeGC);
                  return 0;
               }
               break;
         }
         vp++;
      }

/*
   if (p->mask & GCGraphicsExposures)
      if (wp->atts[IGGraphicsExposures] == 0)
         imask_remove(wp->xmask, (char *)cp);
      else
         imask_put(wpk->xmask, (char *)cp, wp->atts[IWEventMask]);
*/
   return gp;
}

/************************************************************************
*									*
*   gc_copy								*
*									*
************************************************************************/
int
gc_copy
   AL((cp, p))
   DB client_t *cp
   DD xCopyGCReq *p
   DE
{
   register int i;
   register gc_t *sgp, *dgp;

   if ((sgp = (gc_t *)hash_data(vmap, p->srcGC)) == 0) {
      proto_Error(cp, BadDrawable, p->srcGC, 0, X_CopyGC);
      return -1;
   }
   if ((dgp = (gc_t *)hash_data(vmap, p->dstGC)) == 0) {
      proto_Error(cp, BadDrawable, p->dstGC, 0, X_CopyGC);
      return -1;
   }
   if (sgp->depth != dgp->depth) {
      proto_Error(cp, BadMatch, 0, 0, X_CopyGC);
      return -1;
   }
   for (i=0; i<IGCount; i++)  /* copy attributes */
      if (p->mask & 1<<i) {
         dgp->atts[i] = sgp->atts[i];
         switch (1L<<i) {
            case GCTile:
               pixmap_assign(&dgp->tilepxp, sgp->tilepxp);
               break;
            case GCStipple:
               pixmap_assign(&dgp->stipplepxp, sgp->stipplepxp);
               break;
            case GCClipMask:
               pixmap_assign(&dgp->clipmaskpxp, sgp->clipmaskpxp);
               break;
            case GCFont:
               font_assign(&dgp->fp, sgp->fp);
               break;
         }
      }

   return 0;
}

/************************************************************************
*									*
*   gc_free								*
*									*
************************************************************************/
int
gc_free
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   gc_t *gp;

   if ((gp = (gc_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadGC, p->id, 0, X_FreeGC);
      return -1;
   }
   (void)free_gc(gp);

   return 0;
}

/************************************************************************
*									*
*   gc_client_death							*
*									*
************************************************************************/
void
gc_client_death
   AL((client))
   DB client_t *client
   DE
{
   register rid_t cid;
   register gc_t *gp, *ngp;

   for (ngp=gcs; gp=ngp;) {
      ngp = ngp->next;
      if (client_cmp(client, gp->res.client)) {
         cid = gp->cid;
         if (free_gc(gp) == 0) {
            proto_FreeGC(cid);
         }
         else {
            client->refs--;
            gp->res.client = 0;
         }
      }
   }
}

/************************************************************************
*									*
*   gc_ketchup								*
*									*
************************************************************************/
void
gc_ketchup
   VOID
{
   register int i, j, k;
   register rid_t wid;
   register mask_t mask;
   register gc_t *gp;
   u32_t atts[IGCount];

   for (gp=gcs; gp; gp=gp->next) {
      mask = 0;
      for (j=k=0; j<IGCount; j++)
         if (	gp->atts[j] != gc_defs[j] ||
		j == IGForeground ||
		j == IGBackground) {
            atts[k++] = gp->atts[j];
            mask |= 1<<j;
         }
      /*
      ** Pick a pixmap (drawable) of the appropriate depth
      ** for this gc.
      */
      if (gp->depth == vscreen.dp->depth)
         wid = vscreen.dp->pixmap;
      else {
         wid = 0;
         for (i=0; i<vscreen.ndepths; i++)
            if (vscreen.dv[i].depth == gp->depth) {
               wid = vscreen.dv[i].pixmap;
               break;
            }
         if (wid == 0) {
            warn("gc_ketchup: no depth [%d] for gc!  This is bad.\n",
							gp->depth);
            continue;
         }
      }
      proto_CreateGC(gp->cid, wid, mask, atts, gp->wp->mp->mappixels);
   }
}

/************************************************************************
*									*
*   gc_reset								*
*									*
************************************************************************/
void
gc_reset
   VOID
{
   register rid_t cid;
   register gc_t *gp, *ngp;

   for (ngp=gcs; gp=ngp;) {
      ngp = ngp->next;
      cid = gp->cid;
      if (free_gc(gp) == 0)
         proto_FreeGC(cid);
   }
}

/*
**	prevent dangling window references
*/
void
gc_fix_window
   AL((wp))
   DB window_t *wp
   DE
{
   register gc_t *gp;

   for (gp=gcs; gp; gp=gp->next)
      if (gp->wp == wp)
         gp->wp = vscreen.wp;
}

static int
free_gc
   AL((gp))
   DB gc_t *gp
   DE
{
   if (--gp->refs == 0) {
      if (gp->last)
         gp->last->next = gp->next;
      if (gcs == gp)
         gcs = gp->next;
      if (gp->next)
         gp->next->last = gp->last;
      hash_mark(vmap, gp->cid);
      if (gp->tilepxp)
         pixmap_deref(gp->tilepxp);
      if (gp->stipplepxp)
         pixmap_deref(gp->stipplepxp);
      if (gp->clipmaskpxp)
         pixmap_deref(gp->clipmaskpxp);
      gp->res.client->refs--;
      free(gp);
      return 0;
   }
   return -1;
}
