/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   auth.c								*
*									*
*	Maintain all host authorization data.				*
*									*
************************************************************************/
#include <stdio.h>
#include <sys/param.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>

#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/Xproto.h>

#include "xmx.h"

#define DISPLAY_WILD	0xffff

/*
**	types and globals for connecting to remote X servers
*/
typedef struct _auth_t {
	hostaddr_t		host;
	u16_t			display;
	u16_t			namelen;
	char *			name;
	u16_t			datalen;
	char *			data;
	struct _auth_t *	next;
}auth_t;

#include "incl/auth.pvt.h"

static auth_t *authlist;	/* list of auth records */
static auth_t *authlast;	/* last record in list */

/*
**	types and globals for verifying clients
*/
typedef struct {
	u16_t		namelen;
	char *		name;
#ifdef NEED_PROTO
	int		(*add)(char *, u16_t);
	int		(*check)(char *, u16_t);
	int		(*remove)(char *, u16_t);
	int		(*reset)(void);
#else
	int		(*add)();
	int		(*check)();
	int		(*remove)();
	int		(*reset)();
#endif
}auth_proto_t;

#ifdef XDM_AUTHORIZATION_1
#include "xdmauth.h"
#endif
#ifdef SUN_DES_1
#include "rpcauth.h"
#endif

static int	authon = 0;

static auth_proto_t auths[] = {	/* list these "best one last" */
{	(u16_t)18,	"MIT-MAGIC-COOKIE-1",
	mitauth_add,	mitauth_check,	mitauth_remove,	mitauth_reset },
#ifdef XDM_AUTHORIZATION_1
{	(u16_t)19,	"XDM-AUTHORIZATION-1",
	xdmauth_add,	xdmauth_check,	xdmauth_remove,	xdmauth_reset },
#endif
#ifdef SUN_DES_1
{	(u16_t)9,	"SUN-DES-1",
	rpcauth_add,	rpcauth_check,	rpcauth_remove,	rpcauth_reset },
#endif
};
#define NUM_AUTH (sizeof(auths)/sizeof(auth_proto_t))

/************************************************************************
*									*
*   auth_on								*
*									*
*	Are there any authority records
*									*
************************************************************************/
int
auth_on
   VOID
{
   return authon;
}

/************************************************************************
*									*
*   auth_reset								*
*									*
*	Reset all lists.
*									*
************************************************************************/
void
auth_reset
   VOID
{
   register int i;
   register char *filename;

   for (i=0; i<NUM_AUTH; i++)
      auth_reset_local(auths[i].namelen, auths[i].name);

   authon = 0;
   if (filename = authfile())
      (void) read_authority(filename);

   DEBUG1(D_AUTH, "auth_reset: authority file is '%s'\n", filename);
}

/************************************************************************
*									*
*   auth_add								*
*									*
*	Add an authority entry to the list.				*
*									*
************************************************************************/
int
auth_add
   AL((family, length, address, display, namelen, name, datalen, data))
   DB u16_t family
   DD u16_t length
   DD char *address
   DD u16_t display
   DD u16_t namelen
   DD char *name
   DD u16_t datalen
   DD char *data
   DE
{
   register auth_t *ap;

   for (ap=authlist; ap; ap=ap->next)		/* ignore duplicates */
      if (	ap->host.family == family &&
		ap->host.length == length &&
		ap->namelen == namelen &&
		bcmp(ap->host.address, address, length) == 0 &&
		bcmp(ap->name, name, namelen) == 0)
         return 0;

   DEBUG1(D_AUTH, "auth_add: %s\n", rec_str(ap));

   if (MALLOC(ap, auth_t *, sizeof(auth_t)) == 0) {
      ap->host.family = family;
      ap->host.length = length;
      ap->host.address = (char *)0;
      ap->namelen = namelen;
      ap->name = (char *)0;
      ap->datalen = datalen;
      ap->data = (char *)0;
      ap->next = (auth_t *)0;
      if (MALLOC(ap->host.address, char *, length) == 0) {
         bcopy(address, ap->host.address, length);
         if (MALLOC(ap->name, char *, namelen) == 0) {
            bcopy(name, ap->name, namelen);
            if (MALLOC(ap->data, char *, datalen) == 0) {
               bcopy(data, ap->data, datalen);
               add_auth(ap);
               return 0;	/* success */
            }
         }
      }
   }
   if (ap)					/* failed */
      nuke_auth(ap);

   return -1;
}

/************************************************************************
*									*
*   auth_get								*
*									*
*	Find the best cookie for the given display.			*
*	Preference is given first to authorization protocols that do	*
*	NOT appear in the default auths array, THEN to those that	*
*	appear at higher auth array indices.				*
*									*
************************************************************************/
int
auth_get
   AL((family, length, address, display, namelen, name, datalen, data))
   DB u16_t family
   DD u16_t length
   DD char *address
   DD u16_t display
   DD u16_t *namelen
   DD char **name
   DD u16_t *datalen
   DD char **data
   DE
{
   register int i;
   register int max = -1;
   auth_t *ap, *bestap;

   if (family == FamilyLocal) {
      length = hostnamelen;
      address = hostname;
   }
   for (ap=authlist; ap; ap=ap->next)
      if ( ap->host.family == FamilyWild ||
           ( ap->host.family == family &&
             ( ap->host.length == 0 ||
               ( ap->host.length == length &&
                 bcmp(ap->host.address, address, length) == 0 &&
                 ( ap->display == DISPLAY_WILD ||
                   ap->display == display))))) {
         for (i=0; i<NUM_AUTH; i++)
            if (	ap->namelen == auths[i].namelen &&
			bcmp(ap->name, auths[i].name, ap->namelen) == 0)
               break;
         if (i > max) {
            bestap = ap;
            max = i;
         }
      }
   if (max >= 0) {
      DEBUG1(D_AUTH, "auth_get: %s\n", rec_str(bestap));
      *namelen = bestap->namelen;
      *name = bestap->name;
      *datalen = bestap->datalen;
      *data = bestap->data;
      return 0;
   }
   else {
      DEBUG0(D_AUTH, "auth_get: no match\n");
      return -1;
   }
}

/************************************************************************
*									*
*   auth_add_local							*
*									*
*									*
************************************************************************/
int
auth_add_local
   AL((namelen, name, datalen, data))
   DB u16_t namelen
   DD char *name
   DD u16_t datalen
   DD char *data
   DE
{
   register int i, rv;

   DEBUG1(D_AUTH, "auth_add_local: %s\n",
				local_str(name, namelen, data, datalen));

   for (i=0; i<NUM_AUTH; i++)
      if (auths[i].namelen == namelen && bcmp(auths[i].name,name,namelen)==0) {
         if ((rv = (*auths[i].add)(data, datalen)) == 0) {
            authon = 1;
         }
         return rv;
      }

   return -1;
}

/************************************************************************
*									*
*   auth_check_local							*
*									*
*									*
************************************************************************/
int
auth_check_local
   AL((namelen, name, datalen, data))
   DB u16_t namelen
   DD char *name
   DD u16_t datalen
   DD char *data
   DE
{
   register int i;

   DEBUG1(D_AUTH, "auth_check_local: %s\n",
				local_str(name, namelen, data, datalen));

   for (i=0; i<NUM_AUTH; i++)
      if (auths[i].namelen == namelen && bcmp(auths[i].name,name,namelen)==0)
         return (*auths[i].check)(data, datalen);

   return 0;
}

/************************************************************************
*									*
*   auth_remove_local							*
*									*
*									*
************************************************************************/
int
auth_remove_local
   AL((namelen, name, datalen, data))
   DB u16_t namelen
   DD char *name
   DD u16_t datalen
   DD char *data
   DE
{
   register int i;

   DEBUG1(D_AUTH, "auth_remove_local: %s\n",
				local_str(name, namelen, data, datalen));

   for (i=0; i<NUM_AUTH; i++)
      if (auths[i].namelen == namelen && bcmp(auths[i].name,name,namelen)==0)
         return (*auths[i].remove)(data, datalen);

   return -1;
}

/************************************************************************
*									*
*   auth_reset_local							*
*									*
*									*
************************************************************************/
int
auth_reset_local
   AL((namelen, name))
   DB u16_t namelen
   DD char *name
   DE
{
   register int i;

   DEBUG0(D_AUTH, "auth_reset_local: \n");
   for (i=0; i<NUM_AUTH; i++)
      if (auths[i].namelen == namelen && bcmp(auths[i].name,name,namelen)==0)
         return (*auths[i].reset)();

   return -1;
}

/*
**   authfile -- get file name of authority file
*/
static char *
authfile
   VOID
{
   char *file;
   static char buf[MAXPATHLEN];
						/* find authority file */
   if (opt.auth && strcmp(opt.auth, "."))		/* dot uses default */
      file = opt.auth;
   else if (file = (char *)getenv("XAUTHORITY"))
      ;
   else if (file = (char *)getenv("HOME")) {
      sprintf(buf, "%s/.Xauthority", file);
      file = buf;
   }
   if (file != 0)
      if (access(file, R_OK) < 0) {
         if (errno != ENOENT)
            pwarn(file);
         file = (char *)0;
      }

   return file;
}

/*
**   read_authority -- read authority file, replacing current auth list
**
**	by convention, the authority file is always big-endian.
*/
static int
read_authority
   AL((file))
   DB char *file
   DE
{
   FILE *fp;
   int rv;
   u16_t len;
   char buf[64];
   auth_t *ap, *tp;

   if ((fp = fopen(file, "r")) == 0)
      if (opt.auth)
         return err(-1, "%s: can't open\n", file);
      else
         return -1;	/* no error message - not an error */

   if (authlist)
      for (ap=authlist; ap; ap=tp->next) {
         tp = ap;
         nuke_auth(ap);
      }

   authlist = authlast = (auth_t *)0;

   for (;;) {
      if (MALLOC(ap, auth_t *, sizeof(auth_t))) {
         fclose(fp);
         return -1;
      }
      ap->host.address = (char *)0;
      ap->name = (char *)0;
      ap->data = (char *)0;

      if (fread(&ap->host.family, sizeof(u16_t), 1, fp) != 1)	/* family */
         break;
      if (endian == 'l')
         SWAP2(ap->host.family);

      if (rallocstr(fp, &ap->host.length, &ap->host.address))	/* address */
         break;

      if (fread(&len, sizeof(u16_t), 1, fp) != 1)		/* display */
         break;
      if (endian == 'l')
         SWAP2(len);
      if (len >= 64) {
         warn("%s: insane Xauthority file! [%d]\n", file, len);
         break;
      }
      if (fread(buf, len, 1, fp) != 1)
         break;
      buf[len] = '\0';
      ap->display = atoi(buf);

      if (rallocstr(fp, &ap->namelen, &ap->name))		/* name */
         break;

      if (rallocstr(fp, &ap->datalen, &ap->data))		/* data */
         break;

      ap->next = (auth_t *)0;

      add_auth(ap);
   }
   if ((rv = ferror(fp) ? -1 : 0))
      nuke_auth(ap);

   fclose(fp);

   return rv;
}

/*
**   add_auth -- append an auth record to the list, or divert it to
**		the local list if it's for our address/display
*/
static void
add_auth
   AL((ap))
   DB auth_t *ap
   DE
{
   register u16_t family;
   register u16_t len;
   register char *addr;

   DEBUG1(D_AUTH, "add_auth: %s\n", rec_str(ap));

   if (ap->host.family == FamilyLocal) {
      family = FamilyLocal;
      len = hostnamelen;
      addr = hostname;
   }
   else {
      family = me.family;
      len = me.length;
      addr = me.address;
   }
   if (		ap->host.family == family &&
		ap->host.length == len &&
		ap->display == opt.dpy &&
		bcmp(ap->host.address, addr, len) == 0) {
      if (opt.auth)	/* only if -auth was specified */
         auth_add_local(ap->namelen, ap->name, ap->datalen, ap->data);
      ap->name = (char *)0;
      ap->data = (char *)0;
      nuke_auth(ap);
   }
   else if (authlast) {
      authlast->next = ap;
      authlast = ap;
   }
   else
      authlist = authlast = ap;
}

/*
**   nuke_auth -- free space used by an auth record
*/
static void
nuke_auth
   AL((ap))
   DB auth_t *ap
   DE
{
   if (ap->host.address) free(ap->host.address);
   if (ap->name) free(ap->name);
   if (ap->data) free(ap->data);
   free(ap);
}

/*
**   rallocstr -- read a string into an allocated buffer
*/
static int
rallocstr
   AL((fp, lenp, cpp))
   DB FILE *fp
   DD u16_t *lenp
   DD char **cpp
   DE
{
   if (fread(lenp, sizeof(u16_t), 1, fp) == 1) {
      if (endian == 'l')
         SWAP2(*lenp);
      if (*lenp) {
         if (MALLOC(*cpp, char *, *lenp))
            return -1;

         if (fread(*cpp, *lenp, 1, fp) == 1)
            return 0;
      }
   }
   return ferror(fp) ? -1 : 0;
}

static void
print_list
   VOID
{
   register auth_t *ap;

   warn("Authority List:\n");
   for (ap=authlist; ap; ap=ap->next)
      warn("   %s\n", rec_str(ap));
}

static void
print_rec
   AL((ap))
   DB auth_t *ap
   DE
{
   register u16_t i;
   char *fmt;
   char buf[128];

   warn("   %s", dprx_addrfam_str(ap->host.family));
   warn(" addr(");
   fmt = ap->host.family == FamilyLocal ? "%c" : "%02x";
   for (i=0; i<ap->host.length; i++)
      warn(fmt, ((unsigned char *)ap->host.address)[i]);
   warn(")");
   warn(" dpy(%d)", ap->display);
   strncpy(buf, ap->name, ap->namelen);
   buf[ap->namelen] = '\0';
   warn(" name(%s)", buf);
   warn(" data(");
   for (i=0; i<ap->datalen; i++)
      warn("%02x", ((unsigned char *)ap->data)[i]);
   warn(")\n");
}

static char *
rec_str
   AL((ap))
   DB auth_t *ap
   DE
{
   register int i, len, incr;
   register char *fmt;
   char addr[128];
   char name[64];
   char data[128];
   static char buf[512];

   len = MIN(127, ap->host.length);

   if (ap->host.family == FamilyLocal) {
      bcopy(ap->host.address, addr, len);
      addr[len] = '\0';
   }
   else
      data_str(ap->host.address, len, &addr[0]);

   len = MIN(63, (int)ap->namelen);
   strncpy(name, ap->name, len);
   name[len] = '\0';

   len = MIN(127, (int)ap->datalen);
   data_str(ap->data, len, &data[0]);

   sprintf(	buf,
		"fam(%s) addr(%s) dpy(%d) name(%s) data(%s)",
		dprx_addrfam_str(ap->host.family),
		addr,
		ap->display,
		name,
		data);

   return buf;
}

static char *
local_str
   AL((name, namelen, data, datalen))
   DB char *name
   DD int namelen
   DD char *data
   DD int datalen
   DE
{
   static char buf[512];

   if (namelen + datalen > 511 - 15) {
      namelen = MIN(249, namelen);
      datalen = MIN(249, datalen);
   }
   bcopy("name(", buf, 5);
   bcopy(name, &buf[5], namelen);
   bcopy(") data(", &buf[5+namelen], 7);
   data_str(data, datalen, &buf[12+namelen]);
   strcat(buf, ")\n");

   return buf;
}

/*
**	data_str
**
**	turn an arbitrary-length data block into a hex ascii string
**
**	truncates it arbitrarily
*/
static void
data_str
   AL((data, len, bp))
   DB char *data	/* binary data */
   DD int len		/* length of it */
   DD char *bp		/* return string buffer */
   DE
{
   register int i;

   for (i=0; i<len; i++)
      sprintf(&bp[i*2], "%02x", ((unsigned char *)data)[i]);
   bp[i*2] = '\0';
}
