/*
 * Copyright 1991-1997, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*	xmtg.c								*
*									*
************************************************************************/
#ifdef bozo
#include <libgen.h>
#endif

#include <stdio.h>
#include <X11/Xlib.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include "ui.h"
#include "ig.h"
#include "callbacks.h"
#include "fallbacks.h"
#include "node.h"
#include "layout.h"
#include "pathfind.h"
#include "site.h"

#include <xmclib.h>

#ifdef SYSV
#define bcopy(a,b,c)    memcpy(b,a,c)
#endif

#define BAD	-1
#define FLOOR	0
#define SEAT	1
#define HAND	2
#define VIEW	3
#define NOT	4

/*
**	actions
*/
void open_act();
void save_act();
void accept_act();
void cancel_act();
void all_act();
void add_act();
void drop_act();
void floor_act();
void seat_act();
void view_act();
void grab_act();
void quit_act();

/*
**	local routines
*/
static void load_file();
static void save_file();
static void show_add();
static void show_icon_not();
static int im2s();

static Mux *mux;
static node_t *chair;	/* meeting chair - always has the floor */
static char *chairtag;	/* name of chair */
static int nfloors;	/* if this goes to zero, we're hung! */

static char *usage = "usage: xmtg [-option ...] [layoutfile]\n\
\n\
Options include all standard X Toolkit options, plus:\n\
	-chair tag        Server 'tag' always has the floor\n\
	-help             Print usage information\n\
	-layoutpath path  Search for layout file in path\n\
	-mdisplay disp    Display name of X multiplexor (if not -display)\n\
	layoutfile        Use icon layout in file\n\
";

/************************************************************************
*									*
*   Options & Resources							*
*									*
************************************************************************/
static XrmOptionDescRec options[] = {
   { "-mdisplay",  ".mDisplay",  XrmoptionSepArg,  0},
   { "-layoutpath",  ".layoutPath",  XrmoptionSepArg,  0},
   { "-chair",  ".chair",  XrmoptionSepArg,  0},
};

typedef struct _AppResources {
   String	mDisplay;
   String	layoutPath;
   String	chair;
}AppResources;

static AppResources resvals;

static XtResource resources[] = {
   {"mDisplay", "MDisplay", XtRString, sizeof(String),
	XtOffset(AppResources *, mDisplay), XtRString, (XtPointer)0},
   {"layoutPath", "LayoutPath", XtRString, sizeof(String),
	XtOffset(AppResources *, layoutPath), XtRString, (XtPointer)0},
   {"chair", "Chair", XtRString, sizeof(String),
	XtOffset(AppResources *, chair), XtRString, (XtPointer)0},
};

/************************************************************************
*									*
*   Action table							*
*									*
************************************************************************/
static XtActionsRec actions_table[] = {
   {"do-open", open_act},
   {"do-save", save_act},
   {"do-accept", accept_act},
   {"do-cancel", cancel_act},
   {"do-all", all_act},
   {"do-add", add_act},
   {"do-drop", drop_act},
   {"do-floor", floor_act},
   {"do-seat", seat_act},
   {"do-view", view_act},
   {"do-grab", grab_act},
   {"do-quit", quit_act}
};

Widget top, grid;

static int mode = FLOOR;

/*
**	default icon images
*/
#include "bitmaps/floor"
#include "bitmaps/floormask"
#include "bitmaps/seat"
#include "bitmaps/seatmask"
#include "bitmaps/hand"
#include "bitmaps/handmask"
#include "bitmaps/view"
#include "bitmaps/viewmask"
#include "bitmaps/not"
#include "bitmaps/notmask"

/*
**	state->input-mode
*/
int s2im[] = {Floor, Seat, 0, View, 0};

IconGrid gp;

/************************************************************************
*									*
*   Callbacks								*
*									*
************************************************************************/

void
open_cb(w, client_data, call_data)
   Widget w;
   XtPointer client_data;
   XtPointer call_data;
{
   String filename;
   Widget dialog = (Widget)client_data;

   filename = ui_get_dialog_string(dialog);

   load_file(filename);

   ui_free_string(dialog);
}

void
save_cb(w, client_data, call_data)
   Widget w;
   XtPointer client_data;
   XtPointer call_data;
{
   String filename;
   Widget dialog = (Widget)client_data;

   filename = ui_get_dialog_string(dialog);

   save_file(filename);

   ui_free_string(dialog);
}

void
quit_cb(w, client_data, call_data)
   Widget w;
   XtPointer client_data;
   XtPointer call_data;
{
   exit(0);
}

void
all_cb(w, client_data, call_data)
   Widget w;
   XtPointer client_data;
   XtPointer call_data;
{
   ig_select(gp, 0);
}

void
teleptr_cb(w, client_data, call_data)
   Widget w;
   XtPointer client_data;
   XtPointer call_data;
{
   int state;
   TptrId tpID;

   if (mux == 0) return;	/* sanity */

   state = ui_get_toggle_state(w);
   tpID = DefaultTelepointer(mux);

   if (state)
      XmcShowTptr(mux, tpID);
   else
      XmcHideTptr(mux, tpID);
   XmcFlush(mux);
}

void
add_cb(w, client_data, call_data)
   Widget w;
   XtPointer client_data;
   XtPointer call_data;
{
   int id, mode;
   char *display;
   TptrId tpID;
   add_d_t *adp = (add_d_t *)client_data;

   if (mux == 0) return;	/* sanity */

   ui_get_add_info(adp, &display, &mode);

   tpID = mode == FLOOR ? DefaultTelepointer(mux) : 0;
   id = XmcAddDisplay(mux, display, 0, 0, 0, s2im[mode], display, tpID);
   XmcFlush(mux);
   (void) node_add(id, display, mode, 0);

/*  seems like this is getting freed before we get here...hmmm (TODO)
   ui_free_add_info(adp);
*/
}

void
hand_up(np)
   node_t *np;
{
   if (IconState(np->ip) == HAND)
      ig_change_state(gp, np->ip, SEAT);
   else
      ig_change_state(gp, np->ip, HAND);
}

static void
dropit(gp, ip)
   IconGrid gp;
   IGIcon *ip;
{
   node_t *np;

   if (ip) {
      if (np = (node_t *)IconData(ip))
         if (np == chair)
            ui_error("cannot drop chair");
         else {
            if (np->mode == FLOOR)
               if (nfloors == 1) {
                  ui_error("Last floor!  You don't want to do that");
                  return;
               }
               else
                  nfloors--;
            XmcDropDisplay(mux, np->id);
         }
   }
   else
      XmcFlush(mux);
}

void
drop_cb(w, client_data, call_data)
   Widget w;
   XtPointer client_data;
   XtPointer call_data;
{
   if (mux == 0) return;	/* sanity */

   ig_call_callback(gp, dropit);
}

static void
floorit(gp, ip)
   IconGrid gp;
   IGIcon *ip;
{
   node_t *np;

   if (ip) {
      if (np = (node_t *)IconData(ip))
         if (np == chair)
            ui_error("cannot change input mode of chair");
         else
            XmcChangeInputMode(mux, np->id, s2im[FLOOR]);
   }
   else
      XmcFlush(mux);
}

void
floor_cb(w, client_data, call_data)
   Widget w;
   XtPointer client_data;
   XtPointer call_data;
{
   if (mux == 0) return;	/* sanity */

   ig_call_callback(gp, floorit);
}

static void
seatit(gp, ip)
   IconGrid gp;
   IGIcon *ip;
{
   node_t *np;

   if (ip) {
      if (np = (node_t *)IconData(ip))
         if (np == chair)
            ui_error("cannot change input mode of chair");
         else
            if (np->mode == FLOOR && nfloors == 1)
               ui_error("Last floor!  You don't want to do that");
            else
               XmcChangeInputMode(mux, np->id, s2im[SEAT]);
   }
   else
      XmcFlush(mux);
}

void
seat_cb(w, client_data, call_data)
   Widget w;
   XtPointer client_data;
   XtPointer call_data;
{
   if (mux == 0) return;	/* sanity */

   ig_call_callback(gp, seatit);
}

static void
viewit(gp, ip)
   IconGrid gp;
   IGIcon *ip;
{
   node_t *np;

   if (ip) {
      if (np = (node_t *)IconData(ip))
         if (chair && np == chair)
            ui_error("cannot change input mode of chair");
         else
            if (np->mode == FLOOR && nfloors == 1)
               ui_error("Last floor!  You don't want to do that");
            else
               XmcChangeInputMode(mux, np->id, s2im[VIEW]);
   }
   else
      XmcFlush(mux);
}

void
view_cb(w, client_data, call_data)
   Widget w;
   XtPointer client_data;
   XtPointer call_data;
{
   if (mux == 0) return;	/* sanity */

   ig_call_callback(gp, viewit);
}

static void
grabit(gp, ip)
   IconGrid gp;
   IGIcon *ip;
{
   node_t *np;

   if (ip) {
      if (np = (node_t *)IconData(ip))
        /* XmcGrab(mux, np->id);*/;
/*
      ig_change_state(gp, ip, VIEW);
*/
   }
   else
      XmcFlush(mux);
}

void
grab_cb(w, client_data, call_data)
   Widget w;
   XtPointer client_data;
   XtPointer call_data;
{
   if (mux == 0) return;	/* sanity */

   ui_error("grabs are disabled, for now...");
/*
   ig_call_callback(gp, grabit);
*/
}

/************************************************************************
*									*
*   Actions								*
*									*
************************************************************************/

void
open_act(w, event, params, n)
   Widget w;
   XEvent *event;
   String *params;
   Cardinal *n;
{
   ui_popup_open();
}

void
save_act(w, event, params, n)
   Widget w;
   XEvent *event;
   String *params;
   Cardinal *n;
{
   ui_popup_save();
}

void
accept_act(w, event, params, n)
   Widget w;
   XEvent *event;
   String *params;
   Cardinal *n;
{
   if (*n)
      ui_dialog_accept(*params);
}

void
cancel_act(w, event, params, n)
   Widget w;
   XEvent *event;
   String *params;
   Cardinal *n;
{
   if (*n)
      ui_dialog_cancel(*params);
}

void
all_act(w, event, params, n)
   Widget w;
   XEvent *event;
   String *params;
   Cardinal *n;
{
   ig_select(gp, 0);
}

void
add_act(w, event, params, n)
   Widget w;
   XEvent *event;
   String *params;
   Cardinal *n;
{
   ui_popup_add();
}

void
drop_act(w, event, params, n)
   Widget w;
   XEvent *event;
   String *params;
   Cardinal *n;
{
   ig_call_callback(gp, dropit);
}

void
floor_act(w, event, params, n)
   Widget w;
   XEvent *event;
   String *params;
   Cardinal *n;
{
   ig_call_callback(gp, floorit);
}

void
seat_act(w, event, params, n)
   Widget w;
   XEvent *event;
   String *params;
   Cardinal *n;
{
   ig_call_callback(gp, seatit);
}

void
view_act(w, event, params, n)
   Widget w;
   XEvent *event;
   String *params;
   Cardinal *n;
{
   ig_call_callback(gp, viewit);
}

void
grab_act(w, event, params, n)
   Widget w;
   XEvent *event;
   String *params;
   Cardinal *n;
{
   ig_call_callback(gp, grabit);
}

void
quit_act(w, event, params, n)
   Widget w;
   XEvent *event;
   String *params;
   Cardinal *n;
{
   exit(0);
}

/************************************************************************
*									*
*   Application Callbacks						*
*									*
************************************************************************/
static void
xmc_cb(client_data, source, id)
   XtPointer client_data;
   int *source;
   XtInputId *id;
{
   register int i, nevents;
   Mux *mux = (Mux *)client_data;
   XmcEvent event;
   XmcDispInfo *dip;
   node_t *np;

   /*
   ** drain incoming queue, but don't block
   */
   nevents = XmcEventsQueued(mux, XmcQueuedAfterReading);

   for (i=0; i<nevents; i++) {
      if (XmcGetEvent(mux, &event)) {
         ui_error("XmcGetEvent: error");
         return;
      }
      switch (event.type) {
         case SeatButtonPressEvent:
            if (np = node_byid(event.button.id))
               hand_up(np);
            else
               ui_error("SeatButtonPressEvent for unknown display 0x%x",
   							event.button.id);
            break;
         case DisplayInEvent:
            if ((np = node_byid(event.display.id)) == 0) {
               if ((dip = XmcQueryDisplay(mux, event.display.id)) == 0) {
                  ui_error("DisplayEvent for unknown display 0x%x",
   							event.display.id);
                  break;	/* exit switch */
               }
               np = node_add(event.display.id, dip->tag, im2s(dip->mode), 0);
               if (chairtag && strcmp(np->display, chairtag) == 0)
                  chair = np;
            }
            show_add(np);
            break;
         case DisplayOutEvent:
            if (np = node_byid(event.display.id)) {
               if (np->ip) {
                  ig_change_state(gp, np->ip, NOT);
                  ig_set_data(np->ip, 0);	/* just in case */
               }
               node_remove(np);
            }
            break;
         case DisplayRefusedEvent:
            if (np = node_byid(event.display.id)) {
               ui_error("%s: connection refused", np->display);
               if (np->ip)
                  ig_set_data(np->ip, 0);	/* just in case */
               node_remove(np);
            }
            break;
         case ModeFloorEvent:
            if (np = node_byid(event.display.id)) {
               if (np->mode != FLOOR) {
                  nfloors++;
                  ig_change_state(gp, np->ip, FLOOR);
               }
            }
            else
               ui_error("InputModeFloorEvent for unknown display 0x%x",
   							event.display.id);
            break;
         case ModeSeatEvent:
            if (np = node_byid(event.display.id)) {
               if (np->mode == FLOOR) {
                  nfloors--;
                  ig_change_state(gp, np->ip, SEAT);
               }
            }
            else
               ui_error("InputModeSeatEvent for unknown display 0x%x",
   							event.display.id);
            break;
         case ModeViewEvent:
            if (np = node_byid(event.display.id)) {
               if (np->mode == FLOOR) {
                  nfloors--;
                  ig_change_state(gp, np->ip, VIEW);
               }
            }
            else
               ui_error("InputModeViewEvent for unknown display 0x%x",
   							event.display.id);
            break;
         default:
            ui_error("Unknown event type %d", event.type);
            break;
      }
   }
}


/************************************************************************
*									*
*   Other routines							*
*									*
************************************************************************/

static void
load_file(filename)
   char *filename;
{
   ui_error("load_file: \"%s\" (not implemented yet)", filename);
}

static void
save_file(filename)
   char *filename;
{
   ui_error("save_file: \"%s\" (not implemented yet)", filename);
}

static void
show_add(np)
   node_t *np;
{
   if (np->ip = ig_find_icon(gp, np->display)) {

      if (IconData(np->ip))	/* this tag is taken - that's an error */
         np->ip = 0;
      else {
         (void)ig_set_data(np->ip, (void *)np);
         ig_change_state(gp, np->ip, np->mode);
      }
   }
   else
      np->ip = ig_add_icon(gp, np->display, np->mode, 0, -1, -1, np);

   if (np->ip == 0) {
      ui_error("show_add: cannot create icon - duplicate?");
      /* TODO - do something here */
   }
   else if (np->mode == FLOOR)
      nfloors++;
}

/************************************************************************
************************************************************************/
int
main(argc, argv)
   int argc;
   char **argv;
{
   register int i, j;
   register Widget w;
   register node_t *np;
   XtAppContext app;
   char *mDisplay;
   char *layoutFile;
   char *layoutPath;
   String *fb;
   Arg arg[16];
   int n;
   DispId *dlp;
   XmcDispInfo *dip;
   XtPopdownIDRec popdown;
   IGView vp;

   top= XtAppInitialize(	&app,
				"XMtg",
				options, XtNumber(options),
				&argc, argv,
				fallbacks,
				(ArgList)NULL, 0);
   if (--argc) {
      layoutFile = *++argv;

      if (argc > 1) {
         fprintf(stderr, usage);
         exit(-1);
      }
   }
   else
      layoutFile = 0;

   XtAppAddActions(app, actions_table, XtNumber(actions_table));
   XtGetApplicationResources(top, (XtPointer)&resvals,
				resources, XtNumber(resources), NULL, 0);
   XtOverrideTranslations(top,
		XtParseTranslationTable("<Message>WM_PROTOCOLS: do-quit()"));

   ui_create(top, FLOOR, SEAT, VIEW, &grid);

   gp = ig_init(app, grid, 5);
   vp = ig_register_view(gp, "fixed");
   (void)ig_register_image_data(gp, vp,
		FLOOR, floor_bits, floormask_bits, floor_width, floor_height);
   (void)ig_register_image_data(gp, vp,
		SEAT,  seat_bits,  seatmask_bits,  seat_width,  seat_height);
   (void)ig_register_image_data(gp, vp,
		HAND,  hand_bits,  handmask_bits,  hand_width,  hand_height);
   (void)ig_register_image_data(gp, vp,
		VIEW,  view_bits,  viewmask_bits,  view_width,  view_height);
   (void)ig_register_image_data(gp, vp,
		NOT,   not_bits,   notmask_bits,   not_width,   not_height);
   ig_set_view(gp, vp);

   /*
   **	set multiplexor display
   */
   if (mDisplay = (char *)getenv("MDISPLAY"))
      mDisplay = strcpy((char *)malloc(strlen(mDisplay)+1), mDisplay);
   else if (resvals.mDisplay)
      mDisplay = resvals.mDisplay;
   else
      mDisplay = DisplayString(XtDisplay(top));

   /*
   **	process layout file, if any
   */
   if (layoutPath = (char *)getenv("XMTGLAYOUTPATH"))
      layoutPath = strcpy((char *)malloc(strlen(layoutPath)+1), layoutPath);
   else
      layoutPath = resvals.layoutPath;
#ifdef LAYOUTPATH
   if (layoutPath == 0)
      layoutPath = LAYOUTPATH;
#endif

   if (layoutFile && (layoutFile = pathfind(layoutPath, layoutFile, "r")))
      layout_file(layoutFile, show_icon_not);

   XtRealizeWidget(top);
   XFlush(XtDisplay(top));
   ig_realize(gp);

   /*
   **	is there a chair?
   */
   if (chairtag = (char *)getenv("XMTGCHAIR"))
      chairtag = strcpy((char *)malloc(strlen(chairtag)+1), chairtag);
   else
      chairtag = resvals.chair;

   /*
   **	connect to multiplexor
   */
   if (mux = XmcOpen(mDisplay, 0)) {
      if (strcmp(mDisplay, DisplayString(XtDisplay(top))) == 0) {
         XmcSetXEventMask(mux, XtWindow(top), SeatButtonPressMask);
         XmcSetEventMask(mux, DisplayMask | InputModeMask);
      }
      else
         XmcSetEventMask(mux, DisplayMask | InputModeMask);

      if (dlp = XmcListDisplaysWithInfo(mux, &n, &dip)) {
         for (i=0; i<n; i++) {
            np = node_add(dlp[i], dip->tag, im2s(dip->mode), 0);
            if (chairtag && strcmp(np->display, chairtag) == 0)
               chair = np;
            show_add(np, 0);
         }
      }
      if (chairtag && chair == 0)
         ui_error("no chair named '%s' found", chairtag);

      (void) XtAppAddInput(	app,
				XmcConnectionNum(mux),
				(XtPointer)XtInputReadMask,
				(XtInputCallbackProc)xmc_cb,
				(XtPointer)mux);
   }
   else
      ui_error("cannot connect to multiplexor at %s", mDisplay);

   XtAppMainLoop(app);
}

static void
show_icon_not(name, row, col)
   char *name;
   int row, col;
{
   if (name && *name != '+')
      ig_add_icon(gp, name, NOT, 0, row, col, 0);
}

static int
im2s(mode)
   int mode;
{
   switch (mode) {
      case View:	return VIEW;
      case Seat:	return SEAT;
      case Floor:	return FLOOR;
      default:		return BAD;
   }
}
