/*
Copyright 1991-1996 by the University of Edinburgh, Department of
Computer Science
Copyright 1999 by the University of Edinburgh, Division of Informatics

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation, and that the name of the University of Edinburgh not be used
in advertising or publicity pertaining to distribution of the software
without specific, written prior permission.  The University of Edinburgh
makes no representations about the suitability of this software for any
purpose.  It is provided "as is" without express or implied warranty.

	Author:	George Ross
		Division of Informatics
		University of Edinburgh
		gdmr@dcs.ed.ac.uk
*/

/* Parse a mailbox.  Fill the supplied buffer with From/Subject lines */
/* $Id: box.c,v 1.6 1999/11/01 14:53:19 gdmr Exp gdmr $ */

#define MHBOX_RETRIES 3
#define MHBOX_SLEEP 4

#include "box.h"

#include <stdio.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <X11/Xos.h>

/* MIME processing */

#define RFC1522_ORDINARY	0
#define RFC1522_CHARSET		1
#define RFC1522_ENCODING	2
#define RFC1522_ENCODED		3

static char charset[32];

void setCharset(char *registry, char *encoding)
{
	if (!registry) return;
	if (strcasecmp(registry, "iso8859")) return;

	if (!encoding) return;
	if (strlen(encoding) > 1) return;
	if ((*encoding < '0') || (*encoding > '9')) return;

	(void) strcpy(charset, "iso-8859-");
	(void) strcat(charset, encoding);
}

static void doMime(char *thing)
{	char buffer[120];
	char csb[120];
	char *from, *to, *c;
	int encoded;
	int state = RFC1522_ORDINARY;

	if (!(*charset)) return;

	from = thing; to = buffer;

	while (*from) {
		switch (state) {
	case RFC1522_ORDINARY:
	default:
			if (*from == '=' && *(from+1) == '?') {
				state = RFC1522_CHARSET;
				c = csb;
				from += 2;
			} else {
				*to++ = *from++;
			}
			break;

	case RFC1522_CHARSET:
			while (*from && *from != '?')
				*c++ = *from++;
			from++;
			*c = '\0';
			if (strcasecmp(csb, charset)) return;
			state = RFC1522_ENCODING;
			break;

	case RFC1522_ENCODING:
			if (*from != 'q' && *from != 'Q') return;
			from++;
			if (*from != '?') return;
			from++;
			state = RFC1522_ENCODED;
			break;

	case RFC1522_ENCODED:
			if (*from == '?') {
				from++;
				if (!*from) return;
				if (*from == '=') {
					from++;
					state = RFC1522_ORDINARY;
				} else return;
			}
			if (*from == '=') {
				from++;
				if (!*from) return;
				if (('0' <= *from) && (*from <= '9')) {
					encoded = *from - '0';
				} else if (('a' <= *from) && (*from <= 'f')) {
					encoded = *from - 'a' + 10;
				} else if (('A' <= *from) && (*from <= 'F')) {
					encoded = *from - 'A' + 10;
				} else return;
				from++;
				if (!*from) return;
				encoded <<= 4;
				if (('0' <= *from) && (*from <= '9')) {
					encoded += *from - '0';
				} else if (('a' <= *from) && (*from <= 'f')) {
					encoded += *from - 'a' + 10;
				} else if (('A' <= *from) && (*from <= 'F')) {
					encoded += *from - 'A' + 10;
				} else return;
				from++;
				*to++ = (char) encoded;
			} else {
				*to++ = *from++;
			}
		}
	}

	*to = '\0';
	(void) strcpy(thing, buffer);
}

/* Some utility stuff */

static char *mungeSender(char *from, int flags)
{	char *a, *b;
	int n;

	if (flags & BOX_FULLNAMES) return from;
	if (index(from, '"')) return from;

	/* Prefer (..) */
	if (a = index(from, '(')) a++;
	else a = from;
	if (b = index(a, ')')) *b = '\0';
	/* Dump <..> */
	if (b = index(a, '<')) *b = '\0';
	/* Strip trailing blanks */
	n = strlen(a);
	if (n == 0) return from;  /* Nothing left */
	b = a + n;
	for (;;) {
		b--;
		if (isgraph(*b)) break;
		if (a == b) return from;  /* Nothing left */
		*b = '\0';
	}
	return a;
}

static void LowerFront(char *line)
{	if (!line) return;
	while (isalpha(*line)) {
		*line = tolower(*line);
		line++;
	}
}

/* MH format (mailbox is a directory) */

static void MHparseOne(char *file, char *buffer, int flags)
{	FILE *mf;
	char line[160];
	char from[80];
	char subj[80];
	char *n;
	int got;
	int retryCount = MHBOX_RETRIES;

#define TERMINATE(x) x[79] = '\0'; if (n = index(x, '\n')) *n = '\0';

	if (flags & BOX_DEBUG) (void) printf("MHparseOne(%s)\n", file);

	for (;;) {
		if ((mf = fopen(file, "r")) == NULL) {
			(void) sprintf(buffer, "Failed to open %s\n", file);
			return;
		}

		got = 0;
		*from = '\0';  *subj = '\0';
		for (;;) {
			if (fgets(line, 160, mf) == NULL) break;
	
			LowerFront(line);
			if (sscanf(line, "from: %70c", from) > 0) {
				TERMINATE(from)
				got |= 1;
			}
			else if (sscanf(line, "subject: %70c", subj) > 0) {
				TERMINATE(subj)
				got |= 2;
			}
			if (got == 3) break;
		}
		(void) fclose(mf);
		if (*from) break;
		if (retryCount-- <= 0) break;
		sleep(MHBOX_SLEEP);
	}
	
#undef TERMINATE

	if (!*from) (void) strcpy(from, "*Unknown sender*");

	if (flags & BOX_DOMIME) {
		doMime(from);
		doMime(subj);
	}

	(void) sprintf(buffer, "%s << %s >>\n",
		mungeSender(from, flags), subj);
}

static int MHparseMailbox(char *mailbox, char *buffer, int remaining, int flags)
{	struct dirent **namelist;
	int n, i, tally;
	char oneFrom[160];
	char *pos;
	char filename[PATH_MAX];

	if (flags & BOX_DEBUG) (void) printf("MHparseMailbox(%s)\n", mailbox);

	n = scandir(mailbox, &namelist, 0, alphasort);
	if (n < 0) {
		perror("scandir mailbox");
		return -1;
	}

	pos = buffer;  tally = n;
	for (i = 0; i < n; i++) {
		if (*(namelist[i]->d_name) == '.') {
			/* Ignore dot-files */
			tally--;
		} else {
			strcpy(filename, mailbox);
			strcat(filename, "/");
			strcat(filename, namelist[i]->d_name);
			MHparseOne(filename, oneFrom, flags);
			strcpy(pos, oneFrom);
			pos += strlen(oneFrom);
		}
		free((char *)namelist[i]);
	}
	free((char *)namelist);
	return tally;
}

/* Non-MH (mailbox is a plain file) */

static void handlePrevious(char *buffer, int *remaining, int flags,
		char *from, char *stat, char *subj, int *tally)
{	char line[240];
	if (*from == '\0') return;
	if (flags & BOX_DEBUG)
		(void) printf("%d: %s '%s' <<%s>>\n", *tally, from, stat, subj);
	if ((flags & BOX_NEWONLY) && *stat) {
		/* Not a new message, so skip it */
		*from = '\0';
		*stat = '\0';
		*subj = '\0';
		return;
	}
	if (flags & BOX_DOMIME) {
		doMime(from);
		doMime(subj);
	}
	if (*stat == '\0')
		(void) sprintf(line, "%s << %s >>\n",
				mungeSender(from, flags), subj);
	else
		(void) sprintf(line, "%s (%s) << %s >>\n",
				mungeSender(from, flags), stat, subj);
	if ((*remaining -= strlen(line)) > 0)
		(void) strcat(buffer, line);
	*from = '\0';
	*stat = '\0';
	*subj = '\0';
	(*tally)++;
}

int parseMailbox(char *mailbox, char *buffer, int remaining, int flags)
{	FILE *mf;
	char line[160];
	char fromfrom[80];
	char from[80];
	char subj[80];
	char mstat[80];
	char *c, *n;
	int lastBlank;
	int headers;
	int tally = 0;
	int mmdf = 0;
	struct stat ss;

#define TERMINATE(x) x[79] = '\0'; if (n = index(x, '\n')) *n = '\0';

	if (flags & BOX_DEBUG)
		(void) printf("Mailbox: %s, flags: %d\n", mailbox, flags);

	buffer[0] = '\0';

	if (stat(mailbox, &ss) < 0) {
		perror("Couldn't stat() mailbox");
		return -1;
	}
	if (S_ISDIR(ss.st_mode)) {
		return MHparseMailbox(mailbox, buffer, remaining, flags);
	}

	if ((mf = fopen(mailbox, "r")) == NULL) {
		sprintf(buffer, "Failed to open %s", mailbox);
		return -1;
	}

	from[0] = '\0';
	subj[0] = '\0';
	mstat[0] = '\0';
	lastBlank = 1;
	headers = 1;

	for (;;) {
		if ((c = fgets(line, 160, mf)) == NULL) break;
		if (*c == 001) {
			/* Start of a new MMDF message */
			handlePrevious(buffer, &remaining, flags,
				from, mstat, subj, &tally);
			lastBlank = 0;
			headers = 1;
			mmdf = 1;
			while (*c == 001) c++;
			if (*c == '\n') continue;
		}
		LowerFront(line);
		if (flags & BOX_DEBUG)
			(void) printf("%d, %d, <%s>\n", lastBlank, headers, c);
		if (*c == '\n') {
			lastBlank = 1;
			if (*from) headers = 0;
		}
		else if (!mmdf && lastBlank
				&& (sscanf(c, "from %s", fromfrom) > 0) 
				&& *fromfrom != ':') {
			/* Start of a new sendmail message */
			handlePrevious(buffer, &remaining, flags,
				from, mstat, subj, &tally);
			(void) strcpy(from, fromfrom);
			lastBlank = 0;
			headers = 1;
		}
		else if (headers && (sscanf(c, "from: %70c", from) > 0)) {
			/* Proper 822 From: header */
			TERMINATE(from);
			lastBlank = 0;
		}
		else if (headers && (sscanf(c, "subject: %70c", subj) > 0)) {
			/* Message subject */
			TERMINATE(subj);
			lastBlank = 0;
		}
		else if (headers && (sscanf(c, "status: %s", mstat) > 0)) {
			/* Message mstatus */
			lastBlank = 0;
		}
		/* else something we're not interested in, so skip it */
		else lastBlank = 0;
	}
	handlePrevious(buffer, &remaining, flags, from, mstat, subj, &tally);
	(void) fclose(mf);
	return tally;
}

/*
 * $Log: box.c,v $
 * Revision 1.6  1999/11/01 14:53:19  gdmr
 * Ignore case in headers
 *
 * Revision 1.5  1999/08/18 13:35:04  gdmr
 * MHparseOne: need to initialise from and subj
 *
 * Revision 1.4  1999/08/03 09:42:27  gdmr
 * MHparseOne()
 *
 * Revision 1.3  1999/08/02 15:24:48  gdmr
 * MHparseMailbox()
 *
 * Revision 1.2  1999/08/02 13:07:51  gdmr
 * Copyright, ANSIfy
 *
 */
