/**
 * @filename	stats-ult.c
 *
 * @desc	Retrieve stats for the Ultrix & Irix platforms
 */

/* My includes */
#include "upclient.h"
#include "options.h"
#include "stats.h"

/* System includes */
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <nlist.h>
#include <syslog.h>
#include <string.h>
#include <sys/utsname.h>
#include <sys/cpudata.h>
#include <sys/fixpoint.h>

/**
 * @desc	Get statistics
 */
void getstats(unsigned long *puptime, double *pload, int *pidle, char *os, char *oslevel, char *cpu) {
  struct utsname uts;
  time_t now, boottime;
  int fd, up, id, avenrun[3];
  long cpudata_offset;
  struct nlist names[2];
  struct cpudata cpudata;

  /* Open file descriptor to /dev/kmem first */
  if((fd = open("/dev/kmem", O_RDONLY)) < 0)
    exit(-2);
  /* Make sure second nlist name pointer is a NULL so nlist(3) knows when to stop. */
  names[1].n_name = NULL;

  /* Get uptime */
  time(&now);
  names[0].n_name = "boottime";
  if(nlist("/vmunix", names) != 0)
    exit(-3);
  if(lseek(fd, names[0].n_value, SEEK_SET) == -1)
    exit(-4);
  if(read(fd, &boottime, sizeof(boottime)) < 0)
    exit(-5);
  *puptime = (now - boottime) / 60;

  if(cfg_sendload) {
    names[0].n_name = "avenrun";
    if(nlist("/vmunix", names) != 0)
      exit(-6);
    if(lseek(fd, names[0].n_value, SEEK_SET) == -1)
      exit(-7);
    if(read(fd, avenrun, sizeof(avenrun)) < 0)
      exit(-8);
    /* Use the 3rd element (15 minute load average) */
    *pload = FIX_TO_DBL(avenrun[2]);
  }

  if(cfg_sendidle) {
    names[0].n_name = "cpudata";
    if(nlist("/vmunix", names) != 0)
      exit(-10);
    if((fd = open("/dev/kmem", O_RDONLY)) < 0)
      exit(-11);
    if(lseek(fd, names[0].n_value, SEEK_SET) == -1)
      exit(-12);
    if(read(fd, &cpudata_offset, sizeof(cpudata_offset)) < 0)
      exit(-13);
    if(lseek(fd, cpudata_offset, SEEK_SET) == -1)
      exit(-13);
    if(read(fd, &cpudata, sizeof(cpudata)) < 0)
      exit(-13);
    id = cpudata.cpu_cptime[CP_IDLE];
    *pidle = (int)(100.0 * cpudata.cpu_cptime[CP_IDLE] / (cpudata.cpu_cptime[0] + cpudata.cpu_cptime[1] + cpudata.cpu_cptime[2] + cpudata.cpu_cptime[3]));
    /* Check for wrap-around */
    if(*pidle < 0 || *pidle > 100)
      *pidle = -1;
  }

  if(cfg_sendos || cfg_sendcpu) {
    uname(&uts);
  }

  if(cfg_sendos) {
    strncpy(os, uts.sysname, OS_SIZE - 1);
    /* Make "ULTRIX" look prettier */
    if(strcmp(os, "ULTRIX") == 0) {
      strncpy(os, "Ultrix", OS_SIZE - 1);
    }
    if(cfg_sendoslevel) {
      strncpy(oslevel, uts.release, OSLEVEL_SIZE - 1);
    }
  }

  if(cfg_sendcpu) {
    strncpy(cpu, uts.machine, CPU_SIZE - 1);
    /* Make "RISC" look prettier under Ultrix */
    if(strcmp(os, "Ultrix") == 0 && strcmp(cpu, "RISC") == 0) {
      strncpy(cpu, "MIPS", CPU_SIZE - 1);
    }
  }

  /* Close the kernel file descriptor */
  close(fd);
}
