/**
 * @filename	stats-lnx.c
 *
 * @desc	Retrieve stats for the Linux platform
 */

/* My includes */
#include "upclient.h"
#include "options.h"
#include "stats.h"

/* System includes */
#include <stdio.h>
#include <errno.h>
#include <limits.h>
#include <syslog.h>
#include <sys/utsname.h>

/**
 * @desc	Get statistics
 */
void getstats(unsigned long *puptime, double *pload, int *pidle, char *os, char *oslevel, char *cpu) {
  struct utsname uts;
  FILE *fp;
  double up=0, id=0;

  /* Get uptime */
  fp = fopen("/proc/uptime", "r");
  if(!fp) {
    syslog(LOG_ERR, "getstats(): unable to open /proc/uptime: %s", strerror(errno));
    exit(-2);
  }

  if(fscanf(fp, "%lf %lf", &up, &id) != 2) {
    syslog(LOG_ERR, "getstats(): bogus format in /proc/uptime");
    exit(-3);
  }

  fclose(fp);
  *puptime = (int)(up / 60.0);
  if(up == 0) {
    up = 1;
  }

#if defined(NR_LINUX_UPTIME_WRAPAROUNDS)
        /* Make up for those misserable uptime wraparounds */
  *puptime += ((ULONG_MAX / 100) / 60) * NR_LINUX_UPTIME_WRAPAROUNDS;
#endif

  /* Get load */
  if(cfg_sendload) {
    fp = fopen("/proc/loadavg", "r");
    if(fp == NULL) {
      syslog(LOG_ERR, "getstats(): unable to open /proc/loadavg: %s", strerror(errno));
      exit(-2);
    }
    if(fscanf(fp, "%*f %*f %lf %*s", pload) != 1) {
      syslog(LOG_ERR, "getstats(): bogus format in /proc/loadavg");
      exit(-3);
    }
    fclose(fp);
  }

  /* Get idle time */
  if(cfg_sendidle) {
    *pidle = (int)(100.0 * id / up);
  }

  /* Get os info */
  if(cfg_sendos) {
    uname(&uts);
  } else {
    /* Get CPU info */
    if(cfg_sendcpu) {
      uname(&uts);
    }
  }

  if(cfg_sendos) {
    strncpy(os, uts.sysname, OS_SIZE - 1);
    if(cfg_sendoslevel) {
      strncpy(oslevel, uts.release, OSLEVEL_SIZE - 1);
    }
  }

  if(cfg_sendcpu) {
    strncpy(cpu, uts.machine, CPU_SIZE - 1);
  }
}
