/**
 * @filename	options.c
 *
 * @desc	Options file parsing
 */

/* My includes */
#include "options.h"

/* System includes */
#include "stdio.h"

/**
 * Global variables, very ugly, watch through sunglasses!
 *
 * These are the configuration options available.
 */
char cfg_authkey[AUTHKEY_REQUIRED_LEN+1]	= "";
char cfg_upserver[UPSERVER_MAXLEN+1]		= "data.uptimes.net";
int  cfg_interval				= 120;
char cfg_pidfile[PIDFILE_MAXLEN+1]		= ".upclient.pid";
char cfg_proxyserver[PROXYSERVER_MAXLEN+1]	= "";
int  cfg_proxyport				= 8080;
char cfg_proxyuser[PROXYUSER_MAXLEN+1]		= "";
char cfg_proxypass[PROXYPASS_MAXLEN+1]		= "";
int  cfg_sendidle	= 1;
#if defined(PLATFORM_UNIXWARE)
int  cfg_sendload	= 0;
#else
int  cfg_sendload	= 1;
#endif
int  cfg_sendos		= 1;
int  cfg_sendoslevel	= 1;
int  cfg_sendcpu	= 1;
int  cfg_sendcpulevel	= 2;

/**
 * Global variables, very ugly, watch through sunglasses!
 *
 * These are some booleans set to reflect the configuration
 * options.
 */
int have_proxyserver	= 0;
int have_proxyport	= 0;
int have_proxyuser	= 0;
int have_proxypass	= 0;


/**
 * @desc	Print options
 */
#if defined(DEBUG)
static void print_options(void) {
  printf("Option cfg_authkey: [%s]\n",cfg_authkey);
  printf("Option cfg_interval: [%d]\n",cfg_interval);
  printf("Option cfg_upserver: [%s]\n",cfg_upserver);
  printf("Option cfg_proxyserver: [%s]\n",cfg_proxyserver);
  printf("Option cfg_proxyport: [%d]\n",cfg_proxyport);
  printf("Option cfg_proxyuser: [%s]\n",cfg_proxyuser);
  printf("Option cfg_proxypass: [%s]\n",cfg_proxypass);
  printf("Option cfg_pidfile: [%s]\n",cfg_pidfile);
  printf("Option cfg_sendidle: [%d]\n",cfg_sendidle);
  printf("Option cfg_sendload: [%d]\n",cfg_sendload);
  printf("Option cfg_sendos: [%d]\n",cfg_sendos);
  printf("Option cfg_sendoslevel: [%d]\n",cfg_sendoslevel);
  printf("Option cfg_sendcpu: [%d]\n",cfg_sendcpu);
  printf("Option cfg_sendlevel: [%d]\n",cfg_sendlevel);
}
#endif

/**
 * @desc	Process key/value pair
 */
static int process_option(const char *key, const char *value) {
  if(strcmp(key, "AuthKey") == 0) {
    if(strlen(value) == AUTHKEY_REQUIRED_LEN) {
      strcpy(cfg_authkey, value);
    } else {
      printf("ERROR: AuthKey is not %d bytes long!\n", AUTHKEY_REQUIRED_LEN);
      exit(-1);
    }
  }
  else if(strcmp(key, "Interval") == 0) {
    cfg_interval = atol(value);
    /**
     * Send the uptime at most once every 30 seconds
     * Send the uptime at least once every 10 minutes
     */
    if(cfg_interval < 30) {
      printf("ERROR: Uptime reporting interval is less than 30 seconds.\n");
      exit(-1);
    }
    if(cfg_interval > 600) {
      printf("WARNING: Uptime reporting interval is larger than 10 minutes."
             "Choosing an interval between 30 seconds and 10 minutes is recommended.\n");
    }
  }
  else if(strcmp(key, "UptimeServer") == 0) {
    strcpy(cfg_upserver, value);
  }
  else if(strcmp(key, "ProxyServer") == 0) {
    strcpy(cfg_proxyserver, value);
    have_proxyserver = 1;
  }
  else if(strcmp(key, "ProxyPort") == 0) {
    cfg_proxyport = atol(value);
    if(!(cfg_proxyport >= 1 && cfg_proxyport <= 65535)) {
      printf("ERROR: ProxyPort should be in the <1 - 65535> range.\n");
      exit(-1);
    }
    have_proxyserver = 1;
  }
  else if(strcmp(key, "ProxyUsername") == 0) {
    strcpy(cfg_proxyuser, value);
    have_proxyserver = 1;
  }
  else if(strcmp(key, "ProxyPassword") == 0) {
    strcpy(cfg_proxypass, value);
    have_proxyserver = 1;
  }
  else if(strcmp(key, "PidFile") == 0) {
    strcpy(cfg_pidfile, value);
  }
  else if(strcmp(key, "SendIdleTime") == 0) {
    cfg_sendidle = atol(value);
  }
  else if(strcmp(key, "SendLoadAvg") == 0) {
#if defined(PLATFORM_UNIXWARE)
    ;
#else
    cfg_sendload = atol(value);
#endif
  }
  else if(strcmp(key, "SendOS") == 0) {
    cfg_sendos = atol(value);
  }
  else if(strcmp(key, "SendOSLevel") == 0) {
    cfg_sendoslevel = atol(value);
  }
  else if(strcmp(key, "SendCPU") == 0) {
    cfg_sendcpu = atol(value);
  }
  else if(strcmp(key, "SendCPULevel") == 0) {
    cfg_sendcpulevel = atol(value);
    if(!(cfg_sendcpulevel == 1 || cfg_sendcpulevel == 2)) {
      printf("ERROR: SendCPULevel should be in the <1 - 2> range.\n");
      exit(-1);
    }
  }
  else if(key[0] != '#') {
    return 0;
  }

  return 1;
}

/**
 * @desc	isblank() for Solaris
 */
#if defined(PLATFORM_SOLARIS)
static int isblank(const char chr) {
  return(chr == ' ' || chr == '\t');
}
#endif

/**
 * @desc	Skip white space
 */
static int skipwhite(const char *line, int pos) {
  while(isblank(line[pos]))
    pos++;
  return pos;
}

/**
 * @desc	Is it an empty line?
 */
static int isemptyline(const char *line) {
  return(line[skipwhite(line,0)] == '\n');
}

/**
 * @desc	Is it a comment line?
 */
static int iscommentline(const char *line) {
  return(line[skipwhite(line,0)] == '#');
}

/**
 * @desc	Is it a valid end of line?
 */
static int isvalideol(const char *line, int pos) {
  pos = skipwhite(line,pos);

  if(line[pos] == '\n')
    return 1;

  if(line[pos] == '#' && line[strlen(line)-1] == '\n')
    return 1;

  return 0;
}

/**
 * @desc	Get key
 */
static int getkey(const char *line, int pos, char *key) {
  int i = 0;

  pos = skipwhite(line,pos);

  while(isalnum(line[pos]) && i < MAX_KEY_LEN) {
    key[i] = line[pos];
    i++;
    pos++;
  }
  key[i] = '\0';

  if(isblank(line[pos]) || line[pos] == '=')
    return pos;
  else
    return 0;
}

/**
 * @desc	Get sign
 */
static int getsign(const char *line, int pos, const char sign) {
  pos = skipwhite(line,pos);
  if(line[pos] == sign)
    return pos+1;
  return 0;
}

/**
 * @desc	Get value of the key
 */
static int getval(const char *line, int pos, char *value) {
  int i = 0;

  pos = skipwhite(line,pos);

  while(isgraph(line[pos]) && line[pos] != '#' && i < MAX_VAL_LEN) {
    value[i] = line[pos];
    i++;
    pos++;
  }
  value[i] = '\0';

  if(isblank(line[pos]) || line[pos] == '#' || line[pos] == '\n')
    return pos;
  else
    return 0;
}

/**
 * @desc	Parse config file line
 */
static int parseline(const char *line, char *key, char *value) {
  int linepos = 0;

  if(isemptyline(line))
    return LINE_EMPTY;

  if(iscommentline(line))
    return LINE_COMMENT;

  linepos = getkey(line,0,key);
  linepos = getsign(line,linepos,'=');
  linepos = getval(line,linepos,value);

  if(!isvalideol(line,linepos))
    return LINE_ERR;

  return LINE_OPTION;
}

/**
 * @desc	Read and parse config file, activate options
 *
 * @caller	main()
 *
 * @param	none
 *
 * @return	true
 */
int read_config(void) {
  FILE *fp;
  char line[MAX_LINE_LEN];
  char key[MAX_KEY_LEN];
  char value[MAX_VAL_LEN];
  int linenr = 1;
  int linetype;

  /* Try to open config file in current directory */
  if(!(fp = fopen(CONFIGFILE, "r"))) {
    /* Failed, now try to open config file in /etc */
    if(!(fp = fopen("/etc/"CONFIGFILE, "r"))) {
      /* Failed, now try to open config file in /usr/local/etc */
      if(!(fp = fopen("/usr/local/etc/"CONFIGFILE, "r"))) {
        printf("Error: couldn't open config file %s for reading in:\n- current directory\n- /etc\n- /usr/local/etc\n", CONFIGFILE);
        exit(-1);
      }
    }
  }

  /* Read & parse config file */
  while(!feof(fp)) {
    /* Read one line */
    fgets(line,MAX_LINE_LEN,fp);

    /* We got a problem Houston */
    if(line[strlen(line)-1] != '\n') {
      printf("ERROR: line %d too long.\n", linenr);
      exit(-1);
    }

    /* Parse rule */
    linetype = parseline(line,key,value);

    if(linetype == LINE_ERR) {
      printf("WARNING: Error in %s line %d:\n  content: %s\n",CONFIGFILE,linenr,line);
    }

    /* If the line is an option, process it */
    if(linetype == LINE_OPTION) {
      if(!process_option(key,value)) {
        printf("WARNING: Error in %s line %d:\n  key: %s, value: %s\n",CONFIGFILE,linenr,key,value);
      }
    }

    /* The core functionality of this algorithm */
    linenr++;
  }

#if defined(DEBUG)
  print_options();
#endif

  /* Clean up the mess we made */
  fclose(fp);

  /* Life is giving and taking */
  return 1;
}
