/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  This file plays music for the UBS.  Actually, it simply reads filenames one at a time from the queue and passes them off to an external media player via play_song()
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef WIN32
#include <Windows.h>
#endif

#include "ubs-play.h"
#include "ubs-main.h"

/*+
  Catch an event signal, which will be sent by ubs-event if it wants us to do something.  This basically sets the do_event variable to be true, which will cause the main loop to run code for an event rather than playing a song from the queue.

  void sig_event Returns nothing

  int sig The signal number.  Used for debugging purposes only.
+*/
void sig_event(int sig)
{
 log_error_msg(LOG_DEBUG, "Caught event signal");
 do_event = 1;
}

void sig_quit_killmedia(int sig)
{
 FILE *fp;
 char buf[STRBUF];
 pid_t pid;

 if((fp = fopen("var/ubs-media.pid", "r")) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open ubs-media.pid");
 }
 else {
  if(fgets(buf, STRBUF, fp) == NULL) {
   log_error_msg(LOG_ERROR, "Unable to read data from ubs-media.pid");
  }
  else {
   pid = (pid_t)atoi(buf);
   log_error_msg(LOG_DEBUG, "Killing media process %d", pid);
   kill(pid, SIGQUIT);
  }
 }

 snprintf(buf, STRBUF, "var/%s.pid", PROCNAME);
 unlink(buf);

 record_status("Terminated");

 log_error_msg(LOG_EMERG, "Exiting with signal %d", sig);
 exit(sig);
}

/*+
  The main function

  int main Returns OK on success

  int argc Should be empty; this program does not take any arguments

  char *argv[] Should be empty; this program does not take any arguments
+*/
int main(int argc, char *argv[])
{
 FILE *fp;
 char sname[FILEBUF];
 struct media_tag song_tag;

 do_event = 0;

 // initialize the global table and such
 ubs_table_init(&GLOBAL);
 ubs_table_init(&PLAY);

 ubs_init(argv[0]);

 // event handler signal
 if(signal(SIGUSR1, sig_event) == SIG_ERR) {
  console_error("Could not bind SIGUSR1", NO_SIGNAL);
 }

 // initialize values specific to the play daemon
 ubs_table_add(&PLAY, "media", DEF_MEDIA);

 if(read_config(NULL, "global", &GLOBAL)) {
  console_error("Can't read global settings in configuration file", FAIL);
 }
 // play specific context
 if(read_config(NULL, "play", &PLAY)) {
  console_error("Can't read play settings in configuration file", FAIL);
 }

 if(check_running() == FAIL) {
  console_error("Daemon already seems to be running", FAIL);
 }

 // if the killmedia directive is defined, then rebind a few
 // signal calls to take out the media player before exiting.
 // This was implemented as a feature as of v0.17, rather than
 // being hardcoded, as it was slightly controversial
 if(ubs_table_exists(&PLAY, "killmedia")) {
  if(!strcasecmp(ubs_table_data(&PLAY, "killmedia"), "yes")) {
   log_error_msg(LOG_DEBUG, "Rebinding signals to kill media player");
   if(signal(SIGINT, sig_quit_killmedia) == SIG_ERR) {
    console_error("Unable to bind signal SIGINT", NO_SIGNAL);
   }
   if(signal(SIGHUP, sig_quit_killmedia) == SIG_ERR) {
    console_error("Unable to bind signal SIGHUP", NO_SIGNAL);
   }
   if(signal(SIGQUIT, sig_quit_killmedia) == SIG_ERR) {
    console_error("Unable to bind signal SIGQUIT", NO_SIGNAL);
   }
   if(signal(SIGTERM, sig_quit_killmedia) == SIG_ERR) {
    console_error("Unable to bind signal SIGTERM", NO_SIGNAL);
   }
   if(signal(SIGSEGV, sig_quit_killmedia) == SIG_ERR) {
    console_error("Unable to bind signal SIGSEGV", NO_SIGNAL);
   }
  }
 }

#ifndef WIN32
 log_error_msg(LOG_EMERG, "Process started as uid/gid %d/%d", geteuid(), getgid());
#else
 log_error_msg(LOG_EMERG, "Process started");
#endif

#ifndef WIN32
 // fork into the background
 if(fork()) {
  // parent exits
  exit(0);
 }
#endif

 log_error_msg(LOG_DEBUG, "Becoming the process leader");
 setsid();
 log_error_msg(LOG_DEBUG, "Changing working directory to PREFIX");
 chdir(ubs_table_data(&GLOBAL, "prefix"));
 log_error_msg(LOG_DEBUG, "Setting mask to 000");
 umask(0);
 log_error_msg(LOG_DEBUG, "Initializing the queue");
 queue_init();

 record_pid(getpid());
 record_status("OK");

 while(1) {
  if(do_event) {
   if((fp = fopen("tmp/next-event", "r")) == NULL) {
    log_error_msg(LOG_ERROR, "Unable to open next-event file");
    do_event = 0;
    continue;
   }

   if(fgets(sname, BUFSIZ, fp) == NULL) {
    log_error_msg(LOG_ERROR, "Unable to read from next-event file");
    do_event = 0;
    continue;
   }
   fclose(fp);

   chomp_line(sname);
   if(get_media_tag(sname, &song_tag) != OK) {
    log_error_msg(LOG_DEBUG, "Error reading media tag");
   }
   log_error_msg(LOG_DEBUG, "Playing Event: %s", sname);

   if((fp = fopen("var/now_playing", "w")) == NULL) {
    log_error_msg(LOG_ERROR, "Unable to truncate now_playing file");
   }
   else {
    fprintf(fp, "%s", sname);
    fclose(fp);
   }

   do_event = 0;
   log_error_msg(LOG_DEBUG, "Preparing to play song for event");
   play_song(sname);

   // remove event file if it exists
   unlink("tmp/next-event");
  }
  else {
   // get a song from the queue
   while(queue_length() <= 0) {
    log_error_msg(LOG_DEBUG, "Queue is empty...sleeping 1 second");
    sleep(1);
   }
   queue_pop(sname);
   chomp_line(sname);
   if(get_media_tag(sname, &song_tag) != OK) {
    log_error_msg(LOG_DEBUG, "Error reading media tag");
   }

   if((fp = fopen("var/now_playing", "w")) == NULL) {
    log_error_msg(LOG_ERROR, "Unable to write to now_playing file");
   }
   else {
    fprintf(fp, "%s", sname);
    fclose(fp);
   }

   log_error_msg(LOG_DEBUG, "Preparing to chop queue");
   queue_chop();
   log_error_msg(LOG_DEBUG, "Playing song: %s", sname);
   play_song(sname);
   log_event(ubs_table_data(&GLOBAL, "songlog"), sname, song_tag.artist,
             song_tag.title, LOG_EMERG);
  }
  record_status("OK");
 }

 return OK;
}
