/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  This file reads in the ubs.conf file, looking for anything in the events context, then adding these events to the ALLEVENTS array.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "ubs-event.h"

/*+
  Open up a configuration file and put the events into an array.  This function expects each event will be on a line, in the form event = time.

  int read_events Returns NO on success (same as OK), YES if one or more events is invalid, NO_MEM for memory allocation errors, FAIL if 0 events were found

  char *config_fname The filename of the configuration file to parse
+*/
int read_events(char *config_fname)
{
 char buf[BIGBUF], *p, *time_str;
 int bad = NO, i;
 ubs_table event;

 ubs_table_init(&event);

 if((ALLEVENTS = (struct event *)malloc(atoi(ubs_table_data(&EVENT, "numevents")) * sizeof(struct event))) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to allocate memory for ALLEVENTS array");
  return NO_MEM;
 }

 if(atoi(ubs_table_data(&EVENT, "numevents")) == 0) {
  return FAIL;
 }
 log_error_msg(LOG_DEBUG, "%s total events found", ubs_table_data(&EVENT, "numevents"));

 for(i = 1; i <= atoi(ubs_table_data(&EVENT, "numevents")); i++) {
  snprintf(buf, BIGBUF, "event%d", i);
  if(read_config(config_fname, buf, &event)) {
   fprintf(stdout, "ERROR: Cannot parse event #%d\n", i);
   bad = YES;
   continue;
  }
  else {
   // if the event is invalid, then print an error message and get out
   // for the iteration of the loop
   if(ubs_table_data(&event, "name") == NULL) {
    fprintf(stdout, "Event #%d is invalid: No name given\n", i);
    bad = YES;
   }
   else if(ubs_table_data(&event, "logfile") == NULL) {
    fprintf(stdout, "Event #%d is invalid: No logfile given\n", i);
    bad = YES;
   }
   else if(ubs_table_data(&event, "time") == NULL) {
    fprintf(stdout, "Event #%d is invalid: No time given\n", i);
    bad = YES;
   }

   // if a specific time is specified, then split it up into hours/minutes.
   // otherwise, drop -1 in the hours and leave the minutes as is.  Go
   // reverse compatibility!
   time_str = strdup(ubs_table_data(&event, "time"));
   if((p = strchr(time_str, ':')) != NULL) {
    *p = '\0';
    ALLEVENTS[i - 1].hrs = atoi(time_str);
    ALLEVENTS[i - 1].mins = atoi(p + 1);
   }
   else {
    ALLEVENTS[i - 1].mins = atoi(time_str);
    ALLEVENTS[i - 1].hrs = -1;
   }

   if(bad) {
    continue;
   }

   strncpy(ALLEVENTS[i - 1].name, ubs_table_data(&event, "name"), STRBUF);
   strncpy(ALLEVENTS[i - 1].logfile, ubs_table_data(&event, "logfile"), FILEBUF);

   log_error_msg(LOG_DEBUG, "Added event %d: %s/%s/%d:%d", i, ALLEVENTS[i - 1].name, ALLEVENTS[i - 1].logfile, ALLEVENTS[i - 1].hrs, ALLEVENTS[i - 1].mins);
  }
 }

 return bad;
}
