/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <sys/param.h>
#include <sys/stat.h>

#include "ubs-sched.h"

/*+
  This function reads in the current filecache (all the known songs for the current show) and picks one at random.  It does nothing to see if this song was recently played or anything of the sort.  That functionality is handled by the check_against_log function.

  int get_song Returns OK on success and NO_FILE if the filecache can't be opened

  char *sname A pointer to a character array which this function will write the selected filename to
+*/
int get_song(char *sname)
{
 FILE *fp;
 int i = 0, index = 0, num_files;

 if((fp = fopen("tmp/numfiles.dat", "r")) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open numfiles for reading");
  return NO_FILE;
 }

 fscanf(fp, "%d", &num_files);
 fclose(fp);
 index = rand() % num_files;

 if(num_files <= 0) {
  // enqueue default sound (?)
  return FAIL;
 }

 if((fp = fopen("tmp/filecache.dat", "r")) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open filecache for reading");
  return NO_FILE;
 }

 while(fgets(sname, FILEBUF, fp)) {
  i++;
  if(i > index) {
   break;
  }
 }

 fclose(fp);
 chomp_line(sname);
 return OK;
}

/*+
  This function gets called whenever the UBS starts up, and whenever it detects that a show change has occurred.  Basically, it scans all the files in the directory for the current show and writes them to a big file, one filename per line.  This was made recursive as of version 0.7.

  int make_cache Returns OK on success, NO_FILE if the filecache can't be opened, NO_FILE if the numfiles file can't be opened

  char *path The pathname of the directory to scan
+*/
int make_cache(char *path)
{
 FILE *fp;
 int num_files = 0;

 if((fp = fopen("tmp/filecache.dat", "w")) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open filecache for writing");
  return NO_FILE;
 }
 num_files = parse_dir(fp, path);
 fclose(fp);

 if((fp = fopen("tmp/numfiles.dat", "w")) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open numfiles for writing");
  return NO_FILE;
 }

 fprintf(fp, "%d", num_files);
 fclose(fp);

 return OK;
}

/*+
  Finds the number of files in a directory, and writes out all files it finds to a file pointer.

  int parse_dir Returns the number of files found in the given show directory, and all its child directories.  If this functon finds another directory within the current one, it will call itself recursively and keep adding to the filecache.  NO_DIR is returned if a directory cannot be opened for whatever reason.

  FILE *fp A filepointer, which should be opened with write permissions

  char *path The directory to start scanning at
+*/
int parse_dir(FILE *fp, char *path)
{
 DIR *dp;
 struct dirent *ds;
 struct stat fs;
 char buf[FILEBUF];
 int num_files = 0;

 if((dp = opendir(path)) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open directory '%s' for reading", path);
  return NO_DIR;
 }
 log_error_msg(LOG_DEBUG, "Scanning directory '%s'", path);

 while((ds = readdir(dp)) != NULL) {
  // filter out . and ..
  if(strcmp(".", ds->d_name) && strcmp("..", ds->d_name)) {
   snprintf(buf, FILEBUF, "%s/%s", path, ds->d_name);
   if(!stat(buf, &fs)) {
    if(S_ISDIR(fs.st_mode)) {
     // was used in initial debugging...leave it out, because it prints
     // a lot of junk
     //printf("%s is a directory\n", ds->d_name);
     snprintf(buf, FILEBUF, "%s/%s", path, ds->d_name);
     num_files += parse_dir(fp, buf);
    }
    else {
     num_files++;
     fprintf(fp, "%s/%s\n", path, ds->d_name);
     // was used in initial debugging...leave it out, because it prints
     // a lot of junk
     // printf(" %s, %d\n", ds->d_name, num_files);
    }
   }
  }
 }
 closedir(dp);

 if(num_files > 0) {
  return num_files;
 }
 else {
  return FAIL;
 }
}

/*+
  Scans the playlist logfile to see if a filename has been recently played.

  int check_against_log Returns NO if the file is not in the last backnum lines of the playlist log, YES if the file is matched, NO_FILE if the target logfile can't be opened, NO_MEM if malloc fails

  char *filename The filename to check

  char *logfile The logfile to check in

  int backnum The number of lines (at the tail end of the file) to scan
+*/
int check_against_log(char *filename, char *logfile, int backnum)
{
 FILE *fp;
 char line_buf[BIGBUF];
 struct log_entry entry;
 int match = NO, i, num_entries = 0;
 char **backfiles;

 if((backfiles = (char **)malloc(sizeof(char *) * backnum)) == NULL) {
  log_error_msg(LOG_ERROR, "Could not allocate memory for backfiles");
  return NO_MEM;
 }
 for(i = 0; i < backnum; i++) {
  if((backfiles[i] = (char *)malloc(sizeof(char) * BIGBUF)) == NULL) {
   log_error_msg(LOG_ERROR, "Could not allocate memory for backfiles");
   return NO_MEM;
  }
 }

 if((fp = fopen(logfile, "r")) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open logfile for checking");
  return NO_FILE;
 }

 for(i = 0; fgets(line_buf, BIGBUF, fp) != NULL; i = (i + 1) % backnum) {
  memset(&entry, 0x0, sizeof(struct log_entry));
  read_log(&entry, line_buf);
  strncpy(backfiles[i], entry.field1, BIGBUF);
  num_entries++;
 }
 fclose(fp);

 if(num_entries > backnum) {
  for(i = 0; i < backnum; i++) {
   log_error_msg(LOG_DEBUG, "Checking '%s' against backlog file '%s'", filename, backfiles[i]);
   if(!strcasecmp(backfiles[i], filename) ||
      check_against_tags(backfiles[i], filename)) {
    match = YES;
    break;
   }
  }
 }
 else {
  for(i = 0; i < num_entries; i++) {
   if(!strcasecmp(backfiles[i], filename) ||
      check_against_tags(backfiles[i], filename)) {
    match = YES;
    break;
   }
  }
 }

 // all clear
 for(i = 0; i < backnum; i++) {
  free(backfiles[i]);
 }
 free(backfiles);
 return match;
}

/*+
  Checks to see if a filename is already in the present queue

  int check_against_queue Returns NO (same as OK) if the filename isn't in the queue, YES if the filename is in the queue, and NO_FILE if the queue can't be opened

  char *filename The filename to check
+*/
int check_against_queue(char *filename)
{
 FILE *fp;
 char line_buf[BIGBUF];

 if((fp = fopen(ubs_table_data(&GLOBAL, "queue"), "r")) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open queue for checking");
  return NO_FILE;
 }
 while(fgets(line_buf, BIGBUF, fp) != NULL) {
  chomp_line(line_buf);
  if(!strcasecmp(filename, line_buf) ||
     check_against_tags(line_buf, filename)) {
   // found something
   fclose(fp);
   return YES;
  }
 }

 fclose(fp);
 // all clear
 return NO;
}

/*+
  Checks to see if a filename is currently playing on the air.  Since the ubs-play daemon pulls the file from the queue before playing it, there is a slim chance that a file could be queued twice once it is on the air.

  int check_against_cur Returns NO if no match, YES if a match is found, and NO_FILE if the now_playing file cannot be opened.

  char *filename Filename to check
+*/
int check_against_cur(char *filename)
{
 FILE *fp;
 char line_buf[BIGBUF];

 if((fp = fopen("var/now_playing", "r")) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open now_playing for reading");
  return NO_FILE;
 }
 while(fgets(line_buf, BIGBUF, fp) != NULL) {
  chomp_line(line_buf);
  if(!strcasecmp(filename, line_buf) ||
     check_against_tags(line_buf, filename)) {
   // found something
   fclose(fp);
   return YES;
  }
 }

 fclose(fp);
 // all clear
 return NO;
}
