/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  Contains functions common to most UBS daemons.  This includes the version information, signal, and configuration defaults.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <sys/types.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "ubs.h"

/*+
  Displays a quick blurb about the program and the current version.  The version number is declared in config.h, which is generated by autoconf.
+*/
void version(void)
{
 printf("UBS Uninterrupted Broadcasting Daemon -- version %s\n", VERSION);
 printf("For WMHD 90.7 'The Monkey' @ Rose-Hulman Institute of Technology\n");
 printf("http://wmhd.rose-hulman.edu -- http://www.aboleo.net/software/ubs\n");
 printf("By Nik Reiman -- nik@aboleo.net\n");
}

/*+
  The polling signal is used to see if the daemon is alive.  Anyone can send a polling signal (currently defined as SIGURG (SIGUSR isn't defined on all unix platforms, and SIGUSR is pretty much unused, in case you were wondering.

  void sig_poll Returns nothing

  int sig The number of the signal received
+*/
void sig_poll(int sig) {
 log_error_msg(LOG_DEBUG, "Caught a poll signal");
}

/*+
  By default, the main UBS daemons bind a number of various quit common signals (SIGINT, SIGHUP, SIGQUIT, and so forth).  This is where they go to die.  Before the program exits (or crashes, as the case may be), it's nice to log it and such.  This could probably be expanded to include more information for debugging in the future.

  void sig_quit Returns nothing

  int sig The number of the signal received
+*/
void sig_quit(int sig) {
 char buf[STRBUF];

 // call generic cleanup function
 //ubs_cleanup();

 snprintf(buf, STRBUF, "var/%s.pid", PROCNAME);
 unlink(buf);

 record_status("Terminated");

 log_error_msg(LOG_EMERG, "Exiting with signal %d", sig);
 exit(sig);
}

/*+
  Called by all UBS events and daemons upon initialization.  This binds all the default signals, and copies default values into the GLOBAL configuration.  Also records the process name and other good stuff.  This should be called right after the GLOBAL table is initialized.

  int ubs_init Returns OK on success

  char *pname The program name, same as argv[0]
+*/
int ubs_init(char *pname) {
 char *p;

 // record the process name
 if((p = strrchr(pname, '/')) != NULL) {
  strncpy(PROCNAME, p + 1, STRBUF);
 }
 else {
  strncpy(PROCNAME, pname, STRBUF);
 }

 // bind some signals
 if(signal(SIGINT, sig_quit) == SIG_ERR) {
  console_error("Unable to bind signal SIGINT", NO_SIGNAL);
 }
 if(signal(SIGHUP, sig_quit) == SIG_ERR) {
  console_error("Unable to bind signal SIGHUP", NO_SIGNAL);
 }
 if(signal(SIGQUIT, sig_quit) == SIG_ERR) {
  console_error("Unable to bind signal SIGQUIT", NO_SIGNAL);
 }
 if(signal(SIGTERM, sig_quit) == SIG_ERR) {
  console_error("Unable to bind signal SIGTERM", NO_SIGNAL);
 }
 // yes, that's right.  bind segfault.
 if(signal(SIGSEGV, sig_quit) == SIG_ERR) {
  console_error("Unable to bind signal SIGSEGV", NO_SIGNAL);
 }
 // polling signal
 if(signal(SIGURG, sig_poll) == SIG_ERR) {
  console_error("Unable to bind signal SIGURG", NO_SIGNAL);
 }
 // reset the child signal handler
 if(signal(SIGCHLD, SIG_DFL) == SIG_ERR) {
  console_error("Unable to reset SIGCHLD", NO_SIGNAL);
 }

 srand(time(NULL));
 set_defaults();

 return OK;
}

/*+
  Sees if a UBS process is already running before starting.

  int check_running Returns OK if not running, FAIL if the process is already started.
+*/
int check_running(void) {
 if(ping_pid(PROCNAME) == OK) {
  return FAIL;
 }
 else {
  return OK;
 }
}

/*+
  Copies default values into the global context.

  set_defaults Returns nothing.
+*/
void set_defaults(void) {
 // set up some default variable settings
 if(!strcmp(PREFIX, "NONE")) {
  ubs_table_add(&GLOBAL, "prefix", DEF_PREFIX);
 }
 else {
  ubs_table_add(&GLOBAL, "prefix", PREFIX);
 }
 ubs_table_add(&GLOBAL, "queue", DEF_QUEUE);
 ubs_table_add(&GLOBAL, "errorlog", DEF_ERRORLOG);
 ubs_table_add(&GLOBAL, "songlog", DEF_SONGLOG);
 ubs_table_add(&GLOBAL, "loglevel", DEF_LOGLEVEL_STR);
}
