/****************************************************************************/
/*                                                                          */
/*                      Create a signing Key                                */
/*                                                                          */
/*  This file is copyright 2003 IBM. See "License" for details              */
/****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <openssl/rsa.h>
#include <openssl/pem.h>
#include <openssl/evp.h>
#include <tcpa.h>

int main(int argc, char *argv[])
{
    uint32_t ret, parhandle;      /* handle of parent key */
    unsigned char hashpass1[20];  /* hash of parent key password */
    unsigned char hashpass2[20];  /* hash of new key password */
    keydata k;                    /* keydata structure for input key parms */
    keydata q;                    /* keydata structure for resulting key */
    RSA *rsa;                     /* OpenSSL format Public Key */
    FILE *keyfile;                /* output file for public key */
    EVP_PKEY pkey;                /* OpenSSL public key */
    unsigned char pkeyname[256];  /* file name string of public key file */

    if (argc < 5) {
        fprintf(stderr, "Usage: createkey <parent key handle in hex> <parent key password> <new key name> <new key password>\n");
        exit(-1);
    }
    ret = sscanf(argv[1], "%x", &parhandle);
    if (ret != 1) {
        fprintf(stderr, "Invalid argument '%s'\n", argv[1]);
        exit(-1);
    }
    sha1(argv[2], strlen(argv[2]), hashpass1);
    sha1(argv[4], strlen(argv[4]), hashpass2);
    k.version[0] = 1;           /* Version 1.1.0.0 */
    k.version[1] = 1;
    k.version[2] = 0;
    k.version[3] = 0;
    k.keyusage = 0x0010;        /* key Usage - 0x0010 = signing */
                                /*             0x0011 = storage */
    k.keyflags = 0;             /* key flags - none */
    k.authdatausage = 0x01;     /* key requires authorization (password) */
    k.privkeylen = 0;           /* no private key specified here */
    k.pub.algorithm = 0x00000001;       /* key algorithm 1 = RSA */
    k.pub.encscheme = 0x0001;   /* encryption scheme 1 = NONE - signing key */
                                /*                   3   rsastorage */
    k.pub.sigscheme = 0x0002;   /*  2 == signature scheme RSA/SHA1  */
                                /* 1 == none, (storage key) */
    k.pub.keybitlen = 2048;     /* RSA modulus size 2048 bits */
    k.pub.numprimes = 2;        /* required */
    k.pub.expsize = 0;          /* RSA exponent - default 0x010001 */
    k.pub.keylength = 0;        /* key not specified here */
    k.pub.pcrinfolen = 0;       /* no PCR's used at this time */

    ret = TPM_CreateWrapKey(parhandle, hashpass1, hashpass2, NULL, 
		    &k, &q, TPMLOG);
    if (ret != 0) {
        fprintf(stderr, "Error %d from TPM_CreateKey\n", ret);
        exit(-1);
    }
    printf("Created key with name %s\n", argv[3]);

    /* write out keydata struct for future TPM_LoadKey calls */
    snprintf(pkeyname,sizeof(pkeyname), "%s.key", argv[3]);
    keyfile = fopen(pkeyname, "w");
    fwrite(&q,sizeof(q),1,keyfile);
    fclose(keyfile);

    /* write out public key in pem format for future openssl use */
    rsa = convpubkey(&(q.pub));
    if (rsa == NULL) {
        fprintf(stderr, "Error from TSS_convpubkey\n");
        exit(-1);
    }
    OpenSSL_add_all_algorithms();
    snprintf(pkeyname,sizeof(pkeyname),"%s.pem", argv[3]);
    EVP_PKEY_assign_RSA(&pkey, rsa);
    keyfile = fopen(pkeyname, "w");
    if (keyfile == NULL) {
        fprintf(stderr, "Unable to create public key file\n");
        exit(-1);
    }
    ret = PEM_write_PUBKEY(keyfile, &pkey);
    if (ret == 0) {
        fprintf(stderr, "Unable to write public key file\n");
        exit(-1);
    }
    fclose(keyfile);
    RSA_free(rsa);
    exit(0);
}
