/* tn5250 -- an implentation of the 5250 telnet protocol.
 * Copyright (C) 1997 Michael Madore
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "buffer.h"
#include "utility.h"

// Number of bytes to grow buffer by.  This should reduce number of
// news and deletes.
#define BUFFER_DELTA 128

// Just blow up the program on purpose before we do it accidentally.  :-)
void Buffer::range_check (int idx) const
{
	assert (idx >= 0);
	assert (idx < len);
}

Buffer::Buffer ()
{
	len = allocated = 0;
	data = NULL;
}

Buffer::Buffer (const Buffer& rhs)
{
	len = rhs.len;
	allocated = rhs.allocated;
	data = new unsigned char[allocated];
	memcpy (data, rhs.data, len);
}

Buffer::Buffer (unsigned char c)
{
	len = 1;
	allocated = BUFFER_DELTA;
	data = new unsigned char [BUFFER_DELTA];
	data[0] = c;
}

Buffer::Buffer (unsigned char *newdata, int newlen)
{
	allocated = len = 0;
	data = NULL;
	length (newlen);
	memcpy (data, newdata, newlen);
}

Buffer::~Buffer ()
{
	if (data)
		delete[] data;
}

int Buffer::length () const
{
	return len;
}

int Buffer::length (int newlength)
{
	int oldlength = len;
	int newalloc = newlength;
	unsigned char *olddata = data;

	// adjust newalloc to be divisble by BUFFER_DELTA
	assert (newlength >= 0);
	if ((newalloc % BUFFER_DELTA) != 0)
		newalloc = ((newalloc/BUFFER_DELTA) + 1) * BUFFER_DELTA;

	// reallocate if necessary
	if (newalloc != allocated) {
		int mincopy;

		if (newalloc == 0)
			data = NULL;
		else
			data = new unsigned char[newalloc];
		allocated = newalloc;

		mincopy = (oldlength < newlength ? oldlength : newlength);
		if (mincopy > 0)
			memcpy (data, olddata, mincopy);

		if (olddata != NULL)
			delete[] olddata;
	}

	// set the new length, return the old
	len = newlength;
	return oldlength;
}

unsigned char Buffer::at (int idx) const
{
	range_check (idx);
	return data[idx];
}

unsigned char& Buffer::at (int idx)
{
	range_check (idx);
	return data[idx];
}

Buffer& Buffer::set (const Buffer& rhs)
{
	if (&rhs == this)
		return *this;
	return set (rhs.data, rhs.len);
}

Buffer& Buffer::set (unsigned char c)
{
	return set (&c, 1);
}

Buffer& Buffer::set (unsigned char *newdata, int newlen)
{
	length (newlen);
	if (newlen > 0)
		memcpy (data, newdata, newlen);
	return *this;
}

Buffer& Buffer::append (const Buffer& rhs)
{
	assert (&rhs != this);
	return append (rhs.data, rhs.len);
}

Buffer& Buffer::append (unsigned char c)
{
	return append (&c, 1);
}

Buffer& Buffer::append (unsigned char *newdata, int newlen)
{
	int oldlength = len;
	assert (newlen >= 0);
	length (len + newlen);
	if (newlen > 0)
		memcpy (data + oldlength, newdata, newlen);
	return *this;
}

int Buffer::compare (const Buffer& rhs) const
{
	int maxcmp = (len < rhs.len ? len : rhs.len);
	int r;
	
	if (maxcmp == 0)
		r = 0;
	else
		r = memcmp (data, rhs.data, maxcmp);

	if (r != 0 || rhs.len == len)
		return r;

	return (rhs.len > len ? 1 : -1);
}
