/*
 * tcl+gdbm 0.4,
 * Copyright (C) 1994 Technical University of Braunschweig, Germany
 * Written by Christian Lindig (lindig@ips.cs.tu-bs.de)
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

/*
 * tclgdbm-0.8
 * Tclgdbm was hacked together by John Ellson (ellson@lucent.com)
 * derived from tcl+gdbm by Christian Lindig <lindig@ips.cs.tu-bs.de>
 * Additions by H.-Juergen Godau (JG) <godau@wi-inf.uni-essen.de>
 */

/*
 * TclGdbm
 * Merge the best parts of both the above packages, convert to a
 * Tcl_Obj'ified command in its own namespace, and general cleanup.
 * Dave Bodenstab <imdave@mcs.net>
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <tcl.h>
#include <gdbm.h>

/*
 * a database ..
 */
typedef struct db
  {
    char *name;
    int mode;
    int write_mode;
    GDBM_FILE db;
  }
    DB;

/*
 * All open databases
 */
static Tcl_HashTable databases;

/*
 * Counter for handles "gdbm<handle>" returned by open
 */
static int handle = 0;


/*
 * inline functions
 */
#define RetDatum(interp,d) \
	do { \
	     if (d.dptr) \
	       { \
		 Tcl_SetObjResult( interp, Tcl_NewStringObj(d.dptr,d.dsize) ); \
		 free( d.dptr ); \
	       } \
	     return TCL_OK; \
	   } \
	     while( 0 )

#define toExternalString( string, obj ) \
 	do { \
	     char *utf; \
	     int utflength; \
	     utf = Tcl_GetStringFromObj( obj, &utflength ); \
	     Tcl_DStringInit( string ); \
	     Tcl_UtfToExternalDString( NULL, utf, utflength, string ); \
	   } \
	     while( 0 )

/*
 * Lookup the database from the table of all open databases using a
 * given handle. 
 *
 * Returns a pointer to the relevant database record on success or
 * a NULL pointer on failure with result set to an error message.
 */
static DB*
lookup( Tcl_Interp *interp, Tcl_Obj *handle, Tcl_HashEntry **entryp )
{
  char *dbhandle;
  Tcl_HashEntry *entry;

  if ( (entry = Tcl_FindHashEntry(&databases,dbhandle=Tcl_GetString(handle))) == NULL )
    {
      Tcl_AppendResult( interp, "can not find database named \"", dbhandle, "\"", NULL );
      return NULL;
    }

  if ( entryp != NULL )
    *entryp = entry;

  /*
   * retrieve db from hash table
   */
  return (DB*) Tcl_GetHashValue( entry );
}

/*
 * gdbm close <db>
 *
 * Close a given database and remove the corresponding record from the
 * hashtable of open databases.
 */
static int 
close( ClientData client, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[] )
{
  DB *db;
  Tcl_HashEntry *entry;

  if ( objc != 3 )
    {
      Tcl_WrongNumArgs( interp, 2, objv, "dbHandle" );
      return TCL_ERROR;
    }

  if ( (db = lookup(interp,objv[2],&entry)) == NULL )
    return TCL_ERROR;

  Tcl_DeleteHashEntry( entry );

  gdbm_close( db->db );

  ckfree( (char*)db );

  return TCL_OK;
}

/*
 * gdbm insert <db> <key> <content> ...
 * gdbm store <db> <key> <content> ...
 *
 * Insert <content> under <key> into the database associated with
 * handle <db>.
 *
 * When called as "gdbm insert" an existing key value pair will not be
 * replaced and an error is reported.  When called as "gdbm store" an
 * existing pair is silently replaced or an error is reported depending
 * on the current value of write_mode.
 */
static int 
insert( ClientData client, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[] )
{
  datum key, data;
  int i, mode, ret;
  DB *db;
  Tcl_DString kstr, dstr;
  Tcl_Obj *obj;

  if ( objc < 5 )
    {
      Tcl_WrongNumArgs( interp, 2, objv, "dbHandle key data ?data?..." );
      return TCL_ERROR;
    }

  if ( (db = lookup(interp,objv[2],NULL)) == NULL )
    return TCL_ERROR;

  /* check whether insert or (default) replace mode */
  mode = (*Tcl_GetString(objv[1]) == 'i') ? GDBM_INSERT : db->write_mode;

  /*
   * prepare data and store it
   */
  toExternalString( &kstr, objv[3] );

  key.dptr = Tcl_DStringValue( &kstr );
  key.dsize = Tcl_DStringLength( &kstr );

  if ( objc == 5 )
    /* single value */
    toExternalString( &dstr, objv[4] );
  else
    /* multiple values - build up list */
    {
      obj = Tcl_NewListObj( 1, &objv[4] );

      for( i = 5; i < objc; i++ )
	{
	  if ( Tcl_ListObjAppendElement(interp,obj,objv[i]) == TCL_ERROR )
	    {
	      Tcl_DecrRefCount( obj );
	      Tcl_DStringFree( &kstr );
	      return TCL_ERROR;
	    }
	}

      toExternalString( &dstr, obj );
      Tcl_DecrRefCount( obj );
    }

  data.dptr = Tcl_DStringValue( &dstr );
  data.dsize = Tcl_DStringLength( &dstr );

  ret = gdbm_store( db->db, key, data, mode );

  Tcl_DStringFree( &kstr );
  Tcl_DStringFree( &dstr );

  if ( ret != 0 )
    {
      Tcl_AppendResult( interp, "gdbm error: ", gdbm_strerror(gdbm_errno), NULL );
      return TCL_ERROR;
    }

  return TCL_OK;
}

/*
 * gdbm reorganize <db>
 *
 * Reorganize the GDBM database.
 */
static int 
reorganize( ClientData client, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[] )
{
  int ret;
  DB *db;

  if ( objc != 3 )
    {
      Tcl_WrongNumArgs( interp, 2, objv, "dbHandle" );
      return TCL_ERROR;
    }

  if ( (db = lookup(interp,objv[2],NULL)) == NULL )
    return TCL_ERROR;

  ret = gdbm_reorganize( db->db );

  if ( ret != 0 )
    {
      Tcl_AppendResult( interp, "gdbm error: ", gdbm_strerror(gdbm_errno), NULL );
      return TCL_ERROR;
    }

  return TCL_OK;
}

/*
 * gdbm fetch <db> <key>
 * 
 * Retrieve contents from database <db> under key <key> and return
 * it.  If no such <key> in <db> exists an error is reported.
 */
static int 
fetch( ClientData client, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[] )
{
  DB *db;
  datum key, data;
  Tcl_DString kstr;

  if ( objc != 4 )
    {
      Tcl_WrongNumArgs( interp, 2, objv, "dbHandle key" );
      return TCL_ERROR;
    }

  if ( (db = lookup(interp,objv[2],NULL)) == NULL )
    return TCL_ERROR;

  toExternalString( &kstr, objv[3] );

  key.dptr = Tcl_DStringValue( &kstr );
  key.dsize = Tcl_DStringLength( &kstr );

  data = gdbm_fetch( db->db, key );

  Tcl_DStringFree( &kstr );

  if ( data.dptr == NULL )
    {
      Tcl_AppendResult( interp, "gdbm error: key \"", Tcl_GetString(objv[3]), "\" not found", NULL );
      return TCL_ERROR;	
    }

  RetDatum( interp, data );
}

/*
 * gdbm exists <db> <key>
 *
 * Checks wehther <key> exists in <db> and returns 1 if <key> exists,
 * 0 otherwise
 */
static int 
exists( ClientData client, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[] )
{
  DB *db;
  int exists;
  datum key;
  Tcl_DString kstr;

  if ( objc != 4 )
    {
      Tcl_WrongNumArgs( interp, 2, objv, "dbHandle key" );
      return TCL_ERROR;
    }

  if ( (db = lookup(interp,objv[2],NULL)) == NULL )
    return TCL_ERROR;

  toExternalString( &kstr, objv[3] );

  key.dptr = Tcl_DStringValue( &kstr );
  key.dsize = Tcl_DStringLength( &kstr );

  exists = gdbm_exists( db->db, key );

  Tcl_DStringFree( &kstr );

  Tcl_SetObjResult( interp, Tcl_NewIntObj(exists? 1 : 0) );
  return TCL_OK;
}

/*
 * gdbm delete <db> <key>
 *
 * Deletes entry corresponding to <key> in <db>.  If there is no such
 * <key> an error is reported.
 */
static int 
delete( ClientData client, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[] )
{
  DB *db;
  int ret;
  datum key;
  Tcl_DString kstr;

  if ( objc != 4 )
    {
      Tcl_WrongNumArgs( interp, 2, objv, "dbHandle key" );
      return TCL_ERROR;
    }

  if ( (db = lookup(interp,objv[2],NULL)) == NULL )
    return TCL_ERROR;

  toExternalString( &kstr, objv[3] );

  key.dptr = Tcl_DStringValue( &kstr );
  key.dsize = Tcl_DStringLength( &kstr );

  ret = gdbm_delete( db->db, key );

  Tcl_DStringFree( &kstr );

  if ( ret != 0 )
    {
      Tcl_AppendResult( interp, "gdbm error: ", gdbm_strerror(gdbm_errno), NULL );
      return TCL_ERROR;
    }

  return TCL_OK;
}

/*
 * gdbm list <db>
 *
 * Return a list of all keys of <db>
 */
static int 
list( ClientData client, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[] )
{
  DB *db;
  Tcl_Obj *result ;
  datum key, lastkey;

  if ( objc != 3 )
    {
      Tcl_WrongNumArgs( interp, 2, objv, "dbHandle" );
      return TCL_ERROR;
    }
  
  if ( (db = lookup(interp,objv[2],NULL)) == NULL )
    return TCL_ERROR;

  Tcl_SetObjResult( interp, result = Tcl_NewListObj(0,NULL) );

  key = gdbm_firstkey( db->db );
  if ( key.dptr == NULL )
    return TCL_OK;

  /* 
   * Append the actual key to the result.  This key must be
   * remembered until the next key is found.  key is a variable
   * and does not need to be freed, but key.dptr is allocated by
   * gdbm and must be freed.
   */
  do
    {
      Tcl_ListObjAppendElement( interp, result, Tcl_NewStringObj(key.dptr,key.dsize) );

      lastkey = key;
      key = gdbm_nextkey( db->db, lastkey );
      free( lastkey.dptr );
    }
      while( key.dptr );

  return TCL_OK;
}

/*
 * gdbm firstkey <db>
 * 
 * Used in conjuction with nextkey() to retrieve all keys from
 * <db> in an unspecified order.  Returns the a starting key.
 */
static int 
firstkey( ClientData client, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[] )
{
  DB *db;
  datum key;

  if ( objc != 3 )
    {
      Tcl_WrongNumArgs( interp, 2, objv, "dbHandle" );
      return TCL_ERROR;
    }

  if ( (db = lookup(interp,objv[2],NULL)) == NULL )
    return TCL_ERROR;

  key = gdbm_firstkey( db->db );

  RetDatum( interp, key );
}

/*
 * gdbm nextkey <db> <key> 
 *
 * Used in conjuction with firstkey() to retrieve all keys from <db>
 * in an unspecified order.  Returns the next key given a previous key
 * <key>.  Returns an empty string if there are no more keys.
 */
static int 
nextkey( ClientData client, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[] )
{
  DB *db;
  datum oldkey, newkey;
  Tcl_DString kstr;

  if ( objc != 4 )
    {
      Tcl_WrongNumArgs( interp, 2, objv, "dbHandle key" );
      return TCL_ERROR;
    }

  if ( (db = lookup(interp,objv[2],NULL)) == NULL )
    return TCL_ERROR;
  
  toExternalString( &kstr, objv[3] );

  oldkey.dptr = Tcl_DStringValue( &kstr );
  oldkey.dsize = Tcl_DStringLength( &kstr );
  
  newkey = gdbm_nextkey( db->db, oldkey );

  Tcl_DStringFree( &kstr );

  RetDatum( interp, newkey );
}

/*
 * gdbm open <file> [<mode>] where mode one of r,rw,rwc,rwn
 *
 * Open a gdbm database file <file> and return a handle for further
 * access.  The file is opened using the passed mode, or "r" if no
 * mode is supplied 
 *
 * r   - read only      rwc - read, write, create if non existant
 * rw  - read write     rwn - read, write, new - create always new
 */
static int 
open( ClientData client, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[] )
{
  static char *modes[] ={ "r",        "rw",         "rwc",       "rwn", NULL };
  static int values[] ={  GDBM_READER, GDBM_WRITER, GDBM_WRCREAT, GDBM_NEWDB };
  int mode, new;
  char buf[16];
  DB *db;
  Tcl_HashEntry *entry;
  Tcl_DString filename;

  if ( ! (objc == 3 || objc == 4) )
    {
      Tcl_WrongNumArgs( interp, 2, objv, "filename ?accessMode?" );
      return TCL_ERROR;
    }

  /*
   * check mode string if given and set mode accordingly
   */
  if ( objc == 4 )
    {
      if ( Tcl_GetIndexFromObj(interp,objv[3],modes,"accessMode",0,&mode) != TCL_OK )
	return TCL_ERROR;

      mode = values[mode];
    }
  else
    mode = GDBM_READER;

  /*
   * open gdbm and register it
   */
  sprintf( buf, "gdbm%d", handle++ );

  db = (DB*) ckalloc( sizeof(DB) + strlen(buf)+1 );

  db->name = strcpy( (char*)(db + 1), buf );
  db->mode = mode;
  db->write_mode = GDBM_REPLACE;

  toExternalString( &filename, objv[2] );
  db->db = gdbm_open( Tcl_DStringValue(&filename), 0, mode, 0664, NULL );
  Tcl_DStringFree( &filename );

  if ( db->db == NULL )
    {
      ckfree( (char*)db );

      Tcl_AppendResult( interp, "couldn't open \"", Tcl_GetString(objv[2]), "\": ", gdbm_strerror(gdbm_errno), NULL );
      return TCL_ERROR;
    }

  entry = Tcl_CreateHashEntry( &databases, db->name, &new );
  Tcl_SetHashValue( entry, (ClientData) db );

  Tcl_SetObjResult( interp, Tcl_NewStringObj(db->name,-1) );
  return TCL_OK;
}

/*
 * gdbm error number|text
 *
 * Retrieve current value of gdbm_errno as either a number or a text string.
 */
static int 
error( ClientData client, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[] )
{
  static char *modes[] ={ "number", "text", NULL };
  int mode;

  if ( objc != 3 )
    {
      Tcl_WrongNumArgs( interp, 2, objv, "option" );
      return TCL_ERROR;
    }

  if ( Tcl_GetIndexFromObj(interp,objv[2],modes,"option",0,&mode) != TCL_OK )
    return TCL_ERROR;
 
  switch( mode )
    {
  case 0:
      Tcl_SetObjResult( interp, Tcl_NewIntObj(gdbm_errno) );
      break;
  case 1:
      Tcl_SetObjResult( interp, Tcl_NewStringObj(gdbm_strerror(gdbm_errno),-1) );
      break;
    }
 
 return TCL_OK;
}

/*
 * gdbm writemode <db> replace|insert
 *
 * Set default write mode for subsequent gdbm insert's.
 */
static int 
writemode( ClientData client, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[] )
{
  static char *modes[] ={ "replace",    "insert",   NULL };
  static int values[] ={  GDBM_REPLACE, GDBM_INSERT };
  int mode;
  DB *db ;

  if ( objc != 4 )
    {
      Tcl_WrongNumArgs( interp, 2, objv, "dbHandle option" );
      return TCL_ERROR;
    }

  if ( (db = lookup(interp,objv[2],NULL)) == NULL )
    return TCL_ERROR;

  if ( Tcl_GetIndexFromObj(interp,objv[3],modes,"option",0,&mode) != TCL_OK )
    return TCL_ERROR;
 
  db->write_mode = values[mode];
 
  return TCL_OK;
}

/*
 * gdbm <option> <arg> ...
 *
 * Command dispatcher for Tclgdbm package.
 */
static int 
tclgdbm( ClientData client, Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[] )
{
  static char *cmds[] = {
	"close", "delete", "error", "exists", "fetch", "firstkey", "insert",
	"list", "nextkey", "open", "reorganize", "store", "writemode", NULL };
  static int (*f[])(ClientData,Tcl_Interp*,int,Tcl_Obj*CONST*) = {
	 close,   delete,   error,   exists,   fetch,   firstkey,   insert,
	 list,   nextkey,   open,   reorganize,   insert,  writemode, };
  int cmd;

  if ( objc < 2 )
    {
      Tcl_WrongNumArgs( interp, 1, objv, "option ?arg ...?" );
      return TCL_ERROR;
    }

  if ( Tcl_GetIndexFromObj(interp,objv[1],cmds,"option",0,&cmd) != TCL_OK )
    return TCL_ERROR;
  
  return (*f[cmd])( client, interp, objc, objv );
}


/*
 * Not exported in tcl.h
 */
Tcl_Namespace *Tcl_FindNamespace( Tcl_Interp*, char*, Tcl_Namespace*, int );
int Tcl_Export( Tcl_Interp*, Tcl_Namespace*, char*, int );


/*
 * Tclgdbm_Init
 *
 * Initialization for package.
 */
int
Tclgdbm_Init( Tcl_Interp *interp )
{
  static int initialized = 0;
  Tcl_Namespace *np;

#ifdef USE_TCL_STUBS
  if ( Tcl_InitStubs(interp,"8.1",0) == NULL )
    return TCL_ERROR;
#else
  if ( Tcl_PkgRequire(interp,"Tcl",TCL_VERSION,0) == NULL )
    return TCL_ERROR;
#endif

  Tcl_CreateObjCommand( interp, "tclgdbm::gdbm", tclgdbm, NULL, NULL ); 

  np = Tcl_FindNamespace( interp, "tclgdbm", NULL, TCL_LEAVE_ERR_MSG );

  if ( np == NULL )
    return TCL_ERROR;

  if ( Tcl_Export(interp,np,"*",0) != TCL_OK )
    return TCL_ERROR;

  Tcl_PkgProvide( interp, "Tclgdbm", VERSION );

  if ( ! initialized )
    {
      /* init hash table */
      Tcl_InitHashTable( &databases, TCL_STRING_KEYS );

      initialized = 1;
    }

  return TCL_OK;
}

int
Tclgdbm_SafeInit( Tcl_Interp *interp )
{
  return Tclgdbm_Init(interp);
}
