/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
#include "protoSim.h"

// This installers assumes the "installData" points to a 
// (ProtoSimAgent) instance

bool ProtoSimAgent::TimerInstaller(ProtocolTimerInstallCmd cmd,
                                   double                  delay,
                                   ProtocolTimerMgr*       timerMgr, 
                                   const void*             installData)
{
    switch(cmd)
	{
		case PROTOCOL_TIMER_INSTALL:
            ((ProtoSimAgent*)installData)->InstallTimer(delay);
			break;
			
		case PROTOCOL_TIMER_MODIFY:
            ((ProtoSimAgent*)installData)->ModifyTimer(delay);
			break;
			
		case PROTOCOL_TIMER_REMOVE:
            ((ProtoSimAgent*)installData)->RemoveTimer();
			break;
	}
	return true;
}  // end ProtoSimAgent::TimerInstaller()


bool ProtoSimAgent::SocketInstaller(UdpSocketCmd        cmd,
                                    class UdpSocket*    theSocket,
                                    const void*         installData)
{
    switch(cmd)
    {
        case UDP_SOCKET_INSTALL:
            theSocket->SetUserData(installData);
            break;
            
        case UDP_SOCKET_REMOVE:
            theSocket->SetUserData(NULL);
            break;   
    }
    return true;
}  // end ProtoSimAgent::SocketInstaller() 


UdpSocket::UdpSocket()
    : handle(INVALID_SOCKET), domain(PF_INET),
      port(0), connected(false), owner(NULL),
      recv_handler(NULL), install_func(NULL), install_data(NULL),
      user_data(NULL)
{
    
}

UdpSocket::~UdpSocket()
{
	
}

// Tells simulation agent we wish to open a socket
UdpSocketError UdpSocket::Open(unsigned short thePort, NetworkAddressType /*addressType*/)
{
    // Let optional install_func manipulate the socket first
    // if needed
    if (install_func)
        install_func(UDP_SOCKET_INSTALL, this, install_data);       
    if (((ProtoSimAgent*)install_data)->OpenSocket(this, &thePort))
    {
        port = thePort;
        handle = 0;
        return UDP_SOCKET_ERROR_NONE;
    }
    else
    {
        port = 0;
        handle = INVALID_SOCKET;
        return UDP_SOCKET_ERROR_OPEN;   
    }
}  // end UdpSocket::Open()

void UdpSocket::Close()
{
    if (install_func)
        install_func(UDP_SOCKET_REMOVE, this, install_data);
	((ProtoSimAgent*)install_data)->CloseSocket(this);
	handle = INVALID_SOCKET;
    port = 0;
}  // end UdpSocket::Close()

UdpSocketError UdpSocket::SendTo(const NetworkAddress *dest, char *buffer, 
                                 unsigned int len)
{
	// The socket's agent will send the packet
    ProtoSimAgent* agent = 
        dynamic_cast<ProtoSimAgent*>((ProtoSimAgent*)install_data);
	agent->SendTo(this, dest->SimHostAddr(), dest->Port(), buffer, len);
	return UDP_SOCKET_ERROR_NONE;
}  // end UdpSocket::Sendto()

UdpSocketError UdpSocket::RecvFrom(char*            buffer, 
                                   unsigned int*    buflen, 
                                   NetworkAddress*  sourceAddr)
{
    SIMADDR srcAddr;
    unsigned short srcPort;
    if (((ProtoSimAgent*)install_data)->RecvFrom(this, buffer, buflen,
                                                 &srcAddr, &srcPort))
    {
        sourceAddr->SimSetAddress(srcAddr);
        sourceAddr->SetPort(srcPort);
        return UDP_SOCKET_ERROR_NONE; 
    }
    else
    {
        return UDP_SOCKET_ERROR_RECV;    
    }
}

UdpSocketError UdpSocket::JoinGroup(const NetworkAddress* theAddress, 
                         		    unsigned char ttl)
{
    SetTTL(ttl);
    ((ProtoSimAgent*)install_data)->JoinGroup(this, theAddress->SimHostAddr()); 
	return UDP_SOCKET_ERROR_NONE;
}  // end UdpSocket::JoinGroup()

void UdpSocket::LeaveGroup(const NetworkAddress *theAddress)
{
	((ProtoSimAgent*)install_data)->LeaveGroup(this, theAddress->SimHostAddr()); 
}

UdpSocketError UdpSocket::SetTTL(unsigned char ttl)
{
	((ProtoSimAgent*)install_data)->SetSocketTTL(this, ttl);
	return UDP_SOCKET_ERROR_NONE;
}

UdpSocketError UdpSocket::SetLoopback(bool state)
{
	((ProtoSimAgent*)install_data)->SetSocketLoopback(this, state);
	return UDP_SOCKET_ERROR_NONE;
}

UdpSocketError UdpSocket::SetMulticastInterface(unsigned long /*interfaceAddress*/)
{
    return UDP_SOCKET_ERROR_NONE;
}

UdpSocketError UdpSocket::SetTOS(unsigned char /*tos*/)
{
    return UDP_SOCKET_ERROR_NONE;
}

UdpSocketError UdpSocket::SetReuse(bool /*state*/)
{
    return UDP_SOCKET_ERROR_NONE;
}
