#include "networkAddress.h"
#include "debug.h"   // for print out of warnings, etc
#include "sysdefs.h"  // for MIN() definition

#include <stdlib.h>  // for atoi()
#ifdef UNIX
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <unistd.h>  // for gethostname()
#endif // UNIX


bool NetworkAddress::IsMulticast() const
{
    switch(type)
    {
            case IPv4:
                return ((htonl(0xf0000000) & ((struct sockaddr_in*)&addr)->sin_addr.s_addr)
                            == htonl(0xe0000000));

#ifdef HAVE_IPV6
        case IPv6:
        {
            if (IN6_IS_ADDR_V4MAPPED(&(((struct sockaddr_in6*)&addr)->sin6_addr)))
            {
                return (htonl(0xe0000000) ==
                        (htonl(0xf0000000) &
                         IN6_V4MAPPED_ADDR(&(((struct sockaddr_in6*)&addr)->sin6_addr))));
            }
            else
            {
                return (IN6_IS_ADDR_MULTICAST(&(((struct sockaddr_in6*)&addr)->sin6_addr)) ? true : false);
            }
        }
#endif // HAVE_IPV6

#ifdef SIMULATOR_ADDRESS
            case SIM:
#ifdef NS2
                        return (((struct sockaddr_sim *)&addr)->addr & 
                                0x80000000);
#endif // NS2
#ifdef OPNET  // For now our OPNET "model" uses broadcast address always
            return true;
#endif // OPNET
#endif // SIMULATOR_ADDRESS

            default:
                DMSG(0, "NetworkAddress: IsMulticast(): Unsupported address type!\n");
                return false;
    }
}  // end IsMulticast()

const char* NetworkAddress::HostAddressString() const
{
    switch(type)
    {
#ifdef WIN32
		case IPv4:
        case IPv6:
		{
                static char buffer[64];
				unsigned long bufferSize = 64;
                WSAAddressToString((SOCKADDR *)&addr, sizeof(addr), NULL, buffer, &bufferSize);
                return buffer;
		}
#else  // UNIX
		case IPv4:
            return inet_ntoa(((struct sockaddr_in *)&addr)->sin_addr);

#ifdef HAVE_IPV6
        case IPv6:
        {
            static char buffer[64];
            return inet_ntop(AF_INET6, &((struct sockaddr_in6 *)&addr)->sin6_addr,
                             buffer, 64);
        }
#endif // HAVE_IPV6
#endif // if/else WIN32

#ifdef SIMULATOR_ADDRESS
                case SIM:
                {
                        static char text[32];
                        sprintf(text, "%u", ((struct sockaddr_sim*)&addr)->addr);
                        return text;
                }
#endif // SIMULATOR_ADDRESS

            default:
                DMSG(0, "NetworkAddress: HostAddressString(): Unsupported address type!\n");
                return NULL;
    }  // end switch(type)
}  // end NetworkAddress::HostAddressString()


void NetworkAddress::SetPort(unsigned short thePort)
{
    switch(type)
    {
            case IPv4:
                ((struct sockaddr_in*)&addr)->sin_port = htons(thePort);
                break;

#ifdef HAVE_IPV6
        case IPv6:
            ((struct sockaddr_in6*)&addr)->sin6_port = htons(thePort);
                break;
#endif // HAVE_IPV6

#ifdef SIMULATOR_ADDRESS
                case SIM:
                ((struct sockaddr_sim*)&addr)->port = thePort;
                        break;
#endif // SIMULATOR_ADDRESS

            default:
                DMSG(0, "NetworkAddress::PortSet(): Unsupported address type!\n");
                break;
    }
}  // end NetworkAddress::PortSet()

unsigned short NetworkAddress::Port() const
{
    switch(type)
    {
            case IPv4:
                return ntohs(((struct sockaddr_in *)&addr)->sin_port);

#ifdef HAVE_IPV6
        case IPv6:
            return ntohs(((struct sockaddr_in6 *)&addr)->sin6_port);
#endif // HAVE_IPV6

#ifdef SIMULATOR_ADDRESS
                case SIM:
                return (((struct sockaddr_sim*)&addr)->port);
#endif // SIMULATOR_ADDRESS

            default:
                DMSG(0, "NetworkAddress::Port(): Unsupported address type!\n");
                return 0;  // port 0 is an invalid port
    }
}  // end NetworkAddress::Port()


bool NetworkAddress::SetAddress(const struct sockaddr* theAddr)
{
    switch (theAddr->sa_family)
    {
        case AF_INET:
            memcpy(&addr, theAddr, sizeof(struct sockaddr_in));
            type = IPv4;
            len = 4;
            return true;

#ifdef HAVE_IPV6
        case AF_INET6:
            memcpy(&addr, theAddr, sizeof(struct sockaddr_in6));
            type = IPv6;
            len = 16;
            return true;
#endif // HAVE_IPV6

        default:
            DMSG(0, "NetworkAddress::SetAddress() Invalid address type!\n");
            type = NETWORK_INVALID_ADDRESS;
            return false;
    }
}  // end NetworkAddress:SetAddress()


unsigned long NetworkAddress::IPv4HostAddr() const
{
    switch(type)
    {
        case IPv4:
            return ntohl(((struct sockaddr_in *)&addr)->sin_addr.s_addr);

        default:
            DMSG(0, "mdp: GetIPv4Addr(): Non-IPv4 address!\n");
            return INADDR_NONE;
    }
}  // end NetworkAddress:IPv4HostAddr()


int NetworkAddress::CompareHostAddr(const NetworkAddress* theAddr) const
{
    switch(type)
    {
        case IPv4:
            return memcmp(&((struct sockaddr_in *)&addr)->sin_addr,
                          &((struct sockaddr_in *)&theAddr->addr)->sin_addr,
                          4);
#ifdef HAVE_IPV6
        case IPv6:
            return memcmp(&((struct sockaddr_in6*)&addr)->sin6_addr,
                          &((struct sockaddr_in6*)&theAddr->addr)->sin6_addr,
                          16);
#endif // HAVE_IPV6
            default:
                DMSG(0, "NetworkAddress: CompareHostAddr(): unsupported address type!\n");
                return -1;
     }
}  // end NetworkAddress::CompareHostAddress()

// (TBD) Provide a mechanism for async lookup with optional user interaction

#ifdef HAVE_IPV6
bool NetworkAddress::LookupHostAddress(const char *name)
{
#ifdef SIMULATOR_ADDRESS
    SIMADDR theAddr;
    if (1 == sscanf(name, "%i", &theAddr))
    {
        ((struct sockaddr_sim*)&addr)->addr = theAddr;
        type = SIM;
        len = sizeof(SIMADDR);
        return true;
    }
    else
    {
        fprintf(stderr, "NetworkAddress::LookupHostAddress() Invalid simulator address!\n");
        return false;
    }
#else
   // Use DNS to look it up
    // Get host address, looking up by hostname if necessary
        
#ifdef WIN32
    // Startup WinSock for name lookup
    WSADATA wsaData;
    WORD wVersionRequested = MAKEWORD( 2, 2 );
    int err = WSAStartup( wVersionRequested, &wsaData );
    if ( err != 0 )
    {
            DMSG(0, "Error initializing WinSock!\n");
            return false;
    }
#endif //WIN32
    addrinfo* addrInfoPtr;
    if(0 != getaddrinfo(name, NULL, NULL, &addrInfoPtr)) 
        return false;
#ifdef WIN32
    WSACleanup();
#endif //WIN32

    memcpy((char*)&addr, addrInfoPtr->ai_addr, addrInfoPtr->ai_addrlen);
    if (addrInfoPtr->ai_family == PF_INET)
    {
        type = IPv4;
        len =  addrInfoPtr->ai_addrlen;  // IPv4 addresses are always 4 bytes in length

#ifdef WIN32
        freeaddrinfo(addrInfoPtr);
#endif // WIN32
        return true;
    }
    else if (addrInfoPtr->ai_family == PF_INET6)
    {
        type = IPv6;
        len = addrInfoPtr->ai_addrlen;
#ifdef WIN32
        freeaddrinfo(addrInfoPtr);
#endif //WIN32
        return true;
    }
    else
    {
#ifdef WIN32
        freeaddrinfo(addrInfoPtr);
#endif // WIN32
        DMSG(0, "NetworkAddress: gethostbyname() returned unsupported address family!\n");
        return false; // flase
    }
#endif // !SIMULATOR_ADDRESS
}  // end NetworkAddress::LookupHostAddress (IPv6)

#else

bool NetworkAddress::LookupHostAddress(const char *name)
{
#ifdef SIMULATOR_ADDRESS
    SIMADDR theAddr;
    if (1 == sscanf(name, "%i", &theAddr))
    {
        ((struct sockaddr_sim*)&addr)->addr = theAddr;
        type = SIM;
        len = sizeof(SIMADDR);
        return true;
    }
    else
    {
        fprintf(stderr, "NetworkAddress::LookupHostAddress() Invalid simulator address!\n");
        return false;
    }
#else
    // Use DNS to look it up
    struct sockaddr_in* addrPtr = (struct sockaddr_in*)&addr;
    // Get host address, looking up by hostname if necessary
    if (INADDR_NONE != (addrPtr->sin_addr.s_addr = inet_addr(name)))
    {
            addrPtr->sin_family = AF_INET;   /* user entered dotted decimal IP address */
    }
    else
    {
#ifdef WIN32
        // Startup WinSock for name lookup
        WSADATA wsaData;
        WORD wVersionRequested = MAKEWORD( 2, 0 );
        int err = WSAStartup( wVersionRequested, &wsaData );
        if ( err != 0 )
        {
                DMSG(0, "Error initializing WinSock!\n");
                return false;

        }
#endif // Win32
        struct hostent *hp = gethostbyname(name);
#ifdef WIN32
        WSACleanup();
#endif // WIN32
        if(hp)
        {
            addrPtr->sin_family = hp->h_addrtype;
            memcpy((char*)&addrPtr->sin_addr,  hp->h_addr,  hp->h_length);
        }
        else
        {
            return false;
        }
    }
    if (addrPtr->sin_family == AF_INET)
    {
        type = IPv4;
        len =  4;  // IPv4 addresses are always 4 bytes in length
    return true;
    }
    else
    {
        DMSG(0, "NetworkAddress: gethostbyname() returned unsupported address family!\n");
        return false;
    }
#endif // !SIMULATOR_ADDRESS
}  // end NetworkAddress::LookupHostAddress()
#endif //if/else HAVE_IPV6

#ifdef HAVE_IPV6
bool NetworkAddress::LookupHostName(char* buf, unsigned int maxLen) const
{
#ifdef WIN32
        WSADATA wsaData;
        WORD wVersionRequested = MAKEWORD( 2, 0 );
        int err = WSAStartup( wVersionRequested, &wsaData );
        if ( err != 0 )
        {
                DMSG(0, "Error initializing WinSock!\n");
                return false;
    }
#endif // WIN32
    struct hostent* hp = NULL;
    switch (type)
    {
        case IPv4:
            hp = gethostbyaddr((char*)&(((struct sockaddr_in*)&addr)->sin_addr),
                               4, AF_INET);
            break;

        case IPv6:
            hp = gethostbyaddr((char*)&(((struct sockaddr_in*)&addr)->sin_addr),
                               4, AF_INET6);
         break;

#ifdef SIMULATOR_ADDRESS
        case SIM:
            if (buf) strncpy(buf, HostAddressString(), maxLen);
            return true;
#endif // SIMULATOR_ADDRESS

        default:
            DMSG(0, "NetworkAddress::LookupHostName(): unsupported address type!\n");
            return false;
    }  // end switch(type)
#ifdef WIN32
        WSACleanup();
#endif // WIN32
    if (hp)
    {
        unsigned int nameLen = 0;
        if (buf && maxLen) buf[0] = '\0';
        char** alias = hp->h_aliases;
        // Use first alias by default
        if (alias && *alias && buf)
        {
            strncpy(buf, *alias, maxLen);
            nameLen = strlen(buf);
            nameLen = MIN(nameLen, maxLen);
            alias++;
        }
        // Try to find fully-qualified name (longest alias containing '.')
        while (alias && *alias)
        {
            if (strchr(*alias, '.') && (nameLen < strlen(*alias)))
            {
                if (buf)
                {
                    strncpy(buf, *alias, maxLen);
                    nameLen = strlen(buf);
                    nameLen = MIN(nameLen, maxLen);
                }
                alias++;
            }
        }
        return true;
    }
    else
    {
        return false;
    }
}  // end NetworkAddress::LookupHostName() (IPv6)

#else  // old IPv4-only support

// (TBD) Provide a mechanism for async lookup with optional user interaction
bool NetworkAddress::LookupHostName(char *buf, unsigned int maxLen) const
{

    char* addrPtr = (char*)&((struct sockaddr_in*)&addr)->sin_addr;
    switch (type)
    {
        case IPv4:
        {

            struct hostent *hp = gethostbyaddr(addrPtr, sizeof(struct in_addr), AF_INET);
            if(hp)
            {
                // Try to return a fully-qualified name
                // longest alias with '.' in name
                if (buf && maxLen) buf[0] = '\0';
                unsigned int nameLen = 0;
                char **alias = hp->h_aliases;
                // Use first alias by default
                if (alias && *alias)
                {
                    if (buf && maxLen)
                    {
                        strncpy(buf, *alias, maxLen);
                        nameLen = strlen(buf);
                        nameLen = MIN(nameLen, maxLen);
                    }
                    alias++;
                }
                // Try to find fully-qualified name (longest alias containing '.')
                while(alias && *alias)
                {
                    if (strchr(*alias, '.') && (strlen(*alias) > nameLen))
                    {
                        if (buf && maxLen)
                        {
                            strncpy(buf, *alias, maxLen);
                            nameLen = strlen(buf);
                            nameLen = MIN(nameLen, maxLen);
                        }
                    }
                    alias++;
                }
                return true;
            }
            else
            {
                DMSG(0, "NetworkAddress:: LookupHostName(): DNS lookup failed!\n");
                return false;
            }
        }

#ifdef SIMULATOR_ADDRESS
        case SIM:
            if (buf) strncpy(buf, HostAddressString(), maxLen);
            return true;
#endif // SIMULATOR_ADDRESS

            default:
                DMSG(0, "NetworkAddress: LookupHostName(): unsupported address type!\n");
                return false;
    }
}  // end NetworkAddress::LookupHostName()
//#endif //if/else WIN32
#endif // if/else HAVE_IPV6

bool NetworkAddress::LookupLocalHostAddress(char* buf, unsigned int maxLen)
{
    // Try to get fully qualified host name if possible
    char hostName[256];
    hostName[0] = '\0';
    hostName[255] = '\0';
    if (gethostname(hostName, 255) )
    {
            perror("NetworkAddress::LookupLocalHostAddress(): gethostname() error");
        return false;
    }
    else
    {
        if (LookupHostAddress(hostName))
        {
            // Make sure to get fully qualified name and use its resolved address
            LookupHostName(hostName, 255);
            LookupHostAddress(hostName);
            maxLen = MIN(255, maxLen);
            if (buf) strncpy(buf, hostName, maxLen);
            return true;
        }
        else
        {
            return false;
        }
    }
}  // end NetworkAddress::LookupLocalHostAddress()

bool NetworkAddress::IsEqual(const NetworkAddress* theAddr) const
{
    switch(type)
    {
        case IPv4:
            if ((IPv4 == theAddr->type) &&
                (((struct sockaddr_in*)&addr)->sin_port ==
                 ((struct sockaddr_in*)&(theAddr->addr))->sin_port) &&
                (((struct sockaddr_in*)&addr)->sin_addr.s_addr ==
                 ((struct sockaddr_in*)&(theAddr->addr))->sin_addr.s_addr))
                return true;
            else
                return false;

#ifdef HAVE_IPV6
        case IPv6:
                if ((IPv6 == theAddr->type) &&
                (((struct sockaddr_in6*)&addr)->sin6_port ==
                 ((struct sockaddr_in6*)&(theAddr->addr))->sin6_port) &&
                (((struct sockaddr_in6*)&addr)->sin6_addr.s6_addr ==
                 ((struct sockaddr_in6*)&(theAddr->addr))->sin6_addr.s6_addr))
                return true;
            else
                return false;
#endif // HAVE_IPV6 

#ifdef SIMULATOR_ADDRESS
        case SIM:
            if ((SIM == theAddr->type) &&
                (((struct sockaddr_sim*)&addr)->port ==
                 ((struct sockaddr_sim*)&(theAddr->addr))->port) &&
                (((struct sockaddr_sim*)&addr)->addr ==
                 ((struct sockaddr_sim*)&(theAddr->addr))->addr))
                return true;
            else
                return false;

#endif // SIMULATOR_ADDRESS

            default:
                DMSG(0, "NetworkAddress: IsEqual(): unsupported address type!\n");
                return false;
    }
}  // end NetworkAddress::IsEqual()

bool NetworkAddress::HostIsEqual(const NetworkAddress *theAddr) const
{
    switch(type)
    {
        case IPv4:
            if ((IPv4 == theAddr->type) &&
                (((struct sockaddr_in *)&addr)->sin_addr.s_addr ==
                 ((struct sockaddr_in *)&(theAddr->addr))->sin_addr.s_addr))
                return true;
            else
                return false;
#ifdef HAVE_IPV6
        case IPv6:
            if ((IPv6 == theAddr->type) &&
                (((struct sockaddr_in6*)&addr)->sin6_addr.s6_addr ==
                 ((struct sockaddr_in6*)&(theAddr->addr))->sin6_addr.s6_addr))
                return true;
            else
                return false;
#endif // HAVE_IPV6 

#ifdef SIMULATOR_ADDRESS
        case SIM:
            if ((SIM == theAddr->type) &&
                (((struct sockaddr_sim*)&addr)->addr ==
                 ((struct sockaddr_sim*)&(theAddr->addr))->addr))
                return true;
            else
                return false;
#endif // SIMULATOR_ADDRESS

            default:
                DMSG(0, "NetworkAddress::HostIsEqual(): unsupported address type!\n");
                return false;
    }
}  // end NetworkAddress::HostIsEqual()




PortItem::PortItem(unsigned short min, unsigned short max)
        : port_min(min), port_max(max), prev(NULL), next(NULL)
{
}

bool PortItem::Overlap(unsigned short portMin, unsigned short portMax)
{
        if ( ((portMin <= (port_max+1)) &&
                  (portMax >= (port_min-1))) ||
                 ((port_min <= (portMax+1)) &&
                  (port_max >= (portMin-1))))
                return true;
        else
                return false;
}  // end PortItem::Overlap()

void PortItem::Merge(unsigned short portMin, unsigned short portMax)
{
        port_min = MIN(portMin, port_min);
        port_max = MAX(portMax, port_max);
}  // end PortItem::Merge()

unsigned short PortItem::NextValue(unsigned short index)
{
        if (index)
        {
                if (index >= port_max)
                        return 0;
                else
                        return (index+1);
        }
        else
        {
                return port_min;
        }
}  // end PortItem::NextValue()

PortList::PortList()
        : top(NULL), bottom(NULL)
{
}

PortList::~PortList()
{
        Destroy();
}

void PortList::Destroy()
{
        PortItem* next;
        while ((next = top))
        {
                Remove(next);
                delete next;
        }
} // end PortList::Destroy();

void PortList::Insert(PortItem* theItem)
{
        PortItem* next = top;
        while (next)
        {
                if (theItem->port_min < next->port_min)
                {
                        if ((theItem->prev = next->prev))
                                theItem->prev->next = theItem;
                        else
                                top = theItem;
                        if ((theItem->next = next->next))
                                theItem->next->prev = theItem;
                        else
                                bottom = theItem;
                        Collapse();
                        return;
                }
                else
                {
                        next = next->next;
                }
        }  // end while(next)
        if ((theItem->prev = bottom))
                bottom->next = theItem;
        else
                top = theItem;
        bottom = theItem;
        theItem->next = NULL;
        Collapse();
}  // end PortList::Insert()

void PortList::Append(PortItem* theItem)
{
        if ((theItem->prev = bottom))
                bottom->next = theItem;
        else
                top = theItem;
        bottom = theItem;
        theItem->next = NULL;
}  // end PortList::Append()

void PortList::Remove(PortItem* theItem)
{
        if (theItem->next)
                theItem->next->prev = theItem->prev;
        else
                bottom = theItem->prev;
        if (theItem->prev)
                theItem->prev->next = theItem->next;
        else
                top = theItem->next;
}  // end PortList::Remove()

PortItem* PortList::FindMatch(unsigned short value)
{
        PortItem* next = top;
        while (next)
        {
                if ((value >= next->port_min) &&
                        (value <= next->port_max))
                        return next;
                next = next->next;
        }
        return NULL;
} // end PortList::FindMatch();

PortItem* PortList::FindOverlap(unsigned short portMin, unsigned short portMax)
{
        PortItem* next = top;
        while (next)
        {
                if (next->Overlap(portMin, portMax)) return next;
                next = next->next;
        }  // end while(next);
        return NULL;
}  // end PortList::FindOverlap()

void PortList::Collapse()
{
        PortItem* next = top;
        while (next)
        {
                PortItem* x = next;
                next = next->next;
                if (next)
                {
                        if (next->Overlap(x->port_min, x->port_max))
                        {
                                next->Merge(x->port_min, x->port_max);
                                Remove(x);
                                delete x;
                        }
                }
        }
}  // end PortList::Collapse()

PortPool::PortPool()
{
}

bool PortPool::AddPorts(const char* portText)
{
        unsigned int len = strlen(portText);
        char* text = new char[len+1];
        if (!text)
        {
                DMSG(0, "PortPool::AddPorts() Error allocating memory!\n");
                return false;
        }
        strcpy(text, portText);
        char* start = text;
        while (start)
        {
                char* end = strchr(start, ',');
                if (end) *end++ = '\0';
                char* range = strchr(start, '-');
                if (range)
                        *range++ = '\0';
                else
                        range = start;
                unsigned short portMin = atoi(start);
                unsigned short portMax = atoi(range);
                if (!portMin || !portMax || (portMin > portMax))
                {
                        DMSG(0, "PortPool::AddPorts() Invalid specification error!\n");
                        delete text;
                        return false;
                }
                PortItem* theItem = pool.FindOverlap(portMin, portMax);
                if (theItem)
                {
                        DMSG(0, "Found overlap between: %u-%u and %u-%u\n",
                                 portMin, portMax, theItem->port_min, theItem->port_max);
                        theItem->Merge(portMin, portMax);
                        pool.Collapse();
                }
                else
                {
                        theItem = new PortItem(portMin, portMax);
                        pool.Insert(theItem);
                }
                start = end;
        }  // end while(start);
        delete text;
        return true;
}  // end PortPool::AddPorts()

unsigned short PortPool::GetAvailablePort()
{
        PortItem* next = pool.Top();
        unsigned short candidate = 0;
        while (next)
        {
                candidate = next->NextValue(candidate);
                if (candidate)
                {
                        if (!used.FindMatch(candidate))
                        {
                                PortItem* theItem = new PortItem(candidate, candidate);
                                if (theItem)
                                {
                                        used.Append(theItem);
                                        return candidate;
                                }
                                else
                                {
                                        DMSG(0, "PortPool::GetAvailablePort() Error allocating memory!\n");
                                        return 0;
                                }
                        }
                }
                else
                {
                        next = next->Next();
                }
        }  // end while(next)
        DMSG(0, "PortPool::GetAvailablePort() All assigned ports are in use!\n");
        return 0;
}  // end PortPool::GetAvailablePort();

void PortPool::ReturnPort(unsigned short thePort)
{
        PortItem* theItem = used.FindMatch(thePort);
        if (theItem)
        {
                used.Remove(theItem);
                delete theItem;
        }
}  // end PortPool::ReturnPort()

void PortPool::WriteContent(char* textBuffer, unsigned maxLen)
{
        maxLen--; // leave space for '\0'
        textBuffer[0] = '\0';
        char text[64];
        PortItem* next = pool.Top();
        unsigned int len = 0;
        while (next)
        {
                if (next->port_min == next->port_max)
                        sprintf(text, "%hu", next->port_min);
                else
                        sprintf(text, "%hu-%hu", next->port_min, next->port_max);
                if ((next = next->next))
                {
                        strcat(text, ", ");
                }
                len += strlen(text);
                if (len < maxLen)
                        strcat(textBuffer, text);
                else
                        break;
        }  // end while(next);
}  // end PortPool::Display()



