/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 

#include <sys/types.h>
#include <signal.h>  // for kill() and UNIX signals
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>

#ifdef UNIX
#include <sys/wait.h>
#include <ctype.h>
#include <unistd.h>
#endif // UNIX

#ifdef WIN32
#include <direct.h>
#endif // WIN32

#include "mdp.h"
#include "version.h" // for test code version info (TBD) build 
                     // MdpGetVersion() routine into MDPv2 library
#include "debug.h"   // for generic debugging routines

// For some reason, this isn't in the Solaris C++ headers
#if defined(SOLARIS) || defined(SUNOS)
extern "C"
{
    int gethostname (char *name, int namelen);
}
#endif // SOLARIS   

// Or this for SUNOS
#ifdef SUNOS
char *strerror(int err)
{
   char text[16];
   sprintf(text, "Error %d", err);
   return text;
}
#endif // SUNOS

/******************************
 * CmdMdpApp class implementation
 */

CmdMdpApp::CmdMdpApp()
	: MdpApp(),  
#ifndef WIN32
#ifdef NETSCAPE_SUPPORT
      window_id(0), use_netscape(false),
#endif // NETSCAPE_SUPPORT
      post_processor_id(0)
#else
	lIdInst(0), hConv(NULL), window_id(0), 
	post_processor_handle(NULL)
#endif /// if/else !WIN32
{
     mdp_timer.Init(0.0, 0, (ProtocolTimerOwner *)this, 
                    (ProtocolTimeoutFunc)&CmdMdpApp::OnMdpTimeout);
     tx_interval_timer.Init(0.0, 0, (ProtocolTimerOwner *)this, 
                    (ProtocolTimeoutFunc)&CmdMdpApp::OnTxIntervalTimeout);
}

CmdMdpApp::~CmdMdpApp()
{
   
}

void CmdMdpApp::OnExit()
{
    MdpApp::OnExit();   
    StopTxIntervalTimer();
    if (mdp_timer.IsActive())
        mdp_timer.Deactivate();
    dispatcher.Destroy();
    exit(0);
}  // end CmdMdpApp::OnExit()


bool CmdMdpApp::MdpSocketInstaller(MdpSocketInstallCmd   cmd,
                                   MdpSocketHandle       socketHandle,
                                   MdpInstanceHandle     instanceHandle)
{
    CmdMdpApp* app = (CmdMdpApp*)MdpInstanceGetUserData(instanceHandle);
    if (MDP_SOCKET_INSTALL == cmd)
    {     
        if(!app->dispatcher.AddSocketInput((UdpSocket*)socketHandle)) return false;
    }
    else
    {
        app->dispatcher.RemoveSocketInput((UdpSocket*)socketHandle);
    }
    return true;
}  // end CmdMdpApp::MdpSocketInstaller()

bool CmdMdpApp::MdpTimerInstaller(MdpTimerInstallCmd    cmd, 
                                  double                delay,
                                  MdpTimerHandle        timerHandle, 
                                  MdpInstanceHandle     instanceHandle)
{
    CmdMdpApp* app = (CmdMdpApp*)MdpInstanceGetUserData(instanceHandle);    
    if (MDP_TIMER_MODIFY == cmd)
    {
        ASSERT(app->mdp_timer.IsActive());        
        app->mdp_timer.SetInterval(delay);
        app->mdp_timer.Reset();
        app->mdp_timer_handle = timerHandle;
        return true;
    }
    else if (MDP_TIMER_INSTALL == cmd)
    {
        ASSERT(!app->mdp_timer.IsActive());
        app->mdp_timer_handle = timerHandle;
        app->mdp_timer.SetInterval(delay);
        app->dispatcher.InstallTimer(&app->mdp_timer);
        return true;
    }
    else  // PROTOCOL_TIMER_REMOVE
    {
        app->mdp_timer.Deactivate();
        return true;
    }
}  // end CmdMdpApp::MdpTimerInstaller()


void CmdMdpApp::MdpSocketHandler(EventDispatcher::Descriptor /*descriptor*/, 
                                 EventDispatcher::EventType  /*eventType*/, 
                                 const void*                 userData)
{
    MdpSocketOnDataReady((MdpSocketHandle)userData);
}  // end CmdMdpApp::MdpSocketHandler()

bool CmdMdpApp::OnMdpTimeout()
{
    mdp_timer.Deactivate();
    MdpTimerOnTimeout(mdp_timer_handle);
    return false;
} // end CmdMdpApp::OnMdpTimeout()

void CmdMdpApp::StartTxIntervalTimer(double theDelay)
{
    ASSERT(!tx_interval_timer.IsActive());
    tx_interval_timer.SetInterval(theDelay);
    dispatcher.InstallTimer(&tx_interval_timer);
}  // end CmdMdpApp::StartTxIntervalTimer()

void CmdMdpApp::StopTxIntervalTimer()
{
    if (tx_interval_timer.IsActive())
        tx_interval_timer.Deactivate(); 
}  // end CmdMdpApp::StopTxIntervalTimer()

bool CmdMdpApp::OnTxIntervalTimeout()
{
    tx_interval_timer.Deactivate();
    MdpApp::OnTxIntervalTimeout();
    return false;  // tx_interval_timer gets re-installed as needed
}  // end CmdMdpApp::OnTxIntervalTimeout()

void CmdMdpApp::SetPostProcessor(char* theCmd)
{
    MdpApp::SetPostProcessor(theCmd); 
#ifndef WIN32
#ifdef NETSCAPE_SUPPORT     
    // See if it's Netscape post processing 
    // for which we provide special support
    CheckForNetscape();
#endif // NETSCAPE_SUPPORT
#endif // !WIN32
}  // end int CmdMdpApp::SetPostProcessor()




#ifndef WIN32

#ifdef NETSCAPE_SUPPORT
void CmdMdpApp::CheckForNetscape()
{
    // See if it's Netscape post processing 
    // for which we provide special support
    char txt[8];
    const char *ptr = strrchr(PostProcessor(), DIR_DELIMITER);
    if (ptr)
        ptr++;
    else
        ptr = PostProcessor();
    strncpy(txt, ptr, 8);
    for (int i=0; i<8; i++) txt[i] = toupper(txt[i]);
    if (!strncmp(txt, "NETSCAPE", 8)) 
        SetNetscape(true);
    else
        SetNetscape(false);
}  // end CmdMdpApp::CheckForNetscape()
#endif // NETSCAPE_SUPPORT

void CmdMdpApp::PostProcess(const char *path)
{
    const char* myArgs[32];
    
#ifdef NETSCAPE_SUPPORT  
    // Special support for finding a Netscape window and
    // use the openURL remote command to display received file
#define USE_ACTIVE_WINDOW 0xffffffff  
    char wid_text[32], url_text[PATH_MAX+64];
    if (use_netscape)
    {
        int i = 0;
        myArgs[i++] = PostProcessor();
        Window w = 0;
        // (TBD) We could put in a hack to look for the old
        // window if we made "w" static when "window_id" = 0xffffffff
        // This would help us capture the proper window for file display
        // It's still a hack but it might work a little better without
        // too much trouble
        if (NetscapeIsRunning(&w))
        {       
            // Use -remote command to display file in current
            // or new window
            if (window_id)
            {
                if (NetscapeCheckWindow(window_id))
                {
                    // Use the same window as last time
                    myArgs[i++] = "-id";
                    sprintf(wid_text, "0x%lx", window_id);
                    myArgs[i++] = wid_text;
                    myArgs[i++] = "-noraise";
                    myArgs[i++] = "-remote";
                    sprintf(url_text, "openURL(file://%s)", path);
                    myArgs[i++] = url_text;
                }
                else if (USE_ACTIVE_WINDOW == window_id)
                {
                    // Capture the open window we found
                    myArgs[i++] = "-id";
                    sprintf(wid_text, "0x%lx", w);
                    myArgs[i++] = wid_text;
                    window_id = w;
                    myArgs[i++] = "-noraise"; 
                    myArgs[i++] = "-remote";
                    sprintf(url_text, "openURL(file://%s)", path);
                    myArgs[i++] = url_text;         
                }
                else  // user must have closed old window so open another
                {
                    window_id = USE_ACTIVE_WINDOW;
                    //myArgs[i++] = "-raise";
                    myArgs[i++] = "-remote";
                    sprintf(url_text, "openURL(file://%s,new-window)", path);
                    myArgs[i++] = url_text;
                }                
                
            }
            else
            {
                // We're starting fresh, open a new window 
                window_id = USE_ACTIVE_WINDOW;
                //myArgs[i++] = "-raise";
                myArgs[i++] = "-remote";
                sprintf(url_text, "openURL(file://%s,new-window)", path);
                myArgs[i++] = url_text;
            }
        }
        else
        {
            window_id = USE_ACTIVE_WINDOW;
            myArgs[i++] = path;
        }       
        myArgs[i] = NULL;
    }
    else
#endif // NETSCAPE_SUPPORT
    {
        if (post_processor_id) KillPostProcessor();
        int i=0, j=0;
        myArgs[i++] = PostProcessor();
        while (ProcessorOpt(j))
            myArgs[i++] = ProcessorOpt(j++);
        myArgs[i++] = path;
        myArgs[i++] = NULL;
    }  // end if/else (use_netscape)
        
    switch((post_processor_id = fork()))
    {
        case -1:    // error
            fprintf(stderr, "mdp: PostProcess fork() error: %s\n",
                strerror(errno));
            post_processor_id = 0;
            return;

        case 0:     // child
            if (execvp((char*)myArgs[0], (char**)myArgs) < 0)
            {
		        fprintf(stderr, "mdp:  PostProcess execvp() error: %s\n",
                        strerror(errno));
                exit(-1);
            }
            break;

        default:    // parent
            break;
    }
}  // end CmdMdpApp::PostProcess()

void CmdMdpApp::KillPostProcessor()
{
    int status;
    int count = 0;
    if (!post_processor_id) return;
    while((kill(post_processor_id, SIGTERM) !=0) && count < 10)
	{ 
	    if (errno == ESRCH) break;
	    count++;
	    fprintf(stderr, "mdp: KillPostProcessor kill() error: %s\n",
                strerror(errno));
	}
	count = 0;
	while((waitpid(post_processor_id, &status, 0) 
	         != post_processor_id) && count < 10)
	{
	    if (errno == ECHILD) break;
	    count++;
	    fprintf(stderr, "mdp: KillPostProcessor wait() error: %s\n",
                strerror(errno));
	}
	post_processor_id = 0;
}  // end CmdMdpApp::KillPostProcessor()

#else

void CmdMdpApp::PostProcess(const char *path)
{
	if (!hConv)
	{
		HSZ szServerName = ::DdeCreateStringHandle(lIdInst, 
										PostProcessor(),
										CP_WINANSI);
		HSZ szTopicName = ::DdeCreateStringHandle(lIdInst, 
									"WWW_OpenURL",
									CP_WINANSI);
		hConv = ::DdeConnect(lIdInst, szServerName, szTopicName, NULL);
		DdeFreeStringHandle(lIdInst, szTopicName);
		DdeFreeStringHandle(lIdInst, szServerName);
	}

	if (hConv)
	{
		char dde_text[PATH_MAX+128];
		sprintf(dde_text, "\"file://%s\",,0x%08x,0x0,,,NETREPORT",
				path, window_id);
		HSZ  szParams = 
			::DdeCreateStringHandle(lIdInst, dde_text, CP_WINANSI);
		if(DdeClientTransaction(NULL, 0, hConv, szParams, CF_TEXT,
			XTYP_REQUEST, TIMEOUT_ASYNC, &transaction_id))
		{
			DdeFreeStringHandle(lIdInst, szParams);
			return;
		}
		else
		{	
			DdeFreeStringHandle(lIdInst, szParams);
			DdeDisconnect(hConv);
			hConv = NULL;
			fprintf(stderr, "winMdp PostProcessor DDE Error: %s exited!?", PostProcessor());
		}
	}

	// DDE post processor evidently not running
	// Launch post processor with shell command
	// For now we assume the DDE server name & command name are the 
	// same and that the post processor command executable is in
	// the PATH
	window_id = 0xffffffff;  // so next DDE uses active window

	char args[PATH_MAX+512];
	const char* ptr = ProcessorOpt(0);
	if (ptr)
	{
		strcpy(args, ptr);
		int i = 1;
		while((ptr = ProcessorOpt(i++)))
		{
			strcat(args, " ");
			strcat(args, ptr);
		}
		strcat(args, " ");
		strcat(args, path);
	}
	else
	{
		strcpy(args, path);
	}
	


	// Kill old PostProcessor first
	if (post_processor_handle)
	{
		DMSG(0, "Terminating %p ...\n", post_processor_handle);
		if (!TerminateProcess(post_processor_handle, 0))
			DMSG(0, "mdp: TerminateProcess() error ...\n");
		CloseHandle(post_processor_handle);
	}

	DMSG(0, "Launching %s ...\n", PostProcessor());
	char buffer[PATH_MAX];
	char* cd = _getcwd(buffer, PATH_MAX);
	if (cd) DMSG(0, "Current DIR = %s\n", cd);

	SHELLEXECUTEINFO exeInfo;
	exeInfo.cbSize = sizeof(SHELLEXECUTEINFO);
	exeInfo.fMask = SEE_MASK_NOCLOSEPROCESS;
    exeInfo.hwnd = NULL; 
    exeInfo.lpVerb = NULL; 
    exeInfo.lpFile = PostProcessor(); 
    exeInfo.lpParameters = args; 
    exeInfo.lpDirectory = NULL;
    exeInfo.nShow = SW_SHOW;
	//if (!ShellExecute(NULL, "open", PostProcessor(), args, NULL, SW_SHOW))
	if (!ShellExecuteEx(&exeInfo))
	{
		if ((int)exeInfo.hInstApp <= 32) Alert("LESS THAN 32");
		post_processor_handle = NULL;
		fprintf(stderr, "Error launching post processor!\n\"%s %s\"", PostProcessor(), args);
	}
	else
	{
		if ((int)exeInfo.hInstApp <= 32) Alert("LESS THAN 32");
		post_processor_handle = exeInfo.hProcess;
		DMSG(0, "   (handle = %p)\n", post_processor_handle);
	}
}  // end CmdMdpApp::PostProcess()

#endif // if/else !WIN32



// Our application instance
CmdMdpApp theApp;

void CmdMdpApp::SignalHandler(int signum)
{
    switch(signum)
    {
#ifdef UNIX
        // SIGCHLD handler to prevent post processor zombies
        case SIGCHLD:
		{
			int status;
            // See if PostProcessor exited itself
            if (wait(&status) == theApp.post_processor_id)
                theApp.post_processor_id = 0;
            // Re-install signal handler (some systems require it)
            signal(SIGCHLD, SignalHandler);
            break;
		}
#endif // UNIX
            
        case SIGINT:
        case SIGTERM:
            theApp.OnExit();
            break;
        
        default:
             fprintf(stderr, "mdp: Caught unexpected signal: %d\n", signum);
            break;
    }       
}  // end CmdMdpApp::SignalHandler()


#if defined(WIN32) && !defined(_CONSOLE)
int PASCAL WinMain(HINSTANCE instance, HINSTANCE prevInst, LPSTR cmdline, int cmdshow)
#else
int main(int argc, char* argv[])
#endif
{	    
    // Initialize MDP file push application base class
    theApp.Init(CmdMdpApp::MdpTimerInstaller,
                CmdMdpApp::MdpSocketInstaller);
    
    // Set some default post processor for unix 
    //(TBD - use "BROWSER" environment variable?)
    theApp.SetPostProcessor("xv -rmode 5 -root -quit");
        
    // Parse command line options (use "getopt()" to pick up option flags)
	int i = 1;
	while (i < argc)
	{
		
	}

    extern char *optarg;
    extern int optind;
    int c;
    while ((c = getopt(argc, argv, MDPAPP_CMD_LINE_FLAGS)) != EOF) 
    {
        if (!theApp.ProcessCommand(c, optarg))
        {
            exit(-1);
        }                    
    }
    argc -= optind;
    //char **orig_argv = argv;
    argv += optind;
    
    // Assume remaining arguments are the server's tx file list
    for (int i = 0; i < argc; i++)
        theApp.ProcessCommand(0, argv[i]); 





#ifdef NETSCAPE_SUPPORT    
    // See if the command-line set Netscape as post-processor
    // to invoke special handling
    theApp.CheckForNetscape();
#endif // NETSCAPE_SUPPORT
       
    // SIGCHLD handler to prevent zombie post processor processes
    signal(SIGCHLD, CmdMdpApp::SignalHandler);       
    // SIGTERM/SIGINT (CTRL-C) handler to allow for (hopefully) graceful shutdown
    signal(SIGTERM, CmdMdpApp::SignalHandler);
    signal(SIGINT, CmdMdpApp::SignalHandler);
    
    // Start mdpApp server/client participation
    if (!theApp.Start()) exit(-1);

    // Enter the dispatcher main loop
    theApp.Run();
    return 0;
}  // end main()




