/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>  // for PATH_MAX
#include <string.h>

#define VERSION "2.3"

#define MIN(X,Y) ((X<Y)?X:Y)
#define MAX(X,Y) ((X>Y)?X:Y)

const int MAX_LINE = 256;

class FastReader
{
    public:
        FastReader();
        bool Read(FILE* filePtr, char* buffer, unsigned int* len);
        bool Readline(FILE* filePtr, char* buffer, unsigned int* len);
    
    private:
        char         savebuf[256];
        char*        saveptr;
        unsigned int savecount;
};  // end class FastReader

class Flow
{
    friend class FlowList;
    
    public:
        Flow();
        ~Flow();
        void PrintDescription(FILE* f);
        const char* Type() {return type;}
        bool SetType(const char* theType);
        int SrcAddr() {return src_addr;}
        void SetSrcAddr(int value) {src_addr = value;}
        int SrcPort() {return src_port;}
        void SetSrcPort(int value) {src_port = value;}
        int DstAddr() {return dst_addr;}
        void SetDstAddr(int value) {dst_addr = value;}
        int DstPort() {return dst_port;}
        void SetDstPort(int value) {dst_port = value;}
        
        bool IsComposite();
        
        bool TypeMatch(const char* theType)
        {
            if (theType && type)
                return (0 == strcmp(theType, type));
            else
                return (theType == type);
        }
        
        bool Match(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort);
        bool ExactMatch(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort);
        
        unsigned long Bytes() {return byte_count;}
        void AddBytes(unsigned long count) 
        {
            byte_count += count;
            UpdateAverage();
        }
        void SubtractBytes(unsigned long count)
        {
            byte_count -= count;
            UpdateAverage();
        }       
        void Reset()
        {
            byte_count = 0;
            average = 0.0;
        }
        double Average() {return average;}
        
        
        void SetSmoothingFactor(double factor) 
        {smoothing_factor = (factor > 1.0) ? 1.0/factor : 1.0;}
        void UpdateAverage()
        {
            double err = ((double)byte_count) - average;
            average += smoothing_factor * err;
        }
        Flow* Next() {return next;}
        
            
    private:
        char* type;
        int   type_len;
        
        int   src_addr;
        int   src_port;
        int   dst_addr;
        int   dst_port;
        
        // Byte count accumulator  
        // allow negative for debug
        unsigned long   byte_count;
        
        double  smoothing_factor;
        double  average;
            
        Flow* prev;
        Flow* next;  
}; // end class Flow

class FlowList
{
    public:
        FlowList();
        ~FlowList();
        void Destroy();
        void Append(Flow* theFlow);
        void Remove(Flow* theFlow);
        Flow* FindFlowByMatch(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort);
        Flow* Head() {return head;}
    
    private:
        Flow* head;
        Flow* tail;
    
};


Flow::Flow()
    : type(NULL), type_len(0), 
      src_addr(-1), src_port(-1), dst_addr(-1), dst_port(-1),
      byte_count(0), smoothing_factor(1.0), average(0.0),
      prev(NULL), next(NULL)
{
}

Flow::~Flow()
{
    if (type) delete []type;
}

bool Flow::SetType(const char* theType)
{
    if (type) delete []type;
    int len = strlen(theType) + 1;
    if(!(type = new char[len]))
    {
        perror("qplot: Error allocating flow type storage");
        return false;
    }
    strcpy(type, theType);
    type_len = len - 1;
    return true;
}  // end Flow::SetName()

bool Flow::IsComposite()
{
    if ((NULL != type) || 
        (src_addr != -1) || (src_port != -1) ||
        (dst_addr >= 0) || (dst_port >= 0))
    {
        return false;   
    }
    else
    {
        return true;
    }
}  // end Flow::IsComposite()

bool Flow::Match(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort)
{
    if ((type && !TypeMatch(theType)) ||    
        ((dst_port >= 0) && (dst_port != dstPort)) ||
        ((dst_addr != -1) && (dst_addr != dstAddr)) ||
        ((src_port >= 0) && (src_port != srcPort)) ||
        ((src_addr != -1) && (src_addr != srcAddr)))
    {
        return false;
    }
    else
    {
        return true;
    }
}  // end Flow::Match()

bool Flow::ExactMatch(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort)
{
    if ((!TypeMatch(theType)) ||     
        (dst_port != dstPort) ||
        (dst_addr != dstAddr) ||
        (src_port != srcPort) ||
        (src_addr != srcAddr))
    {
        return false;
    }
    else
    {
        return true;
    }
}  // end Flow::ExactMatch()

void Flow::PrintDescription(FILE* f)
{
    if (type)
        fprintf(f, "%s:", type);
    else
        fprintf(f, "*:");
    if (src_addr == -1)
        fprintf(f, "*.");
    else
        fprintf(f, "%d.", src_addr);
    if (src_port < 0)
        fprintf(f, "*->");
    else
        fprintf(f, "%d->", src_port);
    if (dst_addr == -1)
        fprintf(f, "*.");
    else
        fprintf(f, "%d.", dst_addr);
    if (dst_port < 0)
        fprintf(f, "*");
    else
        fprintf(f, "%d", dst_port);
}  // end Flow::PrintDescription()

FlowList::FlowList()
    : head(NULL), tail(NULL)
{
}

FlowList::~FlowList()
{
    Destroy();
}

void FlowList::Append(Flow* theFlow)
{
    if ((theFlow->prev = tail))
        theFlow->prev->next = theFlow;
    else
        head = theFlow;
    theFlow->next = NULL;
    tail = theFlow;
}  // end FlowList::Append()


void FlowList::Remove(Flow* theFlow)
{
    if (theFlow->prev)
        theFlow->prev->next = theFlow->next;
    else
        head = theFlow->next;
    
    if (theFlow->next)
        theFlow->next->prev = theFlow->prev;
    else
        tail = theFlow->prev;
}  // end FlowList::Remove()
        
Flow* FlowList::FindFlowByMatch(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort)
{
    Flow* next_flow = head;
    while (next_flow)
    {
        if (next_flow->ExactMatch(theType, srcAddr, srcPort, dstAddr, dstPort))
            return next_flow;
        next_flow = next_flow->next;
    }
    return NULL;   
}  // end FlowList:FindFlowByName()

void FlowList::Destroy()
{
    Flow* next;
    while ((next = head))
    {
        Remove(next);
        delete next;
    }   
}  // end Destroy()


const char WILDCARD = 'X';

enum QueueEvent {QUEUE_EVENT_INVALID, QUEUE_EVENT_ENQUEUE, QUEUE_EVENT_DEQUEUE , QUEUE_EVENT_DROP};

inline void usage()
{
    fprintf(stderr, "Usage: qplot [version][gif][post][raw][auto][len <protoNameLen>]\n"
                    "             [smooth <factor>][flow <type,srcAddr.port,dstAddr.port>]\n"
                    "             link <srcNode,dstNode> trace <traceFile> [output <outputFile>]\n");
    fprintf(stderr, "          (Wildcard type/address/port parameters with 'X')\n");
}
        
int main(int argc, char* argv[])
{ 
    FlowList flowList;
    double windowSize = 1.0;
    char* trace_file;
    char* outputFile = NULL;
    bool use_gnuplot = true;
    bool make_gif = false;
    bool make_post = false;
    int link_src = 0;
    int link_dst = 0;
    double smoothingFactor = 1.0;
    
    bool wildcard = true;
    bool auto_detect = false;
    unsigned int detect_proto_len = 31;
    
    if (argc < 2)
    {
        usage();
        exit(-1);
    }
    
    fprintf(stderr, "qplot Version %s\n", VERSION);
            
    // Parse command line
    fprintf(stderr, "\n");
    int i = 1;
    while(i < argc)
    {
        if (!strcmp("link", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "qplot: Insufficient \"link\" arguments!\n");
                usage();
                exit(-1);
            }
            if (2 != sscanf(argv[i++], "%d,%d", &link_src, &link_dst))
            {
                fprintf(stderr, "qplot: Error parsing \"link\" description!\n");
                usage();
                exit(-1);
            }
        } 
        else if (!strcmp("smooth", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "qplot: Insufficient \"smooth\" arguments!\n");
                usage();
                exit(-1);
            }
            float value;
            if (1 != sscanf(argv[i++], "%f", &value))
            {
                perror("qplot: Error parsing smoothing factor value");
                usage();
                exit(-1);   
            }
            smoothingFactor = value;
        }
        else if (!strcmp("flow", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "qplot: Insufficient \"flow\" arguments!\n");
                usage();
                exit(-1);
            }
            // Pull out flow type, checking for wildcard
            Flow* theFlow = new Flow;
            if (!theFlow)
            {
                perror("qplot: Error allocating memory for flow");
                exit(-1);
            }
            char* flow_info = strtok(argv[i++], ",");
            if (flow_info) 
            {
                if(WILDCARD != flow_info[0])
                    theFlow->SetType(flow_info);
            }
            else
            {
                fprintf(stderr, "qplot: Error parsing \"flow\" description!\n");
                usage();
                exit(1);
            }
            // Pull out source addr/port, checking for wildcards
            flow_info = strtok(NULL, ",");
            if (flow_info)
            {
                // Parse source address/port
                char* ptr = strchr(flow_info, '.');
                if (ptr) 
                {
                    *ptr++ = '\0';
                    if (WILDCARD != ptr[0])
                        theFlow->SetSrcPort(atoi(ptr));
                }
                if (WILDCARD != flow_info[0])
                    theFlow->SetSrcAddr(atoi(flow_info));
            
                // Pull out destination addr/port, checking for wildcards
                flow_info = strtok(NULL, ",");
                if (flow_info)
                {
                    // Parse source address/port
                    char* ptr = strchr(flow_info, '.');
                    if (ptr) 
                    {
                        *ptr++ = '\0';
                        if (WILDCARD != ptr[0])
                            theFlow->SetDstPort(atoi(ptr));
                    }
                    if (WILDCARD != flow_info[0])
                       theFlow->SetDstAddr(atoi(flow_info));
                }
            }
            flowList.Append(theFlow);
            fprintf(stderr, "qplot: Adding flow: ");
            theFlow->PrintDescription(stderr);
            fprintf(stderr, "\n");
            wildcard = false;
        } 
        else if (!strcmp("auto", argv[i]))
        {
            i++;
            auto_detect = true;
        }     
        else if (!strcmp("len", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "qplot: Insufficient \"auto\" arguments!\n");
                usage();
                exit(-1);
            }   
	        detect_proto_len = atoi(argv[i++]);
        }    
        else if (!strcmp("raw", argv[i]))
        {
            i++;
            use_gnuplot = false;
        } 
        else if (!strcmp("gif", argv[i]))
        {
            i++;
            make_gif = true;
        }  
        else if (!strcmp("post", argv[i]))
        {
            i++;
            make_post = true;
        } 
        else if (!strcmp("trace", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "qplot: Insufficient \"trace\" arguments!\n");
                usage();
                exit(-1);
            }
            trace_file = argv[i++];
        }
        else if (!strcmp("version", argv[i]))
        {
            exit(0);   
        }
        else if (!strcmp("output", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "qplot: Insufficient \"output\" arguments!\n");
                usage();
                exit(-1);
            }
            outputFile = argv[i++];
        }
        else
        {
            
        }
    }   
    
    if (wildcard)
    {
        Flow* theFlow = new Flow;
        if (!theFlow)
        {
            perror("qplot: Error creating wildcard composite flow");
            exit(-1);   
        }   
        flowList.Append(theFlow);
    }
    
    
    if (1.0 != smoothingFactor)
    {
        Flow* nextFlow = flowList.Head();
        while (nextFlow)
        {
            nextFlow->SetSmoothingFactor(smoothingFactor);
            nextFlow = nextFlow->Next();
        }
    }
    // Open ns-2 trace input file
    FILE* infile;
    if (trace_file)
    {
        if(!(infile = fopen(trace_file, "r")))
        {
            perror("qplot: Error opening input trace file");
            usage();
            exit(-1);
        }
    }
    else
    {
        fprintf(stderr, "qplot: No \"trace\" file provided!\n");
        usage();
        exit(-1);
    }
    
    // Open output file
    FILE* outfile;
    char temp_file[PATH_MAX];
    if (outputFile)
    {
	    strcpy(temp_file, outputFile);
	    if (use_gnuplot) strcat(temp_file, ".tmp");
        if(!(outfile = fopen(temp_file, "w+")))
        {
            perror("qplot: Error opening output file");
            usage();
            exit(-1);
        } 
    }
    else  // raw output to stdout
    {
        outputFile = "stdout";
        outfile = stdout;  
        use_gnuplot = false;
    }
    
    
    
    
    char buffer[MAX_LINE];
    unsigned int len = MAX_LINE;
    unsigned int line = 0;
    
    
    double windowStart = 0.0;
    double windowEnd = windowStart + windowSize;
    bool validWindow = false;
    
    FastReader reader;
    float theTime = 0.0;
    while (reader.Readline(infile, buffer, &len))
    {
        line++;
        char proto[32], junkText[16];
        unsigned int linkSrc, linkDst, pktSize, flowId;
        int srcAddr, srcPort, dstAddr, dstPort;
        
        if (len)
        {
            QueueEvent eventType = QUEUE_EVENT_INVALID;
            switch(buffer[0])
            {
                case '+':
                    eventType = QUEUE_EVENT_ENQUEUE;
                    break;
                    
                case '-':
                    eventType = QUEUE_EVENT_DEQUEUE;
                    break;
                    
                case 'd':
                    eventType = QUEUE_EVENT_DROP;
                    break;
                    
                default:
                    break; 
            }
            if (QUEUE_EVENT_INVALID != eventType)
            {
                if (11 == sscanf(&buffer[1], "%f %u %u %32s %u %16s %d %d.%d %d.%d", &theTime, 
                                              &linkSrc, &linkDst, proto, &pktSize, junkText, 
                                              &flowId, &srcAddr, &srcPort, &dstAddr, &dstPort))
                {
                    proto[31] = '\0';
                    unsigned int protoLen = MIN(strlen(proto), 32);
                    protoLen = MIN(detect_proto_len, protoLen);
                    proto[protoLen] = '\0';
                                
                    // Is this line on the our link?
                    if ((linkSrc == link_src) && (linkDst == link_dst))
                    {
                        bool plot = false;
                        // Do we already know this flow?
                        bool match = false;
                        Flow* nextFlow = flowList.Head();
                        while (nextFlow)
                        {
                            if (nextFlow->Match(proto, srcAddr, srcPort, dstAddr, dstPort))
                            {
                                if (!nextFlow->IsComposite()) match = true;
                                switch(eventType)
                                {
                                    case QUEUE_EVENT_ENQUEUE:
                                        nextFlow->AddBytes(pktSize);
                                        break;
                                        
                                    case QUEUE_EVENT_DEQUEUE:
                                        plot = true;
                                        nextFlow->SubtractBytes(pktSize);
                                        break;
                                        
                                    case QUEUE_EVENT_DROP:
                                        nextFlow->SubtractBytes(pktSize);
                                        break;
                                        
                                    default:
                                        break;
                                }
                            }
                            nextFlow = nextFlow->Next();
                        }

                        if (auto_detect && !match)
                        {
		    	            Flow* theFlow = new Flow;
                            if (theFlow)
                            {
                                theFlow->SetType(proto);
                                theFlow->SetSrcAddr(srcAddr);
                                theFlow->SetSrcPort(srcPort);
                                theFlow->SetDstAddr(dstAddr);
                                theFlow->SetDstPort(dstPort);
                                flowList.Append(theFlow);
			                    fprintf(stderr, "qplot: At time %f - Adding flow: ", theTime);;
        	                            theFlow->PrintDescription(stderr);
        	                            fprintf(stderr, "\n");
                                switch(eventType)
                                {
                                    case QUEUE_EVENT_ENQUEUE:
                                        theFlow->AddBytes(pktSize);
                                        break;
                                        
                                    case QUEUE_EVENT_DEQUEUE:
                                        plot = true;
                                        theFlow->SubtractBytes(pktSize);
                                        break;
                                        
                                    case QUEUE_EVENT_DROP:
                                        theFlow->SubtractBytes(pktSize);
                                        break;
                                        
                                    default:
                                        break;
                                }
                            }  
                            else
                            {
                                perror("qplot: Error allocating memory for new Flow");
                                fclose(infile);
                                fclose(outfile);
                                exit(0);   
                            }                      
                        }
                        if (plot)
                        {
                            nextFlow = flowList.Head();
                            fprintf(outfile, "%7.3f", theTime);
                            while(nextFlow)
                            {
                                 fprintf(outfile, ", %f", nextFlow->Average());
                                 nextFlow = nextFlow->Next();  
                            }
                            fprintf(outfile, "\n");
                            plot = false;
                        }
                    }  // end if ((linkSrc == link_src) && (linkDst == link_dst))
                }
                else
                {
                    fprintf(stderr, "qplot: Error parsing ns-2 trace file at line %u.\n", line);
                }  // end if/else (sscanf)         
            }  // end if (QUEUE_EVENT_INVALID != eventType)
        }  // end if(len)
        len = MAX_LINE;
    }  // end while(reader.Readline())
    
     fclose(infile);
     
    // Plot final state
    Flow* nextFlow = flowList.Head();
    if (nextFlow)
    {
        fprintf(outfile, "%7.3f", theTime);
        while(nextFlow)
        {
             fprintf(outfile, ", %f", nextFlow->Average());
             nextFlow = nextFlow->Next();  
        }
        fprintf(outfile, "\n");
    }
   
    fclose(outfile);
    
    // Create final output file with gnuplot header if applicable
    if (use_gnuplot && flowList.Head())
    {
    	if (!outputFile)
        {
            fprintf(stderr, "qplot: Must specify output file name for gnuplot file generation!\n");
            exit(-1);
        }
    	if(!(outfile = fopen(outputFile, "w+")))
	    {
	        perror("qplot: Error opening output file");
	        exit(-1);
	    }	
        if (make_post)
        {
            char postName[256];
            strcpy(postName, outputFile);
            strcat(postName, ".ps");
            fprintf(outfile, "set term post color solid\n");
            fprintf(outfile, "set output '%s'\n", postName); 
        }
        else if (make_gif)
        {           
            char gifName[256];
            strcpy(gifName, outputFile);
            strcat(gifName, ".gif");
            fprintf(outfile, "set term gif\n");
            fprintf(outfile, "set output '%s'\n", gifName);   
        }
        fprintf(outfile, "set title '%s'\n", outputFile);
        fprintf(outfile, "set xlabel 'Time (sec)'\n");
        fprintf(outfile, "set ylabel 'Queue Depth (bytes)'\n");
        fprintf(outfile, "set data style lines\n");
        fprintf(outfile, "set key bottom right\n");
        fprintf(outfile, "set noy2tics\n");
        fprintf(outfile, "set ytics mirror\n");
        fprintf(outfile, "plot ");
        nextFlow = flowList.Head();
        unsigned int x = 2;
        while (nextFlow)
        {
            if (x > 2) 
                fprintf(outfile, ",\\\n'%s' index 1 using 1:%u t '",
                              outputFile, x);
            else
                fprintf(outfile, "\\\n'-' using 1:%u t '", x);
            nextFlow->PrintDescription(outfile);
            fprintf(outfile, "'");
            nextFlow = nextFlow->Next();
            x++;
        }
        fprintf(outfile, "\n\n\n");
        fflush(outfile);
	
	    // Append data from temp file to output file
	    if(!(infile = fopen(temp_file, "r")))
	    {
	        perror("qplot: Error opening our temp file");
	        exit(-1);
	    }
	    int result;
	    char buffer[64];
	    while ((result = fread(buffer, sizeof(char), 64, infile)))
	    {
	         fwrite(buffer, sizeof(char), result, outfile);
	    }
	    fclose(infile);
        unlink(temp_file);
	    fclose(outfile);
    }
    else if (use_gnuplot)
    {
        fprintf(stderr, "qplot: No data to plot!\n");    
    }
    
    fprintf(stderr, "qplot: Done.\n");
    exit(0);
}  // end main()



FastReader::FastReader()
    : savecount(0)
{
    
}

bool FastReader::Read(FILE* filePtr, char* buffer, unsigned int* len)
{
    unsigned int want = *len;   
    if (savecount)
    {
        unsigned int ncopy = MIN(want, savecount);
        memcpy(buffer, saveptr, ncopy);
        savecount -= ncopy;
        saveptr += ncopy;
        buffer += ncopy;
        want -= ncopy;
    }
    while (want)
    {
        unsigned int result = fread(savebuf, sizeof(char), 256, filePtr);
        if (result)
        {
            unsigned int ncopy= MIN(want, result);
            memcpy(buffer, savebuf, ncopy);
            savecount = result - ncopy;
            saveptr = savebuf + ncopy;
            buffer += ncopy;
            want -= ncopy;
        }
        else  // end-of-file
        {
            *len -= want;
            if (*len)
                return true;  // we read something
            else
                return false; // we read nothing
        }
    }
    return true;
}  // end FastReader::Read()

// An OK text readline() routine (reads what will fit into buffer incl. NULL termination)
// if *len is unchanged on return, it means the line is bigger than the buffer and 
// requires multiple reads

bool FastReader::Readline(FILE* filePtr, char* buffer, unsigned int* len)
{   
    unsigned int count = 0;
    unsigned int length = *len;
    char* ptr = buffer;
    unsigned int one = 1;
    while ((count < length) && Read(filePtr, ptr, &one))
    {
        if (('\n' == *ptr) || ('\r' == *ptr))
        {
            *ptr = '\0';
            *len = count;
            return true;
        }
        count++;
        ptr++;
    }
    // Either we've filled the buffer or hit end-of-file
    if (count < length) *len = 0; // Set *len = 0 on EOF
    return false;
}  // end FastReader::Readline()
