/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <limits.h>  // for PATH_MAX

#define VERSION "2.3"

template<class T>
inline const T& MAX(const T& x, const T& y) {return (x>y)?x:y;}
template<class T>
inline const T& MIN(const T& x, const T& y) {return (x<y)?x:y;}

const int MAX_LINE = 256;

class FastReader
{
    public:
        FastReader();
        bool Read(FILE* filePtr, char* buffer, unsigned int* len);
        bool Readline(FILE* filePtr, char* buffer, unsigned int* len);
    
    private:
        char         savebuf[256];
        char*        saveptr;
        unsigned int savecount;
};  // end class FastReader

enum NsTraceEvent  
{
    NS_TRACE_UNDEFINED,
    NS_TRACE_PKT_TXMIT,
    NS_TRACE_PKT_DROP
};

class Flow
{
    friend class FlowList;
    
    public:
        Flow();
        ~Flow();
        void PrintDescription(FILE* f);
        const char* Type() {return type;}
        bool SetType(const char* theType);
        int SrcAddr() {return src_addr;}
        void SetSrcAddr(int value) {src_addr = value;}
        int SrcPort() {return src_port;}
        void SetSrcPort(int value) {src_port = value;}
        int DstAddr() {return dst_addr;}
        void SetDstAddr(int value) {dst_addr = value;}
        int DstPort() {return dst_port;}
        void SetDstPort(int value) {dst_port = value;}
        
        bool TypeMatch(const char* theType)
        {
            if (theType && type)
                return (0 == strcmp(theType, type));
            else
                return (theType == type);
        }
        
        bool Match(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort);
        bool ExactMatch(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort);
        bool IsComposite();
        
        Flow* Next() {return next;}
        
            
    private:
        char* type;
        int   type_len;
        int   src_addr;
        int   src_port;
        int   dst_addr;
        int   dst_port;
            
        Flow* prev;
        Flow* next;  
};

class FlowList
{
    public:
        FlowList();
        ~FlowList();
        void Append(Flow* theFlow);
        void Remove(Flow* theFlow);
        Flow* FindFlowByMatch(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort);
        Flow* Head() {return head;}
    
    private:
        Flow* head;
        Flow* tail;
    
};


Flow::Flow()
    : type(NULL), type_len(0), 
      src_addr(-1), src_port(-1), dst_addr(-1), dst_port(-1),
      prev(NULL), next(NULL)
{
}

Flow::~Flow()
{
    if (type) delete []type;
}

bool Flow::SetType(const char* theType)
{
    if (type) delete []type;
    int len = strlen(theType) + 1;
    if(!(type = new char[len]))
    {
        perror("lossplot: Error allocating flow type storage");
        return false;
    }
    strcpy(type, theType);
    type_len = len - 1;
    return true;
}  // end Flow::SetName()


bool Flow::IsComposite()
{
    if ((NULL != type) || 
        (src_addr != -1) || (src_port != -1) ||
        (dst_addr >= 0) || (dst_port >= 0))
    {
        return false;   
    }
    else
    {
        return true;
    }
}  // end Flow::IsComposite()

bool Flow::Match(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort)
{
    if ((type && !TypeMatch(theType)) ||    
        ((dst_port >= 0) && (dst_port != dstPort)) ||
        ((dst_addr != -1) && (dst_addr != dstAddr)) ||
        ((src_port >= 0) && (src_port != srcPort)) ||
        ((src_addr != -1) && (src_addr != srcAddr)))
    {
        return false;
    }
    else
    {
        return true;
    }
}  // end Flow::Match()

bool Flow::ExactMatch(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort)
{
    if ((!TypeMatch(theType)) ||     
        (dst_port != dstPort) ||
        (dst_addr != dstAddr) ||
        (src_port != srcPort) ||
        (src_addr != srcAddr))
    {
        return false;
    }
    else
    {
        return true;
    }
}  // end Flow::ExactMatch()

void Flow::PrintDescription(FILE* f)
{
    if (type)
        fprintf(f, "%s:", type);
    else
        fprintf(f, "*:");
    if (src_addr == -1)
        fprintf(f, "*.");
    else
        fprintf(f, "%d.", src_addr);
    if (src_port < 0)
        fprintf(f, "*->");
    else
        fprintf(f, "%d->", src_port);
    if (dst_addr == -1)
        fprintf(f, "*.");
    else
        fprintf(f, "%d.", dst_addr);
    if (dst_port < 0)
        fprintf(f, "*");
    else
        fprintf(f, "%d", dst_port);
}  // end Flow::PrintDescription()

FlowList::FlowList()
    : head(NULL), tail(NULL)
{
}

FlowList::~FlowList()
{
}

void FlowList::Append(Flow* theFlow)
{
    if ((theFlow->prev = tail))
        theFlow->prev->next = theFlow;
    else
        head = theFlow;
    theFlow->next = NULL;
    tail = theFlow;
}  // end FlowList::Append()

void FlowList::Remove(Flow* theFlow)
{
    if (theFlow->prev)
        theFlow->prev->next = theFlow->next;
    else
        head = theFlow->next;
    
    if (theFlow->next)
        theFlow->next->prev = theFlow->prev;
    else
        tail = theFlow->prev;
}  // end FlowList::Remove()
        
Flow* FlowList::FindFlowByMatch(const char* theType, int srcAddr, int srcPort, int dstAddr, int dstPort)
{
    Flow* nextFlow = head;
    while (nextFlow)
    {
        if (nextFlow->ExactMatch(theType, srcAddr, srcPort, dstAddr, dstPort))
            return nextFlow;
        nextFlow = nextFlow->next;
    }
    return NULL;   
}  // end FlowList:FindFlowByName()

const char WILDCARD = 'X';

inline void usage()
{
    fprintf(stderr, "Usage: lossplot [raw][auto <protoNameLen>][flow <type,srcAddr.port,dstAddr.port>]"
                           "link <srcNode,dstNode> trace <traceFile> [<outputFile>]\n");
    fprintf(stderr, "       (Wildcard type/address/port parameters with 'X')\n");
}
        
int main(int argc, char* argv[])
{ 
    FlowList flow_list;
    double windowSize = 1.0;
    char* trace_file;
    char* output_file;
    bool use_gnuplot = true;
    bool make_gif = false;
    int link_src = 0;
    int link_dst = 0;
    bool auto_detect = false;
    unsigned int detect_proto_len = 31;
    
    fprintf(stderr, "Lpplot Version %s\n", VERSION);
    if (argc < 2)
    {
        usage();
        exit(-1);
    }
    
    // Parse command line
    fprintf(stderr, "\n");
    int i = 1;
    while(i < argc)
    {
        if (!strcmp("flow", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "lossplot: Insufficient \"flow\" arguments!\n");
                usage();
                exit(-1);
            }
            // Pull out flow type, checking for wildcard
            Flow* theFlow = new Flow;
            if (!theFlow)
            {
                perror("lossplot: Error allocating memory for flow");
                exit(-1);
            }
            char* flow_info = strtok(argv[i++], ",");
            if (flow_info) 
            {
                if(WILDCARD != flow_info[0])
                    theFlow->SetType(flow_info);
            }
            else
            {
                fprintf(stderr, "lossplot: Error parsing \"flow\" description!\n");
                usage();
                exit(1);
            }
            // Pull out source addr/port, checking for wildcards
            flow_info = strtok(NULL, ",");
            if (flow_info)
            {
                // Parse source address/port
                char* ptr = strchr(flow_info, '.');
                if (ptr) 
                {
                    *ptr++ = '\0';
                    if (WILDCARD != ptr[0])
                        theFlow->SetSrcPort(atoi(ptr));
                }
                if (WILDCARD != flow_info[0])
                    theFlow->SetSrcAddr(atoi(flow_info));
            
                // Pull out destination addr/port, checking for wildcards
                flow_info = strtok(NULL, ",");
                if (flow_info)
                {
                    // Parse source address/port
                    char* ptr = strchr(flow_info, '.');
                    if (ptr) 
                    {
                        *ptr++ = '\0';
                        if (WILDCARD != ptr[0])
                            theFlow->SetDstPort(atoi(ptr));
                    }
                    if (WILDCARD != flow_info[0])
                       theFlow->SetDstAddr(atoi(flow_info));
                }
            }
            flow_list.Append(theFlow);
            fprintf(stderr, "lossplot: Adding flow: ");
            theFlow->PrintDescription(stderr);
            fprintf(stderr, "\n");
        }
        else if (!strcmp("link", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "lossplot: Insufficient \"link\" arguments!\n");
                usage();
                exit(-1);
            }
            if (2 != sscanf(argv[i++], "%d,%d", &link_src, &link_dst))
            {
                fprintf(stderr, "lossplot: Error parsing \"link\" description!\n");
                usage();
                exit(-1);
            }
        }         
        else if (!strcmp("auto", argv[i]))
        {
            i++;
            auto_detect = true;
        }    
                
        else if (!strcmp("len", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "lossplot: Insufficient \"len\" arguments!\n");
                usage();
                exit(-1);
            }   
	        detect_proto_len = atoi(argv[i++]);
        }    
        else if (!strcmp("raw", argv[i]))
        {
            i++;
            use_gnuplot = false;
        } 
        else if (!strcmp("gif", argv[i]))
        {
            i++;
            make_gif = true;
        }
        else if (!strcmp("trace", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "lossplot: Insufficient \"trace\" arguments!\n");
                usage();
                exit(-1);
            }
            trace_file = argv[i++];
        }        
        else
        {
            output_file = argv[i++];
        }
    }   
    
    // Open ns-2 trace input file
    FILE* infile;
    if (trace_file)
    {
        if(!(infile = fopen(trace_file, "r")))
        {
            perror("lossplot: Error opening input trace file");
            usage();
            exit(-1);
        }
    }
    else
    {
        fprintf(stderr, "lossplot: No \"trace\" file provided!\n");
        usage();
        exit(-1);
    }
    
    // Open output file
    FILE* outfile;
    char temp_file[PATH_MAX];
    if (output_file)
    {
        strcpy(temp_file, output_file);
	    if (use_gnuplot) strcat(temp_file, ".tmp");
        if(!(outfile = fopen(temp_file, "w+")))
        {
            perror("lossplot: Error opening output file");
            usage();
            exit(-1);
        } 
    }
    else
    {
        output_file = "stdout";
        outfile = stdout;  
        use_gnuplot = false;
    }
    
    char buffer[MAX_LINE];
    unsigned int len = MAX_LINE;
    float theTime = 0.0;
    
    FastReader reader;
    while (reader.Readline(infile, buffer, &len))
    {
        char theEvent, proto[32], junkText[16];
        unsigned int linkSrc, linkDst, pktSize, flowId;
        int srcAddr, srcPort, dstAddr, dstPort;
        if (12 == sscanf(buffer, "%c %f %u %u %s %u %s %d %d.%d %d.%d", 
                                      &theEvent, &theTime, 
                                      &linkSrc, &linkDst, 
                                      proto, &pktSize, junkText, 
                                      &flowId, &srcAddr, &srcPort, &dstAddr, &dstPort))
        {
            proto[31] = '\0';
            unsigned protoLen = MIN(strlen(proto), (unsigned int)32);
            protoLen = MIN(detect_proto_len, protoLen);
            proto[protoLen] = '\0';
                        
            
            if ((linkSrc == link_src) && (linkDst == link_dst))
            {
                NsTraceEvent eventType = NS_TRACE_UNDEFINED;
                if ('-' == theEvent)
                {
                    eventType = NS_TRACE_PKT_TXMIT;
                }
                else if ('d' == theEvent)
                {
                    eventType = NS_TRACE_PKT_DROP;
                }
                else
                {
                    len = MAX_LINE;
                    continue;  // Skip to the next line
                }
                Flow* nextFlow = flow_list.Head();
                bool match = false;
                while (nextFlow)
                {
                    if (nextFlow->Match(proto, srcAddr, srcPort, dstAddr, dstPort))
                    {
                        if (!nextFlow->IsComposite()) match = true;
                        if (NS_TRACE_PKT_DROP == eventType)
                        {
                            // Plot first point of edge
                            fprintf(outfile, "%7.3f", theTime);
                            Flow* nextFlow2 = flow_list.Head();
                            int base = 0;
                            while (nextFlow2)
                            {
                                fprintf(outfile, ", %d", base);
                                nextFlow2 = nextFlow2->Next();
                                base += 2;
                            }
                            fprintf(outfile, "\n");

                            // Plot second point of edge
                            fprintf(outfile, "%7.3f", theTime);
                            nextFlow2 = flow_list.Head();
                            base = 0;
                            while (nextFlow2)
                            {
                                if (nextFlow2 == nextFlow)
                                    fprintf(outfile, ", %d", base+1);
                                else
                                    fprintf(outfile, ", %d", base);
                                nextFlow2 = nextFlow2->Next();
                                base += 2;
                            }
                            fprintf(outfile, "\n");    

                            // Plot third point of edge
                            fprintf(outfile, "%7.3f", theTime);
                            nextFlow2 = flow_list.Head();
                            base = 0;
                            while (nextFlow2)
                            {
                                fprintf(outfile, ", %d", base);
                                nextFlow2 = nextFlow2->Next();
                                base += 2;
                            }
                            fprintf(outfile, "\n"); 
                        }  // end if(NS_TRACE_PKT_DROP == eventType)
                    }
                    nextFlow = nextFlow->Next();      
                }  // end while(nextFlow)
                if (auto_detect && !match)
                {
                    Flow* theFlow = new Flow;
                    if (theFlow)
                    {
                        theFlow->SetType(proto);
                        theFlow->SetSrcAddr(srcAddr);
                        theFlow->SetSrcPort(srcPort);
                        theFlow->SetDstAddr(dstAddr);
                        theFlow->SetDstPort(dstPort);
                        flow_list.Append(theFlow);
			            fprintf(stderr, "lossplot: Adding flow: ");
            	        theFlow->PrintDescription(stderr);
            	        fprintf(stderr, "\n");
                        if (NS_TRACE_PKT_DROP == eventType)
                        {
                           // Plot first point of edge
                            fprintf(outfile, "%7.3f", theTime);
                            Flow* nextFlow2 = flow_list.Head();
                            int base = 0;
                            while (nextFlow2)
                            {
                                fprintf(outfile, ", %d", base);
                                nextFlow2 = nextFlow2->Next();
                                base += 2;
                            }
                            fprintf(outfile, "\n");

                            // Plot second point of edge
                            fprintf(outfile, "%7.3f", theTime);
                            nextFlow2 = flow_list.Head();
                            base = 0;
                            while (nextFlow2)
                            {
                                if (nextFlow2 == nextFlow)
                                    fprintf(outfile, ", %d", base+1);
                                else
                                    fprintf(outfile, ", %d", base);
                                nextFlow2 = nextFlow2->Next();
                                base += 2;
                            }
                            fprintf(outfile, "\n");    

                            // Plot third point of edge
                            fprintf(outfile, "%7.3f", theTime);
                            nextFlow2 = flow_list.Head();
                            base = 0;
                            while (nextFlow2)
                            {
                                fprintf(outfile, ", %d", base);
                                nextFlow2 = nextFlow2->Next();
                                base += 2;
                            }
                            fprintf(outfile, "\n"); 
                        }  // end if(NS_TRACE_PKT_DROP == eventType)
                    }
                    else
                    {
                        perror("lossPatternPlot: Error adding new flow");
                        fclose(infile);
                        fclose(outfile);
                        exit(0);  
                    }
                }  // end if(auto_detect && !match)
            }  // end if ((linkSrc == link_src) && (linkDst == linkDst))
        } // end if(12 == sscanf())
        len = MAX_LINE;
    }  // end while(reader.Readline())
    
    // print last point for completion
    fprintf(outfile, "%7.3f", theTime);
    Flow* nextFlow = flow_list.Head();
    int base = 0;
    while (nextFlow)
    {
        fprintf(outfile, ", %d", base);
        nextFlow = nextFlow->Next();
        base += 2;
    }
    fprintf(outfile, "\n");    
    fprintf(stderr, "lossplot: Done.\n");
    fclose(infile);
    fclose(outfile);
    
    
    // Create final output file with gnuplot header if applicable
    if (use_gnuplot)
    {
    	if (!output_file)
        {
            fprintf(stderr, "lossplot: Must specify output file name for gnuplot file generation!\n");
            exit(-1);
        }
    	if(!(outfile = fopen(output_file, "w+")))
	    {
	        perror("lossplot: Error opening output file");
	        exit(-1);
	    }	
        if (make_gif)
        {           
            char gifName[256];
            strcpy(gifName, output_file);
            strcat(gifName, ".gif");
            fprintf(outfile, "set term gif\n");
            fprintf(outfile, "set output '%s'\n", gifName);   
        }
        fprintf(outfile, "set title '%s'\n", output_file);
        fprintf(outfile, "set xlabel 'Time (sec)'\n");
        fprintf(outfile, "set ylabel 'Loss Events'\n");
        fprintf(outfile, "set data style lines\n");
        fprintf(outfile, "set key top right\n");
        
        Flow* nextFlow = flow_list.Head();
        unsigned int x = 0;
        while (nextFlow)
        {
            x++;
            nextFlow = nextFlow->Next();
        }
        fprintf(outfile, "set yrange[0:%u]\n", 2*x);
        fprintf(outfile, "set noytics\n");
        
        fprintf(outfile, "plot ");
        nextFlow = flow_list.Head();
        x = 2;
        while (nextFlow)
        {
            if (x > 2) 
                fprintf(outfile, ",\\\n'%s' index 1 using 1:%u t '",
                              output_file, x);
            else
                fprintf(outfile, "\\\n'-' using 1:%u t '", x);
            nextFlow->PrintDescription(outfile);
            fprintf(outfile, "'");
            nextFlow = nextFlow->Next();
            x++;
        }
        fprintf(outfile, "\n\n\n");
        fflush(outfile);
	
	    // Append data from temp file to output file
	    if(!(infile = fopen(temp_file, "r")))
	    {
	        perror("lossplot: Error opening our temp file");
	        exit(-1);
	    }
	    int result;
	    char buffer[64];
	    while ((result = fread(buffer, sizeof(char), 64, infile)))
	    {
	         fwrite(buffer, sizeof(char), result, outfile);
	    }
	    fclose(infile);
        unlink(temp_file);
	    fclose(outfile);
    }
    
    exit(0);
}  // end main()


FastReader::FastReader()
    : savecount(0)
{
    
}

bool FastReader::Read(FILE* filePtr, char* buffer, unsigned int* len)
{
    unsigned int want = *len;   
    if (savecount)
    {
        unsigned int ncopy = MIN(want, savecount);
        memcpy(buffer, saveptr, ncopy);
        savecount -= ncopy;
        saveptr += ncopy;
        buffer += ncopy;
        want -= ncopy;
    }
    while (want)
    {
        unsigned int result = fread(savebuf, sizeof(char), 256, filePtr);
        if (result)
        {
            unsigned int ncopy= MIN(want, result);
            memcpy(buffer, savebuf, ncopy);
            savecount = result - ncopy;
            saveptr = savebuf + ncopy;
            buffer += ncopy;
            want -= ncopy;
        }
        else  // end-of-file
        {
            *len -= want;
            if (*len)
                return true;  // we read something
            else
                return false; // we read nothing
        }
    }
    return true;
}  // end FastReader::Read()

// An OK text readline() routine (reads what will fit into buffer incl. NULL termination)
// if *len is unchanged on return, it means the line is bigger than the buffer and 
// requires multiple reads

bool FastReader::Readline(FILE* filePtr, char* buffer, unsigned int* len)
{   
    unsigned int count = 0;
    unsigned int length = *len;
    char* ptr = buffer;
    unsigned int one = 1;
    while ((count < length) && Read(filePtr, ptr, &one))
    {
        if (('\n' == *ptr) || ('\r' == *ptr))
        {
            *ptr = '\0';
            *len = count;
            return true;
        }
        count++;
        ptr++;
    }
    // Either we've filled the buffer or hit end-of-file
    if (count < length) *len = 0; // Set *len = 0 on EOF
    return false;
}  // end FastReader::Readline()
