/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
#include <stdio.h>
        
#include "sysdefs.h"
#include "mdpSession.h"
#include "version.h"
        

/* MDP type definitions */
typedef const void* MdpInstanceHandle;
typedef const void* MdpSessionHandle;
typedef const void* MdpNodeHandle;
typedef const void* MdpObjectHandle;

typedef unsigned long MdpNodeId;
typedef unsigned long MdpObjectTransportId;

typedef const void* MdpTimerHandle;
typedef const void* MdpSocketHandle;

enum MdpTimerInstallCmd
{
    MDP_TIMER_INSTALL, 
    MDP_TIMER_MODIFY, 
    MDP_TIMER_REMOVE
};
typedef bool (MdpTimerInstallCallback)(MdpTimerInstallCmd cmd, 
                                       double             delay,
                                       MdpTimerHandle     timerHandle,
                                       MdpInstanceHandle  instanceHandle);

enum MdpSocketInstallCmd 
{
    MDP_SOCKET_INSTALL, 
    MDP_SOCKET_REMOVE
};
typedef bool (MdpSocketInstallCallback)(MdpSocketInstallCmd cmd,
                                        MdpSocketHandle     socketHandle,
                                        MdpInstanceHandle   instanceHandle);

/* MDP constants */
const MdpInstanceHandle MDP_NULL_INSTANCE = ((MdpInstanceHandle) 0);
const MdpSessionHandle MDP_NULL_SESSION = ((MdpSessionHandle) 0);

const char* MdpVersion() { return VERSION;} 

MdpInstanceHandle MdpInit(const char* localName, 
                          MdpNodeId   localId,
                          MdpError*   error)
{
    // If we're missing the "localName" or "localId",
    // we going to need the local host name
    if (error) *error = MDP_ERROR_NONE;
    char hostName[MDP_NODE_NAME_MAX]; 
    if (!localName || (MDP_NULL_NODE == localId))
    {
#ifdef WIN32
		// Startup WinSock for name lookup
		WSADATA wsaData;
		WORD wVersionRequested = MAKEWORD( 2, 0 );
		int err = WSAStartup( wVersionRequested, &wsaData );
		if ( err != 0 ) 
		{
			if (error) *error = MDP_ERROR_DNS_FAIL;  // (TBD) change 
			DMSG(0, "Error initializing WinSock!\n");
			return MDP_NULL_INSTANCE;

		}
#endif // Win32
        // Try to get fully qualified host name if possible
        hostName[0] = '\0'; 
        hostName[MDP_NODE_NAME_MAX-1] = '\0';   
        if (gethostname(hostName, MDP_NODE_NAME_MAX-1)) 
        {
	        DMSG(0, "MdpInit(): gethostname(%s) error", hostName);
            if (error) *error = MDP_ERROR_DNS_FAIL;
#ifdef WIN32
			WSACleanup();
#endif // WIN32
            return MDP_NULL_INSTANCE;
        }
        else
        {
            struct hostent * he;
            if ((he = gethostbyname(hostName)) == NULL) 
            {
                DMSG(0, "MdpInit(): gethostbyname(%s) error", hostName);
            }
            else
            {
                strncpy(hostName, he->h_name, MDP_NODE_NAME_MAX);
                if (strchr(hostName, '.') == NULL) 
                {
                    char **alias = he->h_aliases;
                    while( alias && *alias )
                    {
                        if (strchr(*alias, '.') && (strlen(*alias)>strlen(hostName)))
                        {    
                            if (strlen(*alias) <= MDP_NODE_NAME_MAX)
                                strncpy(hostName, *alias, MDP_NODE_NAME_MAX);
                        }
                        else
                        {
                            alias++;
                        }
                    }
                }
            }
#ifdef WIN32
			if (MDP_NULL_NODE != localId) WSACleanup();
#endif // WIN32
        }
    }
    
    char nodeName[MDP_NODE_NAME_MAX];
    if (!localName) 
    {
        // If no "localName" is specified, use "user@host" as default name
        // Get user name
#ifdef WIN32
	char userName[16];
	unsigned long namelen = 16;
	char *ptr;
	if (GetUserName(userName, &namelen))
		ptr = userName;
    else
		ptr = NULL;

#else

#if HAVE_CUSERID
        char *ptr = cuserid((char*)NULL);
#else
#if HAVE_GETLOGIN
        char *ptr = getlogin();
#else
#error At least one, HAVE_CUSERID or HAVE_GETLOGIN must be supported
#endif // HAVE_GETLOGIN
#endif // HAVE_CUSERID 
		
#endif // if/else WIN32
		if (ptr)
        {
            strncpy(nodeName, ptr, 16);
            nodeName[16] = '\0';
            strcat(nodeName, "@");
        }
        else
        {
            nodeName[0] = '\0';
        }            
        strncat(nodeName, hostName, 
                MDP_NODE_NAME_MAX - strlen(nodeName));
        localName = nodeName;
    }  // end if(!localName)
    
    // if no (NULL) "localId" is specified, 
    if (MDP_NULL_NODE == localId)
    {
        // Use local station IP address as "local_id"
        NetworkAddress myAddr;
		bool result = myAddr.LookupHostAddress(hostName);
#ifdef WIN32
		WSACleanup();
#endif // WIN32
        if(!result)
        {
            DMSG(0, "Error looking up local host address (%s)",
				  hostName);
            if (error) *error = MDP_ERROR_DNS_FAIL;
            return MDP_NULL_INSTANCE;
        }
        localId = myAddr.IPv4HostAddr();
    }
    
    MdpSessionMgr* mgr = new MdpSessionMgr;
    if (mgr)
    {
        mgr->SetLocalNodeId(localId);
        mgr->SetNodeName(localName);
        // (TBD) Set next_transport_id to random or controlled starting value
        if (error) *error = MDP_ERROR_NONE;
        return ((MdpInstanceHandle)mgr);
    }
    else
    {
        if (error) *error = MDP_ERROR_MEMORY;
        return NULL;   
    }
}  // end MdpInit()

void MdpDestroy(MdpInstanceHandle instance)
{
	delete ((MdpSessionMgr*)instance);
}  // end MdpDestroy()

void MdpInstanceSetUserData(MdpInstanceHandle instance, const void* userData)
{
    ((MdpSessionMgr*)instance)->SetUserData(userData);
}  // end MdpInstanceSetUserData()

const void* MdpInstanceGetUserData(MdpInstanceHandle instance)
{
    return ((MdpSessionMgr*)instance)->GetUserData();
}  // end MdpInstanceGetUserData()
        
MdpNodeId MdpGetLocalNodeId(MdpInstanceHandle instance)
{
    if (MDP_NULL_INSTANCE != instance)
        return ((MdpSessionMgr*)instance)->LocalNodeId();  
    else
        return MDP_NULL_NODE; 
}  // end MdpGetLocalNodeId()

MdpError MdpSetLocalNodeId(MdpInstanceHandle instance, 
                           MdpNodeId         nodeId)
{
    if (MDP_NULL_INSTANCE != instance)
    {
        if (MDP_NULL_NODE != nodeId)
        {
            ((MdpSessionMgr*)instance)->SetLocalNodeId(nodeId);
            return MDP_ERROR_NONE;
        }
        else
        {
            return MDP_ERROR_NODE_INVALID;
        }  
    }
    else
    {
        return MDP_ERROR_INSTANCE_INVALID; 
    }
}  // MdpSetLocalNodeId()

MdpSessionHandle MdpNewSession(MdpInstanceHandle instance,
                               const char*       address, 
                               unsigned short    rxPort,
                               unsigned short    txPort,
                               unsigned char     ttl, 
                               MdpError*         error)
{
    if (MDP_NULL_INSTANCE != instance)
    {
        MdpSession *theSession = ((MdpSessionMgr*)instance)->NewSession();
        if (theSession)
        {
            if (!theSession->SetAddress(address, rxPort, txPort))
            {
                if (error) *error = MDP_ERROR_DNS_FAIL;
                return MDP_NULL_SESSION;
            }
            theSession->SetTTL(ttl);
            theSession->SetTxRate(MDP_DEFAULT_TX_RATE);
            if (error) *error = MDP_ERROR_NONE;
            return (MdpSessionHandle) theSession; 
        }
        else
        {
            if (error) *error = MDP_ERROR_MEMORY;
            return MDP_NULL_SESSION;
        }
    }
    else
    {
        if (error) *error = MDP_ERROR_INSTANCE_INVALID;
        return MDP_NULL_SESSION;
    }
}  // end MdpNewSession()

void MdpDeleteSession(MdpInstanceHandle  instance,
                      MdpSessionHandle   sessionHandle)
{
    if (MDP_NULL_INSTANCE != instance)
    {
        MdpSession* theSession = 
            ((MdpSessionMgr*)instance)->FindSession((MdpSession*)sessionHandle);
        if (theSession)
        {
            theSession->Close(true);
            ((MdpSessionMgr*)instance)->DeleteSession(theSession);
        }
    }
}  // end MdpDeleteSession()

void MdpSessionSetUserData(MdpSessionHandle sessionHandle,
                           const void*     userData)
{
    ((MdpSession*)sessionHandle)->SetUserData(userData);
}  // end MdpSessionSetUserData()

const void* MdpSessionGetUserData(MdpSessionHandle sessionHandle)
{
    return ((MdpSession*)sessionHandle)->GetUserData();
}  // end MdpSessionGetUserData()

MdpError MdpSessionSetStatusReporting(MdpSessionHandle sessionHandle, 
                                      bool             state)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetStatusReporting(state);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetStatusReporting()


MdpError MdpSessionSetTxRate(MdpSessionHandle    sessionHandle, 
                             double              txRate)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetTxRate((unsigned long)txRate);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetTxRate()

MdpError MdpSessionSetTxRateBounds(MdpSessionHandle    sessionHandle, 
                                   double              min,
                                   double              max)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetTxRateBounds((unsigned long)min,
                                                      (unsigned long)max);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetTxRateBounds()

MdpError MdpSessionSetTOS(MdpSessionHandle  sessionHandle,
                          unsigned char     theTOS)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        if (((MdpSession*)sessionHandle)->SetTOS(theTOS))
            return MDP_ERROR_NONE;
        else
            return MDP_ERROR;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetTOS()

MdpError MdpSessionSetMulticastInterfaceAddress(MdpSessionHandle sessionHandle, 
                                                const char*      interfaceAddr)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        if (((MdpSession*)sessionHandle)->SetMulticastInterfaceAddress(interfaceAddr))
            return MDP_ERROR_NONE;
        else
            return MDP_ERROR;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetMulticastInterfaceAddress()

MdpError MdpSessionSetMulticastLoopback(MdpSessionHandle sessionHandle, 
                                        bool             state)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetMulticastLoopback(state);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetMulticastLoopback()

MdpError MdpSessionSetPortReuse(MdpSessionHandle sessionHandle, 
                                bool             state)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetPortReuse(state);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetPortReuse()

MdpError MdpSessionSetGrttProbeInterval(MdpSessionHandle sessionHandle, 
                                        double           intervalMin,
                                        double           intervalMax)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        if (((MdpSession*)sessionHandle)->SetGrttProbeInterval(intervalMin, intervalMax))
            return MDP_ERROR_NONE;
        else
            return MDP_ERROR_PARAMETER_INVALID;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetGrttProbeInterval()

MdpError MdpSessionSetGrttProbing(MdpSessionHandle sessionHandle, 
                                  bool             state)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetGrttProbing(state);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetGrttProbing()


MdpError MdpSessionSetServerGrttMax(MdpSessionHandle sessionHandle, 
                                    double           grttMax)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetGrttMax(grttMax);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetGrttMax()

MdpError MdpSessionSetServerGrttEstimate(MdpSessionHandle sessionHandle, 
                                         double            theGRTT)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetGrttEstimate(theGRTT);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetServerGrttEstimate()

MdpError MdpSessionGetServerGrttEstimate(MdpSessionHandle sessionHandle, 
                                         double*          grttEstimate)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        *grttEstimate = ((MdpSession*)sessionHandle)->GrttEstimate();
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionGetServerGrttEstimate()


void MdpNodeSetUserData(MdpNodeHandle nodeHandle,
                        const void*   userData)
{
    ((MdpNode*)nodeHandle)->SetUserData(userData);
}  // end MdpNodeSetUserData()

const void* MdpNodeGetUserData(MdpNodeHandle nodeHandle)
{
    return ((MdpNode*)nodeHandle)->GetUserData();
}  // end MdpNodeGetUserData()


MdpError MdpSessionGetNodeAddress(MdpSessionHandle sessionHandle, 
                                  MdpNodeId        nodeId,
                                  unsigned long*   address, 
                                  unsigned short*  port)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        MdpServerNode* theNode = 
            (MdpServerNode*)((MdpSession*)sessionHandle)->FindServerById(nodeId);
        if (theNode)
        {
            *address = theNode->Address()->IPv4HostAddr();
            *port = theNode->Address()->Port();
            return MDP_ERROR_NONE;
        }
        else
        {
            return MDP_ERROR_NODE_INVALID;
        }
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionGetNodeAddress()

MdpError MdpSessionGetNodeGrttEstimate(MdpSessionHandle sessionHandle,
                                       MdpNodeId        nodeId, 
                                       double*          grttEstimate)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        if (MDP_NULL_NODE != nodeId)
        {
            MdpServerNode* theNode = 
                (MdpServerNode*)((MdpSession*)sessionHandle)->FindServerById(nodeId);
            if (theNode)
            {
                *grttEstimate = theNode->GrttEstimate();
                return MDP_ERROR_NONE;
            }
            else
            {
                return MDP_ERROR_NODE_INVALID;
            }
        }
        else
        {
            return MdpSessionGetServerGrttEstimate(sessionHandle, grttEstimate);   
        }
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionGetNodeGrttEstimate()

MdpObjectHandle MdpSessionGetNodeNextRxObject(MdpSessionHandle sessionHandle,
                                              MdpNodeId        nodeId,
                                              MdpObjectHandle  previousObject,
                                              MdpError*        error = NULL)
{
    if (*error) *error = MDP_ERROR_NONE;
    if (MDP_NULL_SESSION != sessionHandle)
    {
        MdpServerNode* theNode = 
                (MdpServerNode*)((MdpSession*)sessionHandle)->FindServerById(nodeId);
        if (theNode)
        {
            MdpObject* theObject = 
                theNode->FindRecvObject((MdpObject*)previousObject);
            if (theObject)
            {
                return ((MdpObjectHandle)theObject->Next());
            }
            else
            {
                return ((MdpObjectHandle)theNode->CurrentRecvObject());   
            }
        }
        else
        {
            if (error) *error = MDP_ERROR_NODE_INVALID;
            return MDP_NULL_OBJECT;
        }
    }
    else
    {
        if (error) *error = MDP_ERROR_SESSION_INVALID;
        return MDP_NULL_OBJECT;
    } 
}  // end MdpSessionGetNodeNextRxObject()

MdpError MdpSessionDeactivateNode(MdpSessionHandle sessionHandle,
                                  MdpNodeId        nodeId)
{    
    if (MDP_NULL_SESSION != sessionHandle)
    {
        MdpServerNode* theNode = 
                (MdpServerNode*)((MdpSession*)sessionHandle)->FindServerById(nodeId);
        if (theNode)
        {
            theNode->Deactivate();
            return MDP_ERROR_NONE;
        }
        else
        {
            return MDP_ERROR_NODE_INVALID;
        }
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionDeactivateNode()

MdpError MdpSessionDeleteNode(MdpSessionHandle sessionHandle,
                              MdpNodeId        nodeId)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        MdpServerNode* theNode = 
                (MdpServerNode*)((MdpSession*)sessionHandle)->FindServerById(nodeId);
        if (theNode)
        {
            ((MdpSession*)sessionHandle)->DeleteRemoteServer(theNode);
            return MDP_ERROR_NONE;
        }
        else
        {
            return MDP_ERROR_NODE_INVALID;
        }
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionDeleteNode()



MdpError MdpSessionSetCongestionControl(MdpSessionHandle sessionHandle, bool state)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetCongestionControl(state);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetCongestionControl()

MdpError MdpSessionSetTxCacheDepth(MdpSessionHandle  sessionHandle, 
                                   unsigned long     minCount,
                                   unsigned long     maxCount, 
                                   unsigned long     maxSize)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetTxCacheDepth(minCount, maxCount, maxSize);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetTxCacheDepth()

MdpError MdpSessionSetServerEmcon(MdpSessionHandle sessionHandle, 
                                  bool             state)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetEmconServer(state);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetServerEmcon()

// (TBD) Change this to use MdpNodeId as second argument!!!
MdpError MdpSessionAddAckingClient(MdpSessionHandle sessionHandle, 
                                   const char*      host)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        return ((MdpSession*)sessionHandle)->AddAckingHost(host);
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionServerAddAckingClient()

MdpError MdpSessionRemoveAckingClient(MdpSessionHandle sessionHandle, 
                                      const char*      host)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->RemoveAckingHost(host);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionRemoveAckingClient()

MdpError MdpSessionSetAutoParity(MdpSessionHandle  sessionHandle, 
                                 unsigned char     autoParity)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetAutoParity(autoParity);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }   
}  // end MdpSessionSetAutoParity()

/* Mdp Client Routines */
MdpError MdpSessionOpenClient(MdpSessionHandle sessionHandle, 
                              unsigned long    bufferSize)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        return ((MdpSession*)sessionHandle)->OpenClient(bufferSize);
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionOpenClient()

MdpError MdpSessionCloseClient(MdpSessionHandle sessionHandle)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->CloseClient();
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionCloseClient()


MdpError MdpSessionSetClientUnicastNacks(MdpSessionHandle sessionHandle, 
                                         bool             state)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetUnicastNacks(state);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetClientUnicastNacks()

MdpError MdpSessionSetClientMulticastAcks(MdpSessionHandle sessionHandle, 
                                          bool             state)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetMulticastAcks(state);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetClientMulticastAcks()


MdpError MdpSessionSetClientEmcon(MdpSessionHandle sessionHandle, 
                                  bool             state)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetEmconClient(state);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetClientEmcon()

MdpError MdpSessionSetClientAcking(MdpSessionHandle sessionHandle, 
                                   bool             state)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetClientAcking(state);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetClientAcking()

MdpError MdpSessionSetArchiveDirectory(MdpSessionHandle sessionHandle, 
                                       const char*      path)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        return ((MdpSession*)sessionHandle)->SetArchiveDirectory(path);
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetArchiveDirectory() 

MdpError MdpSessionSetArchiveMode(MdpSessionHandle sessionHandle, 
                                  bool             state)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetArchiveMode(state);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetArchiveMode()

MdpError MdpSessionSetDefaultNackingMode(MdpSessionHandle sessionHandle, 
                                         MdpNackingMode   nackingMode)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetDefaultNackingMode(nackingMode);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetDefaultNackingMode()

MdpError MdpSessionSetClientStreamIntegrity(MdpSessionHandle sessionHandle, 
                                            bool             state)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetStreamIntegrity(state);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetStreamIntegrity()

MdpError MdpSessionSetRobustFactor(MdpSessionHandle sessionHandle, 
                                   unsigned int     value)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetRobustFactor(value);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetRobustFactor()


// Only call _before_ opening as a server
MdpError MdpSessionSetBaseObjectTransportId(MdpSessionHandle     sessionHandle, 
                                            MdpObjectTransportId baseId)
{
    MdpSession* theSession = ((MdpSession*)sessionHandle);
    if ((MDP_NULL_SESSION != sessionHandle) &&  !theSession->IsServer())
    {
        theSession->SetBaseObjectTransportId(baseId);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionSetClientUnicast()

#ifdef PROTO_DEBUG
MdpError MdpSessionSetRecvDropRate(MdpSessionHandle sessionHandle, 
                                   double           rate)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetRecvDropRate(rate);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}

MdpError MdpSessionSetSendDropRate(MdpSessionHandle sessionHandle, 
                                   double           rate)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->SetSendDropRate(rate);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}

void MdpSetMessageTrace(bool state)
{
    SetMdpMessageTrace(state);  // from "mdpSession.h"   
}
#endif // PROTO_DEBUG

MdpObjectHandle MdpSessionQueueTxFile(MdpSessionHandle sessionHandle, 
                                      const char*      path, 
                                      const char*      name,
                                      MdpError*        error)
{
    if (error) *error = MDP_ERROR_NONE;
    if (MDP_NULL_SESSION != sessionHandle)
    {
        // Ask the session to create a new file object and 
        // queue it for transmission
        MdpFileObject *theObject= 
            ((MdpSession*)sessionHandle)->NewTxFileObject(path, name, error);    
        if (theObject)
            return (MdpObjectHandle)theObject;
        else
            return MDP_NULL_OBJECT;
    }
    else
    {
         if (error) *error = MDP_ERROR_SESSION_INVALID;
         return MDP_NULL_OBJECT;
    }
}  // end MdpSessionQueueTxFile()

MdpObjectHandle MdpSessionQueueTxData(MdpSessionHandle   sessionHandle, 
									  const char*        infoPtr,
                                      unsigned short     infoSize,
                                      char*              dataPtr,
									  unsigned long      dataSize,
                                      MdpError*          error)
{
    if (error) *error = MDP_ERROR_NONE;
    if (MDP_NULL_SESSION != sessionHandle)
    {
        // Ask the session to create a new file object and 
        // queue it for transmission
        MdpDataObject *theObject= 
            ((MdpSession*)sessionHandle)->NewTxDataObject(infoPtr, infoSize, 
                                                          dataPtr, dataSize, 
														  error);    
        if (theObject)
            return (MdpObjectHandle)theObject;
        else
            return MDP_NULL_OBJECT;
    }
    else
    {
         if (error) *error = MDP_ERROR_SESSION_INVALID;
         return MDP_NULL_OBJECT;
    }
}  // end MdpSessionQueueTxFile()


MdpError MdpSessionRemoveTxObject(MdpSessionHandle sessionHandle, 
                                  MdpObjectHandle  objectHandle)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        // Find the referenced object and delete it if necessary    
        MdpSession* theSession = (MdpSession*)sessionHandle;
        MdpObject *theObject = theSession->FindTxObject((MdpObject*) objectHandle);
        if (theObject) 
        {
            theSession->RemoveTxObject(theObject);
            return MDP_ERROR_NONE;
        }
        else
        {
            return MDP_ERROR_OBJECT_INVALID;
        }
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;   
    }
}  // end MdpSessionRemoveTxObject()

MdpError MdpSessionAbortRxObject(MdpSessionHandle sessionHandle,
                                 MdpObjectHandle  objectHandle)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {  
        MdpObject *theObject = ((MdpObject*)objectHandle);
        if ((MDP_NULL_OBJECT != objectHandle) || 
            (sessionHandle == (MdpSessionHandle)theObject->Session()))
        {
            theObject->RxAbort();
            return MDP_ERROR_NONE;
        }
        else
        {
            return MDP_ERROR_OBJECT_INVALID;
        }
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionAbortRxObject()


MdpError MdpSessionOpenServer(MdpSessionHandle sessionHandle,
                              int              segmentSize,
                              int              blockSize,
                              int              nparity,
                              unsigned long    bufferSize)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        return ((MdpSession*)sessionHandle)->OpenServer(segmentSize, blockSize, 
                                                        nparity, bufferSize);
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionOpenServer();

MdpError MdpSessionCloseServer(MdpSessionHandle sessionHandle,
                               bool             hardShutdown)
{
    if (MDP_NULL_SESSION != sessionHandle)
    {
        ((MdpSession*)sessionHandle)->CloseServer(hardShutdown);
        return MDP_ERROR_NONE;
    }
    else
    {
        return MDP_ERROR_SESSION_INVALID;
    }
}  // end MdpSessionCloseServer()


void MdpObjectSetUserData(MdpObjectHandle objectHandle,
                          const void*     userData)
{
    ((MdpObject*)objectHandle)->SetUserData(userData);
}  // end MdpObjectSetUserData()

const void* MdpObjectGetUserData(MdpObjectHandle objectHandle)
{
    return ((MdpObject*)objectHandle)->GetUserData();
}  // end MdpObjectGetUserData()

MdpObjectType MdpObjectGetType(MdpObjectHandle objectHandle)
{
    if (MDP_NULL_OBJECT != objectHandle)
        return (((MdpObject*)objectHandle)->Type());
    else
        return MDP_OBJECT_INVALID;
}  // end MdpObjectGetName()


unsigned short MdpObjectGetInfoSize(MdpObjectHandle objectHandle)
{
    if (MDP_NULL_OBJECT != objectHandle)
    {
        return ((MdpObject*)objectHandle)->InfoSize();
    }
    else
    {
        return 0;
    }
}  // end MdpObjectGetInfoSize()


unsigned short MdpObjectGetInfo(MdpObjectHandle    objectHandle, 
                                char*              buffer, 
                                unsigned short*    buflen)
{
    if (MDP_NULL_OBJECT != objectHandle)
    {
        MdpObject* theObject = (MdpObject*)objectHandle;
        const char* ptr = theObject->Info();
        if (ptr)
        {
            unsigned short len = theObject->InfoSize();
            *buflen = MIN(len, *buflen);
            if (buffer) memcpy(buffer, ptr, *buflen);
            return len;
        }
        else
        {
            return 0;
        }
    }
    else
    {
        return 0;
    }
}  // end MdpObjectGetInfo()


MdpError MdpObjectGetFileName(MdpObjectHandle objectHandle, 
                              char*           name, 
                              unsigned int    maxlen)
{
    if (MDP_NULL_OBJECT != objectHandle)
    {
        if (MDP_OBJECT_FILE == ((MdpObject*)objectHandle)->Type())
        {
            MdpFileObject* theObject = (MdpFileObject*)objectHandle;
            unsigned int maxname = MIN(maxlen, PATH_MAX);
            strncpy(name, theObject->Path(), maxname);
            unsigned int path_len = MIN(maxname, strlen(name));
            unsigned int name_len = MIN((maxname-path_len), theObject->InfoSize());
            strncpy(name+path_len, theObject->Info(), name_len);
            maxname = strlen(name);
            maxname = MIN(maxlen, maxname);
            if (maxname < maxlen) name[maxname] = '\0';
            return MDP_ERROR_NONE;
        }
        else
        {
            return MDP_ERROR_OBJECT_INVALID;
        }
    }
    else
    {
        return MDP_ERROR_OBJECT_INVALID;
    }
}  // end MdpObjectGetFileName()



MdpError MdpObjectSetNackingMode(MdpObjectHandle objectHandle, 
                                 MdpNackingMode  nackingMode)
{
    if (MDP_NULL_OBJECT != objectHandle)
	{
		((MdpDataObject *)objectHandle)->SetNackingMode(nackingMode);
		return MDP_ERROR_NONE;
	}
	else
	{
		return MDP_ERROR_OBJECT_INVALID;
	}
}  // end MdpObjectSetNackingMode6()


MdpError MdpObjectSetData(MdpObjectHandle objectHandle, 
                          char*           dataPtr, 
                          unsigned long   dataLen)
{
    if (MDP_NULL_OBJECT != objectHandle)
	{
        if (MDP_OBJECT_DATA == ((MdpObject*)objectHandle)->Type())
	    {
		    if ((((MdpDataObject*)objectHandle)->SetData(dataPtr, dataLen)))
		        return MDP_ERROR_NONE;
            else
                return MDP_ERROR;
	    }
	    else
	    {
		    return MDP_ERROR_OBJECT_INVALID;
	    }
    }
    else
    {
        return MDP_ERROR_OBJECT_INVALID;
    }
}  // end MdpObjectSetDataPtr()

char* MdpObjectGetData(MdpObjectHandle objectHandle,
                       unsigned long*  dataLen, 
					   MdpError*       error)
{
    if (MDP_NULL_OBJECT != objectHandle)
	{
        if (MDP_OBJECT_DATA == ((MdpObject*)objectHandle)->Type())
	    {
		    if (error) *error = MDP_ERROR_NONE;
		    return ((MdpDataObject*)objectHandle)->GetData(dataLen);
	    }
	    else
	    {
		    if (error) *error = MDP_ERROR_OBJECT_INVALID;
		    return NULL;
	    }
    }
    else
    {
        if (error) *error = MDP_ERROR_OBJECT_INVALID;
        return NULL;
    }
}  // end MdpObjectSetDataPtr()


unsigned long MdpObjectGetSize(MdpObjectHandle objectHandle)
{
    if (MDP_NULL_OBJECT != objectHandle)
	{
        return (((MdpObject*)objectHandle)->Size());
    }
    else
    {
        return 0;
    }
}  // end MdpObjectGetSize()

unsigned long MdpObjectGetRecvBytes(MdpObjectHandle objectHandle)
{
    if (MDP_NULL_OBJECT != objectHandle)
	{
        return (((MdpObject*)objectHandle)->RecvBytes());
    }
    else
    {
        return 0;
    }
}  // end MdpObjectGetRecvBytes()


void MdpSetNotifyCallback(MdpInstanceHandle  instance,
                          MdpNotifyCallback* notifyFunc)
{
     ((MdpSessionMgr*)instance)->SetNotifyCallback(notifyFunc);
}  // end MdpSetNotifyCallback()

void MdpSetTimerInstallCallback(MdpInstanceHandle           instance, 
                                MdpTimerInstallCallback*    timerInstaller)
{
    ((MdpSessionMgr*)instance)->SetTimerInstaller((ProtocolTimerInstallFunc*)timerInstaller);
}  // end MdpSetTimerInstallCallback()

void MdpTimerSetUserData(MdpTimerHandle timerHandle, const void* userData)
{
    ((ProtocolTimerMgr*)timerHandle)->SetUserData(userData);
}  // end MdpTimerSetUserData()

const void* MdpTimerGetUserData(MdpTimerHandle timerHandle)
{
    return ((ProtocolTimerMgr*)timerHandle)->GetUserData();
}  // end MdpTimerGetUserData()
        
double MdpTimerOnTimeout(MdpTimerHandle timerHandle)
{
    ASSERT(timerHandle);
    return ((ProtocolTimerMgr*)timerHandle)->DoTimers();
}  // end MdpTimerOnTimeout()
        
void MdpSetSocketInstallCallback(MdpInstanceHandle          instance,
                                 MdpSocketInstallCallback*  socketInstaller)
{
    ((MdpSessionMgr*)instance)->SetSocketInstaller((UdpSocketInstallFunc*)socketInstaller);
}  // end SetTimerInstallCallback()

void MdpSocketSetUserData(MdpSocketHandle socketHandle, const void* userData)
{
    ((UdpSocket*)socketHandle)->SetUserData(userData);
}  // end MdpSocketSetUserData()

const void* MdpSocketGetUserData(MdpSocketHandle socketHandle)
{
    return ((UdpSocket*)socketHandle)->GetUserData();
}  // end MdpSocketGetUserData()

#ifdef WIN32
SOCKET MdpSocketGetDescriptor(MdpSocketHandle socketHandle)
#else  // UNIX
int MdpSocketGetDescriptor(MdpSocketHandle socketHandle)
#endif  
{
    ASSERT(socketHandle);
    return ((UdpSocket*)socketHandle)->Handle();
}  // end MdpSocketGetDescriptor()

void MdpSocketOnDataReady(MdpSocketHandle socketHandle)
{
    ASSERT(socketHandle);
    ((UdpSocket*)socketHandle)->OnReceive();
}  // end MdpSocketOnDataReady()
