/*
 * Copyright (C) 2002 WIDE Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: rootprobe.c,v 1.37 2002/05/30 11:51:19 sekiya Exp $
 *
 *	runs on the following platforms:
 *		BSD/OS 4.0.1/4.1/4.2/4.3
 *		FreeBSD 4.2
 *		FreeBSD 2.2.8
 *		NetBSD 1.5.1
 *		OpenBSD 3.0
 *		Linux
 *		SunOS 5.6: uncomment LDFLAGS in Makefile
 *	does not work without modifications:
 *		Cygwin: need to configure ssmtp.conf
 *	does not test on:
 *		Any versions of Digital Unix
 *		Darwin
 *		Any Solaris
 *		BSD/OS 3.x
 *		FreeBSD 3.x
 *		Any Unixen on VMWare
 */
#if defined(__sun__)
#define BSD_COMP
#endif

#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <errno.h>
#include <signal.h>
#include <ctype.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <sys/select.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <netdb.h>
#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#if defined(__CYGWIN__)
#include "missing/nameser.h"
#else
#include <arpa/nameser.h>
#endif

#if defined(__bsdi__) && __GNUC_MINOR__ > 7
#include <arpa/nameser_compat.h>
#endif

#if defined(__linux__) || defined(__sun__) || defined(__CYGWIN__)
#include <sys/time.h>
#define	FD_COPY(f, t)	bcopy(f, t, sizeof(*(f)))
#endif

#include "server.h"

#define DEVNULL		"/dev/null"

int maxfd = 0;
u_short id;

u_char sendbuf[BUFSIZ];
u_char ifbuf[BUFSIZ * 50];
struct ifconf ifconf;
HEADER *hp = (HEADER *)sendbuf;
int sendsize = 0;
char hostname[BUFSIZ];		/* name to look up for rootprobe */
char myhostname[BUFSIZ];	/* the name returned by gethostname(3) */
int iteration = 0;
int verbose = 0;
int file_write = 0;
char *logfile;
time_t finish_t = 0;
time_t checkif_t = 0;
struct ifaddrs *ifap;
struct timeval current;
struct sockaddr_in rasin;
time_t last_real = 0;
struct in_addr last_real_src;

enum { ROOT, CCTLD } probetype;

struct server *servers;
int nservers;
char *mailer = MAILER;

static void sig_alrm(int);
void init(int, char **);
void do_query(void);
int send_query(struct server *);
void report(FILE *);
char *getrealaddr(char *, struct in_addr);
int is_rfc1918(struct in_addr);
void check_ifaddrs(void);
void make_query(struct server *);

#if defined(__sun__)
int getdomainname(char *, int);
int inet_aton(const char *, struct in_addr *);
#endif

static void sig_alrm(int sig)
{
	return;
}

int
main(argc, argv)
	int argc;
	char **argv;
{
	struct itimerval iv;
	pid_t pid;
	long interval;

	init(argc, argv);

	if (verbose == 0) {
		if ((pid = fork()) < 0) {
			perror("fork");
			exit(1);
		}
		if (pid == 0) {
			int fd;

			setsid();
			chdir("/");
			umask (0);
			if ((fd = open(DEVNULL, O_RDWR, 0)) != -1) {
				(void)dup2(fd, STDIN_FILENO);
				(void)dup2(fd, STDOUT_FILENO);
				(void)dup2(fd, STDERR_FILENO);
				if (fd > 2)
					(void)close (fd);
			}
		} else {
			exit(0);
		}
	}

	switch(probetype) {
	case ROOT:
		interval = INTERVAL_ROOT;
		break;
	case CCTLD:
		interval = INTERVAL_CCTLD;
		break;
	default:
		fprintf(stderr, "unexpected probe type %d\n", probetype);
		exit(1);
	}

	signal(SIGALRM, sig_alrm);
	iv.it_interval.tv_sec = interval;
	iv.it_interval.tv_usec = 0;
	iv.it_value = iv.it_interval;
	if (setitimer(ITIMER_REAL, &iv, NULL) < 0) {
		perror("setitimer");
		exit(1);
	}

	while (1) {
		do_query();
		pause();
	}
}

void
init(argc, argv)
	int argc;
	char **argv;
{
	int i;
	struct server *sp;
	struct timeval timeval;
	char mydomainname[BUFSIZ];
	char *command, *c0;
	struct stat sbuf;

	command = argv[0];
	while ((c0 = strchr(command, '/')) != NULL) {
		command = c0 + 1; /* we assume argv[0] is terminated by null */
	} 
	if (strcmp(command, "cctldprobe") == 0 || strcmp(command, "cctldprobe.exe") == 0)
		probetype = CCTLD;
	else
		probetype = ROOT;

	switch(probetype) {
	case ROOT:
		servers = rootservers;
		nservers = sizeof(rootservers) / sizeof(rootservers[0]);
		break;
	case CCTLD:
		servers = cctldservers;
		nservers = sizeof(cctldservers) / sizeof(cctldservers[0]);
		break;
	}

	while ((i = getopt(argc, argv, "vw:")) != -1) {
		switch (i) {
	        case 'w':
			file_write = 1;
		        logfile = optarg;
			break;
		case 'v':
			verbose = 1;
			break;
		case '?':
			exit(1);
		default:
		}
	}
	argc -= optind;
	argv += optind;
	finish_t = time(0) + MAX_DURATION * 24 * 3600;
	if (argc > 0 && isdigit((int)*argv[0])) {
		if (strcmp(argv[0], "0") == 0)
			finish_t = 0;	/* infinite */
		else
			finish_t = time(0) + atoi(argv[0]) * 24 * 3600;
	}

	for (i = 0; i < nservers; i++) {
		sp = &servers[i];
		sp->dst.sin_family = AF_INET;
#if !defined(__linux__) && !defined(__sun__) && !defined(__CYGWIN__)
		sp->dst.sin_len = sizeof(struct sockaddr_in);
#endif
		sp->dst.sin_port = htons(DNSPORT);
		inet_aton(sp->serveraddr, &sp->dst.sin_addr);
	}
	gettimeofday(&timeval, NULL);
	srandom(timeval.tv_sec ^ timeval.tv_usec);
	id = random() & 0xffff;

	if (gethostname(myhostname, sizeof(myhostname)) < 0)
		goto usedefault;

	if (probetype == ROOT) {
		char *cp;

		strcpy(hostname, myhostname);
		/* if the hostname contains trailing dots, remove them */
		cp = hostname + strlen(hostname);
		while (cp >= hostname && (*cp == '\0' || *cp == '.')) {
			*cp = '\0';
			cp--;
		}
		/* avoid using an empty hostname */
		if (hostname[0] == '\0')
			goto usedefault;
		/* check if the hostname contains at least two labels */
		if (strchr(hostname, '.') == NULL) {
			if (getdomainname(mydomainname,
					  sizeof(mydomainname)) < 0) {
				goto usedefault;
			}
			/*
			 * check if the domain name contains at least
			 * one label.
			 */
			if ((strlen(mydomainname) > 1 &&
			     mydomainname[0] != '.') &&
			    strlen(hostname) + strlen(mydomainname) + 1 <
			    sizeof(hostname)) {
				strcat(hostname, ".");
				strcat(hostname, mydomainname);
			}
		}

		/*
		 * if we can still not find a good host name,
		 * use the default.
		 * (don't accept "mydomain" or "localdomain")
		 */
	  usedefault:
		if (hostname[0] == '\0' ||
		    (cp = strrchr(hostname, '.')) == NULL ||
		    strstr(cp, "domain") != NULL ||
		    strstr(cp, "(none)") != NULL)
			strcpy(hostname, DEFAULT_HOSTNAME);
		if (verbose)
			printf("hostname to look up: %s\n", hostname);
	}
	bzero(&rasin, sizeof(struct sockaddr_in));
#if !defined(__linux__) && !defined(__sun__) && !defined(__CYGWIN__)
	rasin.sin_len = sizeof(struct sockaddr_in);
#endif
	rasin.sin_family = AF_INET;
	rasin.sin_port = htons(REALADDRPORT);
	if (inet_aton(REALADDRSERV, &rasin.sin_addr) == 0) {
		perror("inet_aton");
		exit(1);
	}

	/* check the mailer */
	if (stat(mailer, &sbuf) < 0) {
		if (stat("/bin/mail", &sbuf) < 0) {
			fprintf(stderr, "can't find %s\n", MAILER);
			exit(1);
		} else
			mailer = "/bin/mail";
	}
	if (verbose)
		printf("mailer: %s\n", mailer);
}

void
make_query(sp)
	struct server *sp;
{
	int i;
	u_char *bp;
	char *np, *p;
	u_short qtype, qclass;

	if (probetype == ROOT && sendsize > 0)
		return;		/* initialize done, nothing to do any more. */
	
	memset(sendbuf, 0, sizeof(sendbuf));
	hp->opcode = QUERY;
	hp->rd = 1;
	hp->qdcount = htons(1);
	
	bp = &sendbuf[sizeof(HEADER)];

	if (probetype == ROOT) {
		np = hostname;
	} else {
		np = sp->servername;
	}
	while (*np) {
		p = index(np, '.');
		if (p == NULL) {
			*bp++ = strlen(np);
			while (*np) {
				*bp++ = (u_char)*np++;
			}
		} else {
			*p++ = '\0';
			*bp++ = strlen(np);
			while (*np) {
				*bp++ = (u_char)*np++;
			}
			np++;
		}
	}
	*bp++ = '\0';
	if (probetype == ROOT)
		qtype = htons(T_A);
	else
		qtype = htons(T_SOA);
	memcpy(bp, &qtype, sizeof(qtype));
	bp += sizeof(qtype);
	qclass = htons(C_IN);
	memcpy(bp, &qclass, sizeof(qclass));
	bp += sizeof(qclass);
	sendsize = bp - sendbuf;
	if (verbose) {
		for (i = 0, bp = sendbuf; i < sendsize; i++, bp++)
			printf("%02x ", *bp);
		printf("\n");
	}
}

void
do_query(void)
{
	int i, n, outstanding, success;
	struct timeval timeval, timeout;
	struct server *sp, *sp_next;
	fd_set rfd, rfd0;
	u_char recvbuf[BUFSIZ];
	char buf[BUFSIZ];
	FILE *fp;
	HEADER *hp;

	gettimeofday(&current, NULL);
	if (current.tv_sec > PROJECT_FINISH)
		exit(0);

	FD_ZERO(&rfd0);
	/* XXX: should we randomize the starting server? */
	for (i = 0; i < nservers; i++) {
		sp = &servers[i];
		if (i != 0) {
			/* add a small random delay */
			current.tv_sec +=
				random() % (QI_MAX - QI_MIN) + QI_MIN;
			current.tv_usec += random() % 1000000;
			if (current.tv_usec >= 1000000) {
				current.tv_usec -= 1000000;
				current.tv_sec++;
			}
		}
		sp->tv = current;
		sp->state = S_TOSEND;
		sp->so = -1;	/* we do delayed open */
		sp->realsrc[0] = '\0';
		sp->id = id++;
	}

	i = 0;
	outstanding = 0;
	success = 0;

	while (1) {
		sp_next = NULL;
		timeval.tv_sec = 0x7fffffff;
		timeval.tv_usec = 999999;
		/* search the next event */
		for (i = 0; i < nservers; i++) {
			sp = &servers[i];
			if (sp->state == S_TOSEND || sp->state == S_TORECV) {
				if (verbose && sp->so >= 0)
					printf("(%s)     %ld (%d)",
						sp->servername, sp->tv.tv_sec,
						sp->state);
				if (timeval.tv_sec > sp->tv.tv_sec ||
				    (timeval.tv_sec == sp->tv.tv_sec &&
				     timeval.tv_usec > sp->tv.tv_usec)) {
					if (verbose && sp->so >= 0)
						printf(": min");
					timeval = sp->tv;
					sp_next = sp;
				}
				if (verbose && sp->so >= 0)
					printf("\n");
			}
		}
		if (verbose) {
			if (sp_next != NULL)
				printf("next event is for %s (%d) (outstanding %d)\n",
		 			sp_next->servername, sp_next->state,
					outstanding);
			else
				printf("sp_next NULL (outstanding %d)\n",
					outstanding);
		}
		if (sp_next == NULL && outstanding == 0)
			break;
		gettimeofday(&current, NULL);
		if (current.tv_sec > sp_next->tv.tv_sec ||
		    (current.tv_sec == sp_next->tv.tv_sec &&
		     current.tv_usec > sp_next->tv.tv_usec)) {
			/* fire the event */
			if (sp_next->state == S_TOSEND) {
				/* send a packet */
				if (send_query(sp_next) < 0)
					continue;
				sp_next->tv_sent = sp_next->tv = current;
				sp_next->tv.tv_sec += MAX_TIMEOUT;
				FD_SET(sp_next->so, &rfd0);
				sp_next->state = S_TORECV;
				outstanding++;
				continue;
			} else if (sp_next->state == S_TORECV) {
				/* timeout */
				outstanding--;
				FD_CLR(sp_next->so, &rfd0);
				if (verbose)
					printf("timeout for %s\n",
						sp_next->servername);
				shutdown(sp_next->so, 2);
				close(sp_next->so);
				sp_next->so = -1;
				sp_next->state = S_TIMEDOUT;
				continue;
			}
		}
		timeout.tv_sec = sp_next->tv.tv_sec - current.tv_sec;
		timeout.tv_usec = sp_next->tv.tv_usec - current.tv_usec;
		if (sp_next->tv.tv_usec < current.tv_usec) {
			timeout.tv_usec += 1000000;
			timeout.tv_sec--;
		}
		if (verbose)
			printf("timeout value (%ld.%06ld)\n",
				timeout.tv_sec, timeout.tv_usec);
		FD_COPY(&rfd0, &rfd);
		if ((n = select(maxfd + 1, &rfd, NULL, NULL, &timeout)) < 0) {
			perror("select");
		}
		gettimeofday(&current, NULL);
		for (i = 0; i < nservers; i++) {
			sp = &servers[i];
			if (sp->so >= 0 && FD_ISSET(sp->so, &rfd)) {
				if ((n = recvfrom(sp->so, recvbuf,
					sizeof(recvbuf), 0, NULL, 0)) < 0) {
					perror("recvfrom");
				}
				FD_CLR(sp->so, &rfd0);
				close(sp->so);
				hp = (HEADER *)recvbuf;
				sp->s_rcode = hp->rcode;
				sp->so = -1;
				sp->state = S_RECVD;
				outstanding--;
				if (current.tv_usec < sp->tv_sent.tv_usec) {
					sp->rtt = (current.tv_sec -
						   sp->tv_sent.tv_sec - 1)
						  * 1000 +
						  (1000000 + current.tv_usec -
						   sp->tv_sent.tv_usec) / 1000;
				} else {
					sp->rtt = (current.tv_sec -
						   sp->tv_sent.tv_sec) * 1000 +
						  (current.tv_usec -
						   sp->tv_sent.tv_usec) / 1000;
				}
				success++;
				if (verbose)
					printf("received from %s (%ld ms)\n",
						sp->servername, sp->rtt);
			}
		}
	}

	if (verbose)
		report(stderr);

	if (verbose == 0 && success > 0) {
		char *mode = NULL;

		switch(probetype) {
		case ROOT:
			mode = "root";
			break;
		case CCTLD:
			mode = "cctld";
			break;
		}


		if (file_write) {
			sprintf(buf, "\"%s probe by %s\" %s",
				mode, myhostname, REPORTADDR);
			if ((fp = fopen (logfile, "a")) == NULL) {
				perror("fopen");
				exit(1);
			}
			fprintf(fp, "%s\n", buf);
		} else {
			sprintf(buf, "%s -s \"%s probe by %s\" %s",
				mailer, mode, myhostname, REPORTADDR);
			if ((fp = popen(buf, "w")) == NULL) {
				perror("popen");
				exit(1);
			}
		}

		report(fp);
		if (file_write)
			fclose(fp);
		else
			pclose(fp);
	}

	if (finish_t != 0 && time(0) > finish_t) {
		sleep(30);
		exit(0);
	}
}

void
report(fp)
	FILE *fp;
{
	int i;
	struct server *sp;

	for (i = 0; i < nservers; i++) {
		sp = &servers[i];
		if (*sp->realsrc == '\0') {
			fprintf(fp, "%ld %s %s %s",
				sp->tv_sent.tv_sec, inet_ntoa(sp->src.sin_addr),
				sp->ifname, sp->servername);
			if (probetype != ROOT)
				fprintf(fp, "(%s)", sp->serveraddr);
			fprintf(fp, ": ");
		} else {
			fprintf(fp, "%ld #%s %s %s",
				sp->tv_sent.tv_sec, sp->realsrc,
				sp->ifname, sp->servername);
			if (probetype != ROOT)
				fprintf(fp, "(%s)", sp->serveraddr);
			fprintf(fp, ": ");
		}
		switch (sp->state) {
		case S_RECVD:
			fprintf(fp, "rtt %ld ms", sp->rtt);
			switch (sp->s_rcode) {
			case NOERROR:
				fprintf(fp, "\n");
				break;	/* do nothing */
			case FORMERR:
				fprintf(fp, " FORMERR\n");
				break;
			case SERVFAIL:
				fprintf(fp, " SERVFAIL\n");
				break;
			case NXDOMAIN:
				fprintf(fp, " NXDOMAIN\n");
				break;
			case NOTIMP:
				fprintf(fp, " NOTIMP\n");
				break;
			case REFUSED:
				fprintf(fp, " REFUSED\n");
				break;
#ifdef	YXDOMAIN
			case YXDOMAIN:
				fprintf(fp, " YXDOMAIN\n");
				break;
#endif
#ifdef	YXRRSET
			case YXRRSET:
				fprintf(fp, " YXRRSET\n");
				break;
#endif
#ifdef	NXRRSET
			case NXRRSET:
				fprintf(fp, " NXRRSET\n");
				break;
#endif
#ifdef	NOTAUTH
			case NOTAUTH:
				fprintf(fp, " NOTAUTH\n");
				break;
#endif
#ifdef	NOTZONE
			case NOTZONE:
				fprintf(fp, " NOTZONE\n");
				break;
#endif
			default:
				fprintf(fp, " unknown(%d)\n", sp->s_rcode);
			}
			break;
		case S_TIMEDOUT:
			fprintf(fp, "timed out (%dsec)\n", MAX_TIMEOUT);
			break;
		case S_ERROR:
			fprintf(fp, "error: %s\n", strerror(sp->s_errno));
			break;
		default:
			fprintf(fp, "unknown\n");
		}
	}
}

int
send_query(sp)
	struct server *sp;
{
	struct ifreq *ifrp;
	struct sockaddr_in *sin;
	u_char *p;
	int socklen;

	if (sp->so < 0 && (sp->so = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror("socket");
		exit(1);
	}
	if (sp->so > maxfd)
		maxfd = sp->so;

	if (connect(sp->so, (struct sockaddr *)&sp->dst,
		sizeof(struct sockaddr_in)) < 0) {
		perror("connect");
		exit(1);
	}
	socklen = sizeof(struct sockaddr_in);
	if (getsockname(sp->so, (struct sockaddr *)&sp->src, &socklen) < 0) {
		perror("getsockname");
		exit(1);
	}
	check_ifaddrs();
	ifrp = NULL;
	for (p = ifbuf; p < &ifbuf[ifconf.ifc_len]; ) {
		ifrp = (struct ifreq *)p;
#if defined(__linux__) || defined(__sun__) || defined(__CYGWIN__)
		p += sizeof(struct ifreq);
#else
		p += IFNAMSIZ + ifrp->ifr_addr.sa_len;
#endif

#if defined(__CYGWIN__)
		if (ifrp->ifr_addr.sa_family != AF_INET)
#else
		if (ifrp->ifr_addr.sa_family != AF_INET || ifrp->ifr_dstaddr.sa_family != AF_INET)
#endif
			continue;
		sin = (struct sockaddr_in *)&ifrp->ifr_addr;
		if (sin->sin_addr.s_addr == sp->src.sin_addr.s_addr) {
			p = NULL;
			break;
		}
	}
	sp->ifname[0] = '\0';
	if (p == NULL && ifrp != NULL) {
		strncpy(sp->ifname, ifrp->ifr_name, IFNAMSIZ);
	}
	sp->s_rcode = 0;
	sp->s_errno = 0;

	if (is_rfc1918(sp->src.sin_addr))
		getrealaddr(sp->realsrc, sp->src.sin_addr);

	if (verbose)
		printf("sending to %s through %s\n",
			sp->servername, sp->ifname);

	make_query(sp);

	gettimeofday(&current, NULL);

	if (send(sp->so, sendbuf, sendsize, 0) < 0) {
		perror("send");
		sp->s_errno = errno;
		sp->state = S_ERROR;
		shutdown(sp->so, 2);
		close(sp->so);
		sp->so = -1;
		return -1;
	}
	return 0;
}

char *
getrealaddr(p, src)
	char *p;
	struct in_addr src;
{
	static char buf[BUFSIZ] = { '\0', };
	int rs, err;

	if (src.s_addr == last_real_src.s_addr) {
		gettimeofday(&current, NULL);
		if (current.tv_sec - last_real < CHECKINTERVAL) {
			strcpy(p, buf);
			return buf;
		}
	}
	last_real_src.s_addr = src.s_addr;
	if ((rs = socket(AF_INET, SOCK_STREAM, 0)) < 0)
		return strerror(errno);
	if (connect(rs, (struct sockaddr *)&rasin,
		sizeof(struct sockaddr_in)) < 0) {
		err = errno;
		close(rs);
		return strerror(err);
	}
	if (recv(rs, buf, sizeof(buf), 0) < 0) {
		err = errno;
		close(rs);
		return strerror(err);
	}
	buf[IPADDRLEN - 1] = '\0';	/* truncate if necessary */
	strcpy(p, buf);
	close(rs);
	return buf;
}

int
is_rfc1918(addr)
	struct in_addr addr;
{
	u_char *p, *q;

	p = (u_char *)&addr;
	q = p + 1;
	if (*p == 10)
		return 1;
	if (*p == 172 && *q >= 16  && *q < 32)
		return 1;
	if (*p == 192 && *q == 168)
		return 1;
	return 0;
}

void
check_ifaddrs()
{
	int s;

	if (current.tv_sec - checkif_t < CHECKINTERVAL)
		return;

	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror("socket");
		exit(1);
	}
	ifconf.ifc_len = sizeof(ifbuf);
	ifconf.ifc_buf = ifbuf;
	if (ioctl(s, SIOCGIFCONF, &ifconf) < 0) {
		perror("ioctl:SIOCGIFCONF");
		exit(1);
	}
	close(s);
	checkif_t = current.tv_sec;
}
