/*
 * Copyright (c) 2009 Nhat Minh Lê <nhat.minh.le@huoc.org>
 *
 * Permission to use, copy, modify, and/or distribute this software
 * for any purpose with or without fee is hereby granted, provided
 * that the above copyright notice and this permission notice appear
 * in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/* LINTLIBRARY */

#include <assert.h>
#include <stdlib.h>
#include <string.h>

#ifndef __NetBSD__
#include <regxml/compat.h>
#endif
#include <regxml/buffer.h>
#include <regxml/regxml.h>

#define REGXML_BUFFER_DEFAULT 1024

int
regxml_buffer_init(struct regxml_buffer *buf, size_t n)
{
	_DIAGASSERT(buf != NULL);

	if (n == 0)
		n = REGXML_BUFFER_DEFAULT;

	buf->rxb_base = malloc(n * sizeof *buf->rxb_base);
	if (buf->rxb_base == NULL)
		return REGXML_ESYSTEM;

	buf->rxb_index = 0;
	buf->rxb_size = n;

	return 0;
}

void
regxml_buffer_free(struct regxml_buffer *buf)
{
	_DIAGASSERT(buf != NULL);

	free(buf->rxb_base);
}

int
regxml_buffer_appendraw(struct regxml_buffer *buf, const char *s, size_t len)
{
	char *new;
	size_t n;

	_DIAGASSERT(buf != NULL);
	_DIAGASSERT(s != NULL);

	if (len == 0)
		return 0;

	if (buf->rxb_size - buf->rxb_index < len + 1) {
		for (n = buf->rxb_size; n - buf->rxb_index < len + 1; n *= 2)
			continue;
		new = realloc(buf->rxb_base, n * sizeof *buf->rxb_base);
		if (new == NULL)
			return REGXML_ESYSTEM;
		buf->rxb_base = new;
		buf->rxb_size = n;
	}

	(void)memcpy(buf->rxb_base + buf->rxb_index, s, len);
	buf->rxb_index += len;
	buf->rxb_base[buf->rxb_index] = '\0';

	return 0;
}

int
regxml_buffer_append(struct regxml_buffer *buf, const char *s)
{
	return regxml_buffer_appendraw(buf, s, strlen(s));
}

int
regxml_buffer_strdup(struct regxml_buffer *buf, char **sptr)
{
	char *s;

	_DIAGASSERT(buf != NULL);
	_DIAGASSERT(sptr != NULL);

	s = strdup(buf->rxb_base);
	if (s == NULL)
		return REGXML_ESYSTEM;
	*sptr = s;
	return 0;
}
