/*
 * Ppower daemon.  Handles the x10 hardware and client requests.
 * Copyright (C) 1999  Steven Brown
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * Steven Brown <swbrown@ucsd.edu>
 *
 * $Id: ppowerd.c,v 1.12 2000/02/06 01:25:42 swbrown Exp $
 */

#include <config.h>
#include <stdio.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/poll.h>
#include <errno.h>
#include <signal.h>
#include "error.h"
#include "x10.h"
#include "read.h"
#include "socket.h"
#include "ppowerd.h"
#include "conf.h"
#include "conf-proto.h"
#include "error-proto.h"
#include "pid.h"
#include "snprintf.h"

/* Local prototypes. */
static void ppowerd_show_help(void);
static void ppowerd_show_version(void);
static void ppowerd_transmit_command(X10 *x10, Client_Command *client_command);
static void ppowerd_exit(int sig);
static void ppowerd_wait(int sig);
static void ppowerd_refresh(int sig);
static void setsignal(int sig, void (*handler)(int));

/* Global things. */

/* Name of the program. */
char *progname;

/* All our fd's in poll() form. */
struct pollfd pollfd[POLL_MAX_COUNT];

/* The number of user monitor fd's we have open in the pollfd structure. */
int user_monitor_sockets=0;

/* Name of the conf file. */
char *conf_name=CONFIG_FILE;


/* Locals. */

/* Sockets used by the daemon. */
static int daemon_monitor_socket;
static int daemon_socket;

/* Location of the pid file. */
static char pid_buffer[PATH_MAX];

/* Commandline options. */
static struct option long_options[] = {
	{"version", 0, 0, 'v'},
	{"help", 0, 0, 'h'},
	{"conf", 1, 0, 'f'},
	{"debug", 1, 0, 'd'},
	{"no-background", 0, 0, 'n'},
	{0, 0, 0, 0}
};
#define SHORT_OPTIONS "vhf:d:n"


/* Main... */
int main(int argc, char *argv[]) {
	X10 *x10;
	int retval;
	int user_socket;
	int i;
	Client_Command client_command;
	int longindex;
	int optchar;
	int no_background=0;
	
	/* Save the name of the program. */
	progname=argv[0];
	
	/* Parse the arguments. */
	while((optchar=getopt_long(argc, argv, SHORT_OPTIONS, long_options, &longindex)) != EOF) {
		
		/* Handle each argument. */
		switch(optchar) {
			
			/* Was it a long option? */
			case 0:
				
				/* Hrmm, something we don't know about? */
				fatal("Unhandled long getopt option '%s'.", long_options[longindex].name);
			
			/* If it was an error, exit right here. */
			case '?':
				exit(1);
			
			/* Was it a version request? */
			case 'v':
				ppowerd_show_version();
				exit(0);
			
			/* Was it a help request? */
			case 'h':
				ppowerd_show_help();
				exit(0);
			
			/* Was it an alternate conf file? */
			case 'f':
				
				/* Save the alternate conf file name. */
				conf_name=strdup(optarg);
				if(!conf_name) {
					fatal("Out of memory.");
				}
				
				break;
			
			/* Was it a no-backgrounding request? */
			case 'n':
				
				/* Mark that we shouldn't background. */
				no_background=1;
				
				break;
				
			/* Was it a debug level set? */
			case 'd':
				
				/* Save the value. */
				debuglvl=strtol(optarg, NULL, 10);
				if((debuglvl == LONG_MIN || debuglvl == LONG_MAX) && errno) {
					fatal("Invalid debug level.");
				}
				if(debuglvl < 0 || debuglvl > DEBUG_MAX) {
					fatal("Invalid debug level.");
				}
				
				break;
			
			/* It was something weird.. */
			default:
				panic("Unhanlded getopt return value %i.", optchar);
		}
	}
	
	/* If there were any extra arguments, we should complain. */
	if(optind < argc) {
		fatal("Extra argument on commandline, '%s'.", argv[optind]);
	}
	
	/* Parse our conf file for settings. */
	conf_parse(conf_name);
	
	/* Create the path of the pid file. */
	if(snprintf(pid_buffer, PATH_MAX, "%s/%s", conf_daemon_dir, DAEMON_PID_FILE) == -1) {
		fatal("Pid file path too long.");
	}
	
	/* Make sure we are not already running (.pid file check). */
	if(pid_read(pid_buffer) != -1) {
		fatal("Ppowerd is already running.");
	}
	
	/* Free the stale daemon sockets if we have them. */
	(void) unlink(conf_daemon_monitor_socket_path);
	(void) unlink(conf_daemon_socket_path);
	
	/* Create the daemon socket used for requests. */
	debug(DEBUG_STATUS, "Creating daemon socket '%s'.", conf_daemon_socket_path);
	daemon_socket=socket_create(conf_daemon_socket_path, conf_daemon_socket_mode, conf_daemon_socket_uid, conf_daemon_socket_gid);
	
	/* Create the daemon socket used for monitoring. */
	debug(DEBUG_STATUS, "Creating daemon monitor socket '%s'.", conf_daemon_monitor_socket_path);
	daemon_monitor_socket=socket_create(conf_daemon_monitor_socket_path, conf_daemon_monitor_socket_mode, conf_daemon_monitor_socket_uid, conf_daemon_monitor_socket_gid);
	
	/* Open the x10 device. */
	debug(DEBUG_STATUS, "Opening tty %s.", conf_tty);
	x10=x10_open(conf_tty);
	
	/* Save poll information for the three default fd's we handle. */
	pollfd[POLL_X10].fd=x10->fd;
	pollfd[POLL_X10].events=POLLIN;
 	pollfd[POLL_DAEMON].fd=daemon_socket;
	pollfd[POLL_DAEMON].events=POLLIN;
	pollfd[POLL_DAEMON_MONITOR].fd=daemon_monitor_socket;
	pollfd[POLL_DAEMON_MONITOR].events=POLLIN;
	
	/* Fork into the background. */
	if(!no_background) {
		debug(DEBUG_STATUS, "Forking into background.");
		
		/* Close all our tty's fd's.  We can't use them anymore. */
		close(0);
		close(1);
		close(2);
		
		/* 
		 * Inform error code that we're backgrounded, don't display
		 * stuff to the terminal anymore.  Use syslog.
		 */
		errors_to_syslog(1);
		
		/* Fork and exit the parent. */
		if(fork()) exit(0);
		
		/* The child creates a new session. */
		if(setsid() == -1) {
			fatal("Could not setsid().");
		}
	}
	
	/* We are now running.  Save the pid and be ready to cleanup on signal. */
	debug(DEBUG_STATUS, "Creating pid file and hooking termination signals.");
	if(pid_write(pid_buffer, getpid()) != 0) {
		fatal("Could not write pid file '%s'.", pid_buffer);
	}
	setsignal(SIGHUP, ppowerd_refresh);
	setsignal(SIGINT, ppowerd_exit);
	setsignal(SIGQUIT, ppowerd_exit);
	setsignal(SIGTERM, ppowerd_exit);
	setsignal(SIGCHLD, ppowerd_wait);
	
	/* We loop forever handling input and output. */
	for(;;) {
		
		/* 
		 * Wait for input to be available. 
		 */
		debug(DEBUG_STATUS, "Waiting for events.");
		retval=poll(pollfd, POLL_DEFAULT_COUNT + user_monitor_sockets, -1);
		if(retval == -1) {
			
			/* If it's just interruption, restart the poll. */
			if(errno == EINTR) {
				debug(DEBUG_EXPECTED, "Signal received in poll, restarting poll.");
				continue;
			}
			
			/* Nope, poll broke. */
			fatal("Poll failed.");
		}
		debug(DEBUG_STATUS, "Woken for events.");
		
		/* Was this from the x10? */
		if(pollfd[POLL_X10].revents) {
			debug(DEBUG_STATUS, "X10 wants attention.");
			
			/* Handle the x10's command. */
			read_x10(x10);
		}
		
		/* 
		 * Was it a new client connection?
		 */
		if(pollfd[POLL_DAEMON].revents) {
			debug(DEBUG_STATUS, "Accepting user socket connection.");
			
			/* Accept the user connection. */
			user_socket=accept(pollfd[POLL_DAEMON].fd, NULL, NULL);
			if(user_socket == -1) {
				debug(DEBUG_UNEXPECTED, "Could not accept user socket.");
				continue;
			}
			
			/* The socket needs to be non-blocking. */
			if(fcntl(user_socket, F_SETFL, O_NONBLOCK) == -1) {
				fatal("Could not set user socket to non-blocking.");
			}
			
			/* Read the request from the client. */
			if(!socket_read(user_socket, &client_command, sizeof(Client_Command), USER_READ_TIMEOUT)) {
				debug(DEBUG_UNEXPECTED, "Gave up waiting for user command.");
				
				/* We timed out, close the socket and continue. */
				if(close(user_socket) != 0) {
					debug(DEBUG_UNEXPECTED, "Problem closing timed-out user command socket.");
				}
				continue;
			}
			
			/* Was it a transmit command request? */
			if(client_command.request == REQUEST_COMMAND) {
				
				/* Send this command to the x10 hardware. */
				ppowerd_transmit_command(x10, &client_command);
			}
			
			/* It was a wacky client command.. */
			else {
				debug(DEBUG_UNEXPECTED, "Weird client command received, '%i'.", client_command.request);
			}

			/* Close the user's socket. */
			if(close(user_socket) != 0) {
				debug(DEBUG_UNEXPECTED, "Problem closing completed user command socket.");
			}
			debug(DEBUG_STATUS, "User socket closed.");
		}
		
		/* Are we closing a user monitor? */
		for(i=user_monitor_sockets; i > 0; i--) {
			
			/* Did this one disconnect? */
			if(pollfd[POLL_USER_MONITOR + (i - 1)].revents) {
				debug(DEBUG_STATUS, "Disconnecting user monitor socket.");
				
				/* Close the user monitor socket. */
				if(close(pollfd[POLL_USER_MONITOR + (i - 1)].fd) != 0) {
					fatal("Could not close user monitor socket.");
				}
				
				/* Shorten the pollfd list to remove our spot. */
				(void) memmove(&pollfd[POLL_USER_MONITOR + (i - 1)], &pollfd[POLL_USER_MONITOR + i], (user_monitor_sockets - i) * sizeof(struct pollfd));
				
				/* We have one less now. */
				user_monitor_sockets--;
			}
		}
		
		/* Are we adding a new user monitor? */
		if(pollfd[POLL_DAEMON_MONITOR].revents) {
			debug(DEBUG_STATUS, "Accepting user monitor socket connection.");
			
			/* Accept the user monitor connection. */
			retval=accept(pollfd[POLL_DAEMON_MONITOR].fd, NULL, NULL);
			if(retval == -1) {
				fatal("Could not accept user monitor socket.");
			}
			
			/* If there are too many monitor connections, drop it. */
			if(user_monitor_sockets >= DAEMON_MONITOR_MAX) {
				debug(DEBUG_EXPECTED, "Dropping extra monitor connection.");
				
				/* Close the socket. */
				if(close(retval) != 0) {	
					fatal("Could not close user monitor socket.");
				}
				
				/* 
				 * Note that if we continue here, we must have nothing
				 * important to do after this as if someone floods the
				 * daemon with connections, it wouldn't ever process
				 * anything.
				 */
				continue;
			}
			
			/* The socket needs to be non-blocking. */
			if(fcntl(retval, F_SETFL, O_NONBLOCK) == -1) {
				fatal("Could not set monitor to non-blocking.");
			}
			
			/* Save this socket. */
			pollfd[POLL_USER_MONITOR + user_monitor_sockets].fd=retval;
			pollfd[POLL_USER_MONITOR + user_monitor_sockets].events=POLLIN;
			pollfd[POLL_USER_MONITOR + user_monitor_sockets].revents=0;
			user_monitor_sockets++;
		}
	}
}


/* Show the help screen for ppowerd. */
static void ppowerd_show_help(void) {
	printf("'ppowerd' is the daemon that controlls the cm11a computer interface to the\n");
	printf("x10 hardware.  It takes commands to give to the cm11a as well as performing\n");
	printf("actions when the x10 hardware enters a specified state.  It can also pass\n");
	printf("the information it gets from the cm11a to the user.\n");
	printf("\n");
	printf("Usage: %s [OPTION]...\n", progname);
	printf("\n");
	printf("  -v, --version           display program version\n");
	printf("  -h, --help              give help on usage\n");
	printf("  -f, --conf=FILE	  specify alternate configuration file\n");
	printf("  -n, --no-background     do not fork into the background\n");
	printf("  -d, --debug=LEVEL       set the debug level, 0 is off, the\n");
	printf("                          compiled in default is %i and the max\n", DEBUGLVL);
	printf("                          level allowed is %i\n", DEBUG_MAX);
	printf("\n");
	printf("Report bugs to <swbrown@ucsd.edu>.\n");
	return;
}


/* Show the version info for ppowerd. */
static void ppowerd_show_version(void) {
	printf("ppowerd (%s) %s\n", PACKAGE, VERSION);
}


/* Transmit a client command to the x10 hardware. */
static void ppowerd_transmit_command(X10 *x10, Client_Command *client_command) {
	int i;
	unsigned char buffer[2];
	
	/* Transmit all the addresses for devicecodes. */
	for(i=0; i < client_command->devices; i++) {
		debug(DEBUG_ACTION, "Addressing housecode 0x%02x, devicecode 0x%02x.", client_command->housecode, client_command->device[i]);
		
		/* Set the header byte to be an address. */
		buffer[0]=HEADER_DEFAULT;
		
		/* The second byte is the house:device. */
		buffer[1]=(client_command->housecode << 4) | client_command->device[i];
		
		/* Write the address to the x10 hardware. */
		if(!x10_write_message(x10, &buffer, 2)) {
			debug(DEBUG_UNEXPECTED, "Couldn't set address.");
		}
	}
	
	
	/* Send the command to the x10 hardware. */
	
	/* Set the header to be a function. */
	buffer[0]=HEADER_DEFAULT | HEADER_FUNCTION;
	
	/* If this is a dim/bright, the header also has the value. */
	if(client_command->command == COMMAND_DIM || client_command->command == COMMAND_BRIGHT) {
		buffer[0]=(buffer[0] | client_command->value << 3) & 0xff;
	}
		
	/* The second byte is house:command. */
	buffer[1]=(client_command->housecode << 4) | client_command->command;
	
	/* Send the message to the x10 hardware. */
	if(!x10_write_message(x10, &buffer, 2)) {
		debug(DEBUG_UNEXPECTED, "Couldn't send function.");
	}
	
	return;
}


/* Refresh the configuration. */
void ppowerd_refresh(int sig) {
	fatal("Refresh not implemented yet.");
}


/* Cleanup and exit. */
void ppowerd_exit(int sig) {
	debug(DEBUG_STATUS, "Cleaning up for exit.");
	
	/* Close the daemon sockets. */
	close(daemon_monitor_socket);
	close(daemon_socket);
	
	/* *** Close all the monitor sockets. */
	
	/* Unlink the two filesystem-bound sockets if they exist. */
	(void) unlink(conf_daemon_socket_path);
	(void) unlink(conf_daemon_monitor_socket_path);
	
	/* Unlink the pid file if we can. */
	(void) unlink(pid_buffer);
	
	/* Flush conf memory. */
	conf_free();
	
	/* And exit.. */
	exit(0);
}


/* Wait for child processes to finish up. */
void ppowerd_wait(int sig) {
	pid_t pid;
	
	/* Wait for any child processes to finish up. */
	while((pid=waitpid(-1, NULL, WNOHANG)) != 0) {
		
		/* Did waitpid return an error? */
		if(pid == -1) {
			
			/* 
			 * If it's a 'no children' error, that's fine and
			 * we're done.
			 */
			if(errno == ECHILD) break;
			
			/* Nope, it was truly fatal. */
			else {
				fatal("waitpid() failed: %s", strerror(errno));
			}
		}
		
		/* A child process has finished. */
		debug(DEBUG_ACTION, "Child pid %i finished.", pid);
	}
	
	return;
}


/* Sets the handler to call for a signal. */
static void setsignal(int sig, void (*handler)(int)) {
	struct sigaction sigact;
	
	/* Fill the sigaction structure. */
	sigact.sa_handler=handler;
	sigact.sa_flags=0;
	if(sigemptyset(&sigact.sa_mask) != 0) {
		fatal("sigemptyset() failed.");
	}
	
	/* Set the signal's handler using sigaction. */
	if(sigaction(sig, &sigact, NULL) != 0) {
		fatal("sigaction() failed.");
	}
	
	return;	
}

