/*
 * Error handlers for the daemon and client.
 * Copyright (C) 1999  Steven Brown
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * Steven Brown <swbrown@ucsd.edu>
 *
 * $Id: error.c,v 1.4 2000/02/06 01:25:42 swbrown Exp $
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <assert.h>
#include <syslog.h>
#include "error.h"

/* Name of the program. */
extern char *progname;

/* Debugging level. */
int debuglvl=DEBUGLVL;

/* Errors to syslog toggle. */
int use_syslog = 0;

/* Buffer to construct messages in and it's size. */
#define BUFFER_SIZE 4096
char buffer[BUFFER_SIZE];


/* Errors to syslog.  Caused errors to be sent to syslog instead of the terminal. */
void errors_to_syslog(int toggle) {
	use_syslog = toggle;
}


/* Fatal error handler. */
void fatal(char *message, ...) {
	va_list ap;
	va_start(ap, message);
	
	/* Print the error message. */
	vsnprintf(buffer, BUFFER_SIZE, message, ap);
	if(use_syslog) {
		syslog(LOG_DAEMON | LOG_ERR, "%s: %s", progname, buffer);
	}
	else {
		fprintf(stderr, "%s: %s\n", progname, buffer);
	}
	
	/* Exit with an error code. */
	va_end(ap);
	exit(1);
}


/* Normal error handler. */
void error(char *message, ...) {
	va_list ap;
	va_start(ap, message);
	
	/* Print the error message. */
	vsnprintf(buffer, BUFFER_SIZE, message, ap);
	if(use_syslog) {
		syslog(LOG_DAEMON | LOG_ERR, "%s: %s", progname, buffer);
	}
	else {
		fprintf(stderr, "%s: %s\n", progname, buffer);
	}
	
	va_end(ap);
	return;
}


/* Warning handler. */
void warn(char *message, ...) {
	va_list ap;
	va_start(ap, message);
	
	/* Print the warning message. */
	vsnprintf(buffer, BUFFER_SIZE, message, ap);
	if(use_syslog) {
		syslog(LOG_DAEMON | LOG_ERR, "%s: warning: %s", progname, buffer);
	}
	else {
		fprintf(stderr, "%s: warning: %s\n", progname, buffer);
	}
	
	va_end(ap);
	return;
}


/* Panic error handler. */
void panic(char *message, ...) {
	va_list ap;
	va_start(ap, message);
	
	/* Print the error message. */
	vsnprintf(buffer, BUFFER_SIZE, message, ap);
	if(use_syslog) {
		syslog(LOG_DAEMON | LOG_ERR, "%s: PANIC: %s", progname, buffer);
	}
	else {
		fprintf(stderr, "%s: PANIC: %s\n", progname, buffer);
	}
	
	/* Exit with an error code. */
	va_end(ap);
	exit(1);
}


/* Debugging error handler. */
void debug(int level, char *message, ...) {
	va_list ap;
	va_start(ap, message);
	
	/* We only do this code if we are at or above the debug level. */
	if(debuglvl >= level) {
	
		/* Print the error message. */
		vsnprintf(buffer, BUFFER_SIZE, message, ap);
		if(use_syslog) {
			syslog(LOG_DAEMON | LOG_ERR, "%s: (debug): %s", progname, buffer);
		}
		else {
			fprintf(stderr, "%s: (debug): %s\n", progname, buffer);
		}
	}
	
	va_end(ap);
}
