%{
/*
 * Bison parser for the configuration file.
 * Copyright (C) 1999  Steven Brown
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * Steven Brown <swbrown@ucsd.edu>
 *
 * $Id: conf-parse.y,v 1.10 1999/09/19 18:43:46 kefka Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <errno.h>
#include "conf.h"
#include "conf-proto.h"
#include "x10.h"
#include "error.h"
#include "stacklist.h"
#include "table.h"

extern int yylex(void);

#define YYERROR_VERBOSE

%}

%union {
	char *identifier;
	char *string;
	int integer;
	
	/* Elements for passing up in the stack. */
	struct device *device;
	struct alias *alias;
	struct stacklist stacklist;
	struct function function;
	struct state state;
	struct macro_commands macro_commands;
	struct macro_command macro_command;
	struct table *table;
	struct expression expression;
}

%token <identifier> IDENTIFIER
%token <string> STRING
%token MACRO
%token HMACRO
%token COMMAND
%token DEVICE
%token ALIAS
%token TTY
%token DAEMON_SOCKET
%token DAEMON_MONITOR_SOCKET
%token RUN
%token <integer> INTEGER
%token HOUSECODE
%token DAEMON_DIR
%token DAEMON_COMMAND_TIMEOUT
%token IF
%token ELSE

%token ALL_UNITS_OFF
%token ALL_LIGHTS_ON
%token ON
%token OFF
%token DIM
%token BRIGHT
%token ALL_LIGHTS_OFF
%token EXTENDED_CODE
%token HAIL_REQUEST
%token HAIL_ACKNOWLEDGE
%token PRESET_DIM1
%token PRESET_DIM2
%token EXTENDED_DATA_TRANSFER
%token STATUS_ON
%token STATUS_OFF
%token STATUS_REQUEST

%token COMMA
%token COLON
%token BRACE_LEFT
%token BRACE_RIGHT
%token PAREN_LEFT
%token PAREN_RIGHT
%token SEMI

%token LAMP
%token APPLIANCE
%token MOTION_DETECTOR
%token SIGNAL
%token TRANSCEIVER

%token EXAMPLE

%type <macro_commands> commands commands_braced
%type <macro_command> command_statement
%type <string> opt_name name
%type <state> state
%type <function> function
%type <alias> alias
%type <integer> type
%type <stacklist> device_list alias_list
%type <device> device
%type <table> state_list
%type <expression> expression

%start conf

%%

conf: declaration_statements
	| /* empty */

declaration_statements: declaration_statements declaration_statement
	| declaration_statement

declaration_statement: DEVICE device_list type {
		Stacklist_Entry *stacklist_entry;
		Device *device;
		Alias *alias;
		char *name;
		
		/* Apply the type given to each device in the list. */
		for(stacklist_entry=$2.first; stacklist_entry != NULL; stacklist_entry=stacklist_entry->previous) {
			device=(Device *) stacklist_entry->info;
			device->type=$3;
		}
		
		/* 
		 * Steal the entries from the list and add them to the
		 * devicelist.  We also need to make aliases for each so we
		 * can look them up.
		 */
		for(stacklist_entry=$2.first; stacklist_entry != NULL; stacklist_entry=$2.first) {
			device=(Device *) stacklist_entry->info;
			
			/* Remove the top one from the list. */
			stacklist_remove(&$2, stacklist_entry);
			
			/* Insert the device to the devicelist. */
			stacklist_insert(&conf_devicelist, device);
			
			/* Create an alias for this device with its name. */
			name=strdup(device->name);
			if(!name) {
				fatal("Out of memory.");
			}
			alias=conf_alias_create(name, 1, device);
			stacklist_insert(&conf_aliaslist, alias);
		}
		
		/* Free the stacklist leftovers. */
		stacklist_free(&$2);
	}
	| HOUSECODE IDENTIFIER {
		
		/* If the housecode has already been set, error. */
		if(conf_housecode != -1) {
			yyerror("HOUSECODE was already set.");
			break;
		}
		
		/* This must be a letter from 'a' to 'p'. */
		if(strlen($2) != 1) {
			yyerror("Invalid housecode, '%s'.", $2);
			break;
		}
		if((tolower($2[0]) < 'a') && (tolower($2[0]) > 'p')) {
			yyerror("Invalid housecode, '%s'.", $2);
			break;
		}
		
		/* Save its value. */
		conf_housecode=housecode_table[tolower($2[0])];
	}
	| ALIAS alias_list IDENTIFIER {
		Stacklist_Entry *stacklist_entry;
		Alias *alias;
		Alias *newalias;
		int i;
		
		/* Make sure the identifier given doesn't conflict. */
		alias=conf_aliaslist_lookup($3);
		if(alias) {
			yyerror("New alias name '%s' conflicts with previous definition.");
			break;
		}
		
		/* If we have no valid aliases, just skip this statement. */
		if($2.entries == 0) {
			stacklist_free(&$2);
			break;
		}
		
		/* Create a new alias structure. */
		newalias=conf_alias_create($3, 0);
		
		/* Add the device pointers from each alias to our new alias. */
		for(stacklist_entry=$2.first; stacklist_entry != NULL; stacklist_entry=stacklist_entry->previous) {
			alias=(Alias *) stacklist_entry->info;
			for(i=0; i < alias->devices; i++) {
				conf_alias_insert_device(newalias, alias->device[i]);
			}
		}
		
		/* Add this alias to the aliaslist. */
		stacklist_insert(&conf_aliaslist, newalias);
		
		/* Free the alias list. */
		stacklist_free(&$2);
	}
	| TTY STRING {
		
		/* If the tty has already been set, this is an error. */
		if(conf_tty) {
			yyerror("TTY already set to '%s'.", conf_tty);
			break;
		}
		
		/* Set this tty as the tty to use. */
		conf_tty=$2;
	}
	| DAEMON_SOCKET INTEGER STRING STRING {
		struct passwd *passwd;
		struct group *group;
		
		/* If it was already set, error. */
		if(conf_daemon_socket_uid != -1 && conf_daemon_socket_gid != -1) {
			yyerror("DAEMON_SOCKET already specified.");
			break;
		}
		
		/* Lookup the UID for the user name and save it. */
		passwd=getpwnam($3);
		if(!passwd) {
			yyerror("Could not get the UID for passwd '%s'.", $3);
			break;
		}
		conf_daemon_socket_uid=passwd->pw_uid;
		
		/* Lookup the GID for the group name and save it. */
		group=getgrnam($4);
		if(!group) {
			yyerror("Could not get the GID for group '%s'.", $4);
			break;
		}
		conf_daemon_socket_gid=group->gr_gid;
		
		/* Save the rest of the settings. */
		conf_daemon_socket_mode=$2;
	}
	| DAEMON_MONITOR_SOCKET INTEGER STRING STRING {
		struct passwd *passwd;
		struct group *group;
		
		/* If it was already set, error. */
		if(conf_daemon_monitor_socket_uid != -1 && conf_daemon_monitor_socket_gid != -1) {
			yyerror("DAEMON_MONITOR_SOCKET already specified.");
 			break;
		}
		
		/* Lookup the UID for the user name and save it. */
		passwd=getpwnam($3);
		if(!passwd) {
			yyerror("Could not get the UID for user '%s'.", $3);
			break;
		}
		conf_daemon_monitor_socket_uid=passwd->pw_uid;
		
		/* Lookup the GID for the group name and save it. */
		group=getgrnam($4);
		if(!group) {
			yyerror("Could not get the GID for group '%s'.", $4);
			break;
		}
		conf_daemon_monitor_socket_gid=group->gr_gid;
		
		/* Save the rest of the settings. */
		conf_daemon_monitor_socket_mode=$2;
	}
	| MACRO opt_name PAREN_LEFT state_list PAREN_RIGHT commands {
		Macro *macro;
		
		/* Allocate a macro structure. */
		macro=malloc(sizeof(Macro));
		if(!macro) fatal("Out of memory.");
		
		/* Fill all the fields of the macro. */
		macro->name=$2;
		macro->statetable=$4;
		macro->macro_commands=$6;
		
		/* 
		 * *** 
		 * Need to make sure the macro doesn't conflict.  Names can
		 * conflict.
		 */
		
		/* Add the macro to the list. */
		stacklist_insert(&conf_macrolist, macro);
	}
	| DAEMON_DIR STRING {
		
		/* If it was already set, error. */
		if(conf_daemon_dir) {
			yyerror("DAEMON_DIR already specified as '%s'.", conf_daemon_dir);
			break;
		}
		
		/* Save this value. */
		conf_daemon_dir=$2;
	}
	| DAEMON_COMMAND_TIMEOUT INTEGER {
		
		/* If it was already set, error. */
		if(conf_daemon_command_timeout != -1) {
			yyerror("DAEMON_COMMAND_TIMEOUT already specified as '%i'.", conf_daemon_command_timeout);
			break;
		}
		
		/* Save this value. */
		conf_daemon_command_timeout=$2;
	}
	| EXAMPLE {
		
		/* 
		 * Trying to use the example configuration without modifying
		 * it first is a bad idea. 
		 */
		yyerror("You have not yet configured ppower, please edit '%s'.", CONFIG_FILE);
		exit(1);
	}

opt_name: name {$$=$1;}
	| /* empty */ {$$=NULL;}

name: IDENTIFIER {$$=$1;}

type: LAMP {$$=DEVICE_LAMP;}
	| APPLIANCE {$$=DEVICE_APPLIANCE;}
	| MOTION_DETECTOR {$$=DEVICE_MOTION_DETECTOR;}
	| SIGNAL {$$=DEVICE_SIGNAL;}
	| TRANSCEIVER {$$=DEVICE_TRANSCEIVER;}

state_list: state_list SEMI state {
		
		/* Add the state to the table. */
		table_insert($$, &$3);
	}
	| state {
		
		/* Create a table to pass up. */
		$$=table_create(sizeof(State));
		
		/* Insert this state to the table. */
		table_insert($$, &$1);
	}

state: alias_list COLON function {
		Stacklist_Entry *stacklist_entry;
		Alias *alias;
		
		/* Initialize the state and save the function. */
		$$.devices=0;
		$$.device=NULL;
		$$.function=$3;
		
		/* Add all the device pointers in each alias to the state. */
		for(stacklist_entry=$1.first; stacklist_entry != NULL; stacklist_entry=stacklist_entry->previous) {
			alias=(Alias *) stacklist_entry->info;
			
			/* Make room for these devices. */
			$$.device=realloc($$.device, ($$.devices + alias->devices) * sizeof(Device *));
			if($$.device == NULL) {
				fatal("Out of memory.");
			}
			
			/* Copy the new ones in. */
			(void) memcpy($$.device + $$.devices, alias->device, alias->devices * sizeof(Device *));
			
			/* Mark that we have these new ones. */
			$$.devices += alias->devices;
		}
		
		/* Free the alias list. */
		stacklist_free(&$1);
	}

commands: BRACE_LEFT commands_braced BRACE_RIGHT {
		$$=$2;
	}
	| BRACE_LEFT BRACE_RIGHT {
		
		/* We have nothing. */
		$$.commands=0;
		$$.macro_command=NULL;
	}
	| command_statement {
		
		/* We have only one statement, add it. */
		$$.commands=1;
		$$.macro_command=malloc(sizeof(Macro_Command));
		if(!$$.macro_command) {
			fatal("Out of memory.");
		}
		$$.macro_command[0]=$1;
	}

commands_braced: commands_braced commands {
		int i;
		
		/* Carry up the passed commands as our own. */
		$$=$1;
		
		/* Make space for the new commands. */
		$$.macro_command=realloc($$.macro_command, ($$.commands + $2.commands) * sizeof(Macro_Command));
		if(!$$.macro_command) {
			fatal("Out of memory.");
		}
		
		/* Add these statements to the command list. */
		for(i=$$.commands; i < $$.commands + $2.commands; i++) {
			$$.macro_command[i]=$2.macro_command[i - $$.commands];
		}
		$$.commands += $2.commands;
		
		/* Free the cruft left on the passed commands structure. */
		free($2.macro_command);
	}
	| commands {
		$$=$1;
	}

command_statement: COMMAND state {
		
		/* Store the state. */
		$$.type=MACRO_COMMAND_COMMAND;
		$$.info.command_command.state=$2;
	}
	| RUN STRING {
		
		/* Store the commandline. */
		$$.type=MACRO_COMMAND_RUN;
		$$.info.command_run.commandline=$2;
	}
	| IF PAREN_LEFT expression PAREN_RIGHT commands {
		
		/* Store if's info. */
		$$.type=MACRO_COMMAND_IF;
		$$.info.command_if.expression=$3;
		$$.info.command_if.macro_commands=$5;
	}
	| ELSE commands {
		
		/* Store else's info. */
		$$.type=MACRO_COMMAND_ELSE;
		$$.info.command_else.macro_commands=$2;
	}

expression: {
		
		/* *** */
	}

alias_list: alias_list COMMA alias {
	
		/* If the alias didn't error, insert it. */
		if($3 != NULL) stacklist_insert(&$$, $3);
	}
	| alias {
		
		/* Initialize the alias list. */
		stacklist_init(&$$);
		
		/* If this alias didn't error, insert it. */
		if($1 != NULL) stacklist_insert(&$$, $1);
	}

alias: IDENTIFIER {
		Alias *alias;
		
		/* Aliases alias aliases. :)  Find the alias this one specifies. */
		alias=conf_aliaslist_lookup($1);
		if(!alias) {
			yyerror("No such alias name, '%s'.", $1);
			$$=NULL;
			break;
		}
		
		/* Pass this one up. */
		$$=alias;
	}

device_list: device_list COMMA device {
	
		/* If the device didn't error, insert it. */
		if($3 != NULL) stacklist_insert(&$$, $3);
	}
	| device {
		
		/* Initialize the device list. */
		stacklist_init(&$$);
		
		/* If this device didn't error, insert it. */
		if($1 != NULL) stacklist_insert(&$$, $1);
	}

device: IDENTIFIER {
		char *endptr;
		int housecode_index;
		int devicecode_index;
		
		/* 
		 * This should be a basic type like 'A2'.  Note they are case
		 * insensitive as are all identifiers.
		 */
		if(tolower($1[0]) < 'a' || tolower($1[0]) > 'p') {
			yyerror("Identifier '%s' is not a valid device.", $1);
			$$=NULL;
			break;
		}
		housecode_index=tolower($1[0]) - 'a';
		
		/* The digit should be between 1 and 16. */
		errno=0;
		devicecode_index=strtol(&$1[1], &endptr, 10);
		if(errno || devicecode_index < 1 || devicecode_index > 16) {
			yyerror("Identifier '%s' is not a valid device.", $1);
			$$=NULL;
			break;
		}
		devicecode_index--;
		
		/* We shouldn't have any more identifier. */
		if(*endptr != 0) {
			yyerror("Identifier '%s' is not a valid device.", $1);
			$$=NULL;
			break;
		}
		
		/* Create this device and pass it up. */
		$$=conf_device_create($1, housecode_table[housecode_index], devicecode_table[devicecode_index], DEVICE_UNDEFINED);
	}

function: ALL_UNITS_OFF {$$.command=COMMAND_ALL_UNITS_OFF;}
	| ALL_LIGHTS_ON {$$.command=COMMAND_ALL_LIGHTS_ON;}
	| ON {$$.command=COMMAND_ON;}
	| OFF {$$.command=COMMAND_OFF;}
	| DIM COLON INTEGER {
		$$.command=COMMAND_DIM;
		
		/* The integer must be from 0 to 22. */
		if($3 < 0 || $3 > 22) {
			
			/* Error and return the value as 0. */
			yyerror("Dim value '%i' out of range.  Must be 0-22.", $3);
			$$.extended1=0;
			break;
		}
		
		$$.extended1=$3;
	}
	| BRIGHT COLON INTEGER {
		$$.command=COMMAND_BRIGHT;
		
		/* The integer must be from 0 to 22. */
		if($3 < 0 || $3 > 22) {
			
			/* Error and return the value as 0. */
			yyerror("Bright value '%i' out of range.  Must be 0-22.", $3);
			$$.extended1=0;
			break;
		}
		
		$$.extended1=$3;
	}
	| ALL_LIGHTS_OFF {$$.command=COMMAND_ALL_LIGHTS_OFF;}
	| EXTENDED_CODE {$$.command=COMMAND_EXTENDED_CODE;}
	| HAIL_REQUEST {$$.command=COMMAND_HAIL_REQUEST;}
	| HAIL_ACKNOWLEDGE {$$.command=COMMAND_HAIL_ACKNOWLEDGE;}
	| PRESET_DIM1 {$$.command=COMMAND_PRESET_DIM1;}
	| PRESET_DIM2 {$$.command=COMMAND_PRESET_DIM2;}
	| EXTENDED_DATA_TRANSFER COLON INTEGER COLON INTEGER {
		$$.command=COMMAND_EXTENDED_DATA_TRANSFER;
		
		/* The extended values must be 0-255. */
		if($3 < 0 || $3 > 255) {
			yyerror("Extended value '%i' out of range.", $3);
			$$.extended1=0;
			$$.extended2=0;
			break;
		}
		if($5 < 0 || $5 > 255) {
			yyerror("Extended value '%i' out of range.", $5);
			$$.extended1=0;
			$$.extended2=0;
			break;
		}
		
		$$.extended1=$3;
		$$.extended2=$5;
	}
	| STATUS_ON {$$.command=COMMAND_STATUS_ON;}
	| STATUS_OFF {$$.command=COMMAND_STATUS_OFF;}
	| STATUS_REQUEST {$$.command=COMMAND_STATUS_REQUEST;}

%%
