/*
 * The CmdlineParser namespace contains the definitions needed to parse
 * an argv-style command-line.
 */

#ifndef __CLP_H__
#define __CLP_H__

#include <map>
#include <vector>
#include <string>
#include <stdexcept>

namespace CLP
{
    class parse_error : public std::runtime_error {
    public:
	parse_error(const std::string& what) : std::runtime_error(what) { }
    virtual ~parse_error() throw() {}
    };

    class option_error : public parse_error {
    public:
	option_error(const std::string& what, const std::string& opt_name)
	    : parse_error(what), m_option_name(opt_name) { }
    virtual ~option_error() throw() {}
	std::string option_name() const { return m_option_name; }
    private:
	std::string m_option_name;
    };

    class invalid_option_error : public option_error {
    public:
	invalid_option_error(const std::string& opt_name)
	    : option_error("invalid option error", opt_name) { }
    };

    class missing_value_error : public option_error {
    public:
	missing_value_error(const std::string& opt_name)
	    : option_error("missing value error", opt_name) { }
    };

    class value_present_error : public option_error {
    public:
	value_present_error(const std::string& opt_name)
	    : option_error("value present error", opt_name) { }
    };

    enum ArgumentType {
	ARGUMENT_NONE,
	ARGUMENT_STRING
    };

    enum ActionType {
	ACTION_MAP,       // place the option into the output map
	ACTION_CALLBACK   // use the callback to process the option
    };

    typedef std::map<std::string, std::string> option_map_t;
    typedef std::vector<std::string> args_t;
    typedef void (*callback_t)(const std::string& tag,
			       const std::string& value,
			       void* data);

    struct option_t {
	const char* key;
	char short_option;
	const char* long_option;
	ArgumentType arg_type;
	ActionType action;
	callback_t callback;
	void* data;
    };

    extern void parse(int argc, char ** argv, const option_t * options,
		      option_map_t& option_output, args_t& args);
}

#endif
