/* Open Digita Services  --  Camera Device Protocol definitions.
  
   Copyright (C) 1998, 1999 James C. Thompson <jim.thompson@pobox.com>
   Copyright (C) 1998, 1999 Viljo Hakala <viljo.hakala@pcuf.fi>
   
   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2 of the License, or (at your
   option) any later version.
   
   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */

/*--------------------------------------------------------------------------
  System include files */

#include <stdio.h>
#include <setjmp.h>

#if defined(HAVE_CONFIG_H)
#include "config.h"
#endif

#if defined(HAVE_UNISTD_H)
#include <unistd.h>
#endif

#include <fcntl.h>

/*--------------------------------------------------------------------------
  Local include files */

#include "usb.h"
#include "cameraP.h"
#include "format-mem.h"

/* Start editing here: */

int
usb_open(ODSCamera camera, const char *device)
{
  camera->fd = open(device, O_RDWR | O_NOCTTY | O_NONBLOCK);

  /* Open the port */
  if (camera->fd < 0) 
    {
      camera->error = kODSUnixError;    /* Assignment */
      return -1;
    }

  camera->host_size = 1023;
  camera->device_size = 1023;
  camera->mode = ods_presend;

#if defined(DEBUG)
  printf("serial port opened\n");
#endif

  return 0;
}

int 
usb_close(ODSCamera camera)
{
  close(camera->fd);
  return 0;
}

int
usb_send_message(ODSCamera camera)
{
  char *buffer = camera->buffer;

#if defined(DEBUG)
  printf("usb writing %ld bytes.\n", camera->bufbytes);
#endif

  while (camera->bufused < camera->bufbytes)
    {
      int remaining = camera->bufbytes - camera->bufused;
      int to_write = remaining < USB_QUANTA ? remaining : USB_QUANTA;
      int wrote_here = write(camera->fd, buffer, to_write);

#if defined(DEBUG)
      printf("usb write wrote %d:\n", wrote_here);
      format_chars(stdout, buffer, 0, wrote_here);
#endif

      if (wrote_here == 0)
	{
	  return -1;
	}
      
      camera->bufused += wrote_here;
      buffer += wrote_here;
    }

  return 0;
}

extern jmp_buf env;

int
usb_receive_message(ODSCamera camera)
{
  unsigned long	message_len;
  unsigned char *buffer = camera->buffer;
  unsigned long read_here;

#if defined(DEBUG)
  printf("usb_receive_message: entering\n");
#endif

  ensure_buffer_size(camera, USB_QUANTA);
  read_here = read(camera->fd, buffer, USB_QUANTA);

#if defined(DEBUG)
  printf("usb receive read %ld:\n", read_here);
  format_chars(stdout, (char *) buffer, 0, read_here);
#endif

  if (read_here == 0)
    {
      return -1;
    }

  buffer += read_here;
  camera->bufbytes += read_here;

  /* Copy the overall message length. */
  memcpy(&message_len, camera->buffer, sizeof(message_len));
  swap_long(message_len);

  message_len += 4;

  /* Set an arbitrary limit to the size of message_len, based on what
     we know about the CDP.  It usually sends messages of less than
     20K.  Allow up to 64K to be generous.  If we get out of sync with 
     the camera, a message length of ridiculously huge size results;
     this is to attempt to detect that condition, and at least prevent 
     a core dump. */
  if (message_len > 0x10000)
    {
      message_len = 0x10000;
      fprintf(stderr, "*** Warning: unusual message size detected.  Protocol may be out of sync.\n");
    }

#if defined(DEBUG)
  printf("usb_receive_message: size = %ld, already read = %ld\n",
	 message_len, camera->bufbytes);
#endif

  /* Make sure we have enough room to read the entire message. */
  ensure_buffer_size(camera, message_len);

  /* Loop and read the remainder of the message. */
  while (camera->bufbytes < message_len)
    {
      read_here = read(camera->fd, buffer, USB_QUANTA);

#if defined(DEBUG)
      printf("usb receive read %ld:\n", read_here);
      format_chars(stdout, (char *) buffer, 0, read_here);
#endif

      if (read_here == 0)
	{
#if defined(DEBUG)
	  printf("usb receive error\n");
	  printf("    long jumping back to high level\n");
#endif
	  longjmp(env, 0);
	  return -1;
	}

      buffer += read_here;
      camera->bufbytes += read_here;
    }

#if defined(DEBUG)
  printf("usb_receive_message: returning normally\n");
#endif

  return 0;
}
