/*
 * ntesla - a program for designing Tesla Coils
 * 
 * Copyright (C) 1997  Steven A. Falco
 * 
 * Email contact: sfalco@worldnet.att.net
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define EXTERN extern
#include "protos.h"

/* Build the screen image */
void
capgen(int mode)
{
	int iii;

	Row = 0;
S "  *Style: %s  *Material: %s  *Number of plates: %d\n",
   Tcparms.cap_type == FLAT ? "flat" : "rolled",
   Cap_table[Tcparms.cap_dielectric].dielectric_name,
   Tcparms.cap_plates);
S "  *Length: %s  *Width: %s  *Thickness of dielectric: %s\n",
   UnitOut((void *)&Tcparms.cap_length, Tcparms.units ? TOINCH : TOCM, Ub0),
   UnitOut((void *)&Tcparms.cap_width, Tcparms.units ? TOINCH : TOCM, Ub1),
   UnitOut((void *)&Tcparms.cap_gap, Tcparms.units ? TOINCH : TOMM, Ub2));
	if(Tcparms.cap_type == ROLLED) {
S "  *Diameter of roll: %s\n",
   UnitOut((void *)&Tcparms.cap_diameter, Tcparms.units ? TOINCH : TOCM, Ub0));
	}
S "\n");
	if(mode) {
S "Set style to 'f' for a flat-plate capacitor, or 'r' for a rolled-plate\n");
S "capacitor.  Dielectric choices are entered by number:\n");
		for(iii = 0; iii < Dialectric_count; iii++) {
S "\t%d = %s\n", iii, Cap_table[iii].dielectric_name);
		}
S "\n");
	}
S "Desired capacitance is %.3g uF; the above design has %.3g uF.\n",
   Tcparms.primary_cap, calc_cap());
S "Desired working voltage is %g V; the above design is good for %g V.\n",
   Tcparms.primary_volts * 2.5, calc_voltage());
S "\n");
}

/* Capacitor menu */
void
capacitor()
{
	char buf[BS], *bp;
	int itmpval;
	int iii;

	while(1) {
		capgen(1);
		
		/* Clear the screen, then display the menu */
		CLEAR_SCREEN; for(iii = 0; iii < Row; iii++) P Display[iii]);

		/* Put up general instructions */
		general_instructions();

		/* Get the users choice */
		if(fgets(buf, BS, stdin) == NULL) {
			/* We are done */
			break;
		}

		switch(tolower(buf[0])) {

		  case 's':
		  	bp = buf + 1;
			while(*bp != 0 && *bp == ' ') bp++;
			if(tolower(*bp) == 'f') {
				Tcparms.cap_type = FLAT;
			} else if(tolower(*bp) == 'r') {
				Tcparms.cap_type = ROLLED;
			} else {
				E "\007Sorry - style cannot be set to %c - try again ", *bp);
			}
		  	break;

		  case 'm':
			iii = atoi(buf + 1);
		  	if(iii < 0 || iii >= Dialectric_count) {
				E "\007Sorry - %d is not a dielectric choice ", iii);
			} else {
				Tcparms.cap_dielectric = iii;
			}
		  	break;

		  case 'l':
			UnitIn(atof(buf + 1), (void *)&Tcparms.cap_length,
			  Tcparms.units ? TOINCH : TOCM);
		  	break;

		  case 'w':
			UnitIn(atof(buf + 1), (void *)&Tcparms.cap_width,
			  Tcparms.units ? TOINCH : TOCM);
		  	break;

		  case 't':
			UnitIn(atof(buf + 1), (void *)&Tcparms.cap_gap,
			  Tcparms.units ? TOINCH : TOMM);
		  	break;

		  case 'n':
			itmpval = atoi(buf + 1);
			if(itmpval > 1) {
				Tcparms.cap_plates = atoi(buf + 1);
			} else {
				E "\007Must have at least 2 plates ");
			}
		  	break;

		  case 'd':
			UnitIn(atof(buf + 1), (void *)&Tcparms.cap_diameter,
			  Tcparms.units ? TOINCH : TOCM);
		  	break;

		  case 'h':
		  	cap_help();
			break;

		  case 'p':
		  	print_buf();
			break;

		  case 'q':
		  	save_parms();
			EXIT(0);

		  case 'r':
		  	return;

		  case '\n':
		  case '\r':
		  	break;

		  default:
			E "\007Sorry - %c is illegal - try again ", buf[0]);
			break;
		}
	}

	return;
}

double
calc_cap()
{
	int pairs;	/* How many pairs of plates */
	double area;	/* total area of parallel plates */

	if(Tcparms.cap_type == FLAT) {
		/* Flat caps have one less pair than they have plates */
		pairs = Tcparms.cap_plates - 1;
		area = pairs * Tcparms.cap_length * Tcparms.cap_width;
	} else {
		/* Rolled caps are tricky.  Usually there are two plates
		 * but lets stay general.
		 */
		if(Tcparms.cap_plates & 1) {
			/* Odd number of plates - no bonus for rolling because
			 * inner and outer plates are at the same polarity.
			 */
			pairs = Tcparms.cap_plates - 1;
			area = pairs * Tcparms.cap_length * Tcparms.cap_width;
		} else {
			/* Even number of plates - get a bonus of 1 plate
			 * for rolling because inner and outer plates are at
			 * opposite polarity.
			 */
			pairs = Tcparms.cap_plates;
			area = pairs * Tcparms.cap_length * Tcparms.cap_width;

			/* But, a rolled cap loses a little effective plate
			 * length from its bonus pair on the outer turn, so
			 * reduce the area by one turn.
			 */
			area -= PI * Tcparms.cap_diameter * Tcparms.cap_width;
		}
	}

	return	0.0254e6	/* metric conversion and F to uF */
		*
		PERMITTIVITY	/* Of free space */
		*
		Cap_table[Tcparms.cap_dielectric].dielectric_constant
		*
		area / Tcparms.cap_gap;
}

double
calc_voltage()
{
	return Cap_table[Tcparms.cap_dielectric].breakdown
		* Tcparms.cap_gap * 1e3;
}

/* From Ed Sonderman's spread-sheet tesla calculator */
double
toroid_cap()
{
	double tmp;

	tmp =	2.0 * PI * PI 
		* (Tcparms.terminal_major - Tcparms.terminal_minor)
		* (Tcparms.terminal_minor / 2.0) / (4.0 * PI);

	return	(1.0 +
		(0.2781 - Tcparms.terminal_minor / Tcparms.terminal_major))
		* 2.8 * sqrt(tmp);
}

void
cap_help()
{
	char buf[BS];
	int iii;

	while(1) {
		
		/* Build the screen image */
		Row = 0;
S "Your capacitor should be submerged in a high-voltage oil.  Various mineral\n");
S "oils are used for this purpose.  If you don't use oil, your capacitor will\n");
S "probably have arcing and/or corona problems, which wastes energy and can\n");
S "cause a catastrophic failure.\n");
S "\n");
S "Rolled capacitors are easy to build from aluminum roof flashing (typically\n");
S "in 14\" width), and Low-Density Polyethylene plastic (LDPE).  The capacitor\n");
S "is placed in a length of PVC pipe, which is then filled with oil.  A vacuum\n");
S "pump may be used to help get the air out from between the plates, but\n");
S "simply waiting a week or so works too.\n");
S "\n");
S "Folks have also built flat-plate capacitors, typically placed in plastic\n");
S "containers full of oil.  In this case, with the plates horizontal, the\n");
S "vacuum pump is essential to get the air out.  There are detailed\n");
S "instructions on the Tesla mailing lists and the net for building both\n");
S "styles of capacitors.  Sources of materials are also suggested.\n");
S "\n");


		/* Clear the screen, then display the menu */
		CLEAR_SCREEN; for(iii = 0; iii < Row; iii++) P Display[iii]);

		/* Put up help instructions */
		help_instructions();

		/* Get the users choice */
		if(fgets(buf, BS, stdin) == NULL) {
			/* We are done */
			break;
		}

		switch(tolower(buf[0])) {

		  case 'p':
		  	print_buf();
			break;

		  case 'q':
		  	save_parms();
			EXIT(0);

		  case 'r':
		  	return;

		  case '\n':
		  case '\r':
		  	break;

		  default:
			E "\007Sorry - %c is illegal - try again ", buf[0]);
			break;
		}
	}

	return;
}

/* Build the screen image */
void
toroidgen(int mode)
{
	Row = 0;
	Tcparms.terminal_cap = toroid_cap();
S "  *Outer diameter: %s  *Cross-section diameter: %s\n",
   UnitOut((void *)&Tcparms.terminal_major, Tcparms.units ? TOINCH : TOCM, Ub0),
   UnitOut((void *)&Tcparms.terminal_minor, Tcparms.units ? TOINCH : TOCM, Ub1));
S "\n");
S "  Toroid capacitance = %g pF\n", Tcparms.terminal_cap);
S "\n");
}

void
toroid()
{
	char buf[BS];
	double tmpval;
	int iii;

	while(1) {
		toroidgen(1);	

		/* Clear the screen, then display the menu */
		CLEAR_SCREEN; for(iii = 0; iii < Row; iii++) P Display[iii]);

		/* Put up general instructions */
		general_instructions();

		/* Get the users choice */
		if(fgets(buf, BS, stdin) == NULL) {
			/* We are done */
			break;
		}

		switch(tolower(buf[0])) {

		  case 'o':
			UnitIn(atof(buf + 1), (void *)&tmpval,
			  Tcparms.units ? TOINCH : TOCM);
			if(tmpval > Tcparms.terminal_minor) {
				Tcparms.terminal_major = tmpval;
			} else {
				E "\007Outer diameter must be bigger than cross-sectional diameter ");
			}
		  	break;

		  case 'c':
			UnitIn(atof(buf + 1), (void *)&tmpval,
			  Tcparms.units ? TOINCH : TOCM);
			if(Tcparms.terminal_major > tmpval) {
		  		Tcparms.terminal_minor = tmpval;
			} else {
				E "\007Outer diameter must be bigger than cross-sectional diameter ");
			}
		  	break;

		  case 'h':
		  	toroid_help();
			break;

		  case 'p':
		  	print_buf();
			break;

		  case 'q':
		  	save_parms();
			EXIT(0);

		  case 'r':
		  	return;

		  case '\n':
		  case '\r':
		  	break;

		  default:
			E "\007Sorry - %c is illegal - try again ", buf[0]);
			break;
		}
	}

	return;
}

void
toroid_help()
{
	char buf[BS];
	int iii;

	while(1) {
		
		/* Build the screen image */
		Row = 0;
S "You should have a conductive toroid (donut shape) at the top of your\n");
S "secondary coil.  This top terminal serves several purposes: 1) it stores\n");
S "energy which increases the performance of your coil, and 2) it generates a\n");
S "field which protects the top turn of your secondary.  Without a toroid, you\n");
S "will probably throw sparks off the top turn of your secondary.  This will\n");
S "damage the insulation and may burn your coil form.\n");
S "\n");
S "The capacitance of the toroid should be greater than or equal to the self-\n");
S "capacitance of your secondary.  When you complete the secondary design, you\n");
S "may want to re-visit the toroid design screen.\n");
S "\n");
S "Sometimes folks use a sphere instead of a toroid.  This helps with item 1\n");
S "above but is not very effective with item 2.\n");
S "\n");
S "Commercial toroids are expensive.  To make your own, bend some flexible\n");
S "plastic tubing into a ring.  Attach a center disk to mount the toroid to\n");
S "your coil, then cover the whole thing with conductive aluminum tape.  This\n");
S "is typically available in plumbing supply or home center stores.  There is\n");
S "much more information on toroids on the Tesla mailing lists and on the net.\n");
S "\n");


		/* Clear the screen, then display the menu */
		CLEAR_SCREEN; for(iii = 0; iii < Row; iii++) P Display[iii]);

		/* Put up help instructions */
		help_instructions();

		/* Get the users choice */
		if(fgets(buf, BS, stdin) == NULL) {
			/* We are done */
			break;
		}

		switch(tolower(buf[0])) {

		  case 'p':
		  	print_buf();
			break;

		  case 'q':
		  	save_parms();
			EXIT(0);

		  case 'r':
		  	return;

		  case '\n':
		  case '\r':
		  	break;

		  default:
			E "\007Sorry - %c is illegal - try again ", buf[0]);
			break;
		}
	}

	return;
}
