/*
 * ninredir.c:
 * 
 * this file contains code dealing with /rel* commands for Ninja IRc
 * 
 * these currently include /relm, /relsm, /reln, /relsn
 * 
 * written by Kraig Amador and Joshua J. Drake
 *
 */

#include "irc.h"
#include "dma.h"
#include "output.h"
#include "edit.h"
#include "ircaux.h"

#include "ninja.h"
#include "ninredir.h"

/* /rel and /rels lists */
#define MAX_RELAYS 10
static u_char *redirect_msgs[MAX_RELAYS] = { 0, 0, 0, 0, 0,  0, 0, 0, 0, 0 };
static u_char *redirect_sent_msg[MAX_RELAYS] = { 0, 0, 0, 0, 0,  0, 0, 0, 0, 0 };
static u_char *redirect_notice[MAX_RELAYS] = { 0, 0, 0, 0, 0,  0, 0, 0, 0, 0 };
static u_char *redirect_sent_notice[MAX_RELAYS] = { 0, 0, 0, 0, 0,  0, 0, 0, 0, 0 };


/*
 * the /relm and /relsm commands..
 * (relay message and relay sent message)
 */
void
ninja_redirect(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char **list = NULL, *tmp, tmpbuf[1024];
   char *curchan = get_channel_by_refnum(0);
   int x, number = -1;

   *tmpbuf = '\0';
   if (!command || !*command)
     return;
   upper(command);
   if (my_strcmp(command, "RELM") == 0)
      list = redirect_msgs;
   else
   if (my_strcmp(command, "RELSM") == 0)
      list = redirect_sent_msg;
   else
   if (my_strcmp(command, "RELN") == 0)
      list = redirect_notice;
   else
   if (my_strcmp(command, "RELSN") == 0)
      list = redirect_sent_notice;
   else
     {
	put_error("Unknown relay command!");
	return;
     }
   
   /* check for args.. */
   tmp = next_arg(args, &args);
   if (!tmp)
     {
	/* nothing?  default to relaying the last one to the current channel..
	 * that is if we have a current channel and a last one ;) 
	 */
	if (curchan && list[0])
	  {
	     my_strncpy(tmpbuf, curchan, sizeof(tmpbuf)-1);
	     tmpbuf[sizeof(tmpbuf)-1] = '\0';
	     number = 0;
	  }
	else
	  {
	     usage(command, "[-l] or [<number> [<nick/channel>]]");
	     return;
	  }
     }
   
   /* list? */
   if (my_stricmp(tmp, "-l") == 0)
     {
	for (x = MAX_RELAYS-1; x >= 0; x--)
	  {
	     if (list[x])
	       put_info("/%s %2.2d %s", command, x + 1, list[x]);
	  }
	return;
     }
   
   /* got a #? */
   if (number == -1
       && tmp 
       && isdigit(*tmp))
     {
	number = my_atoi(tmp);
	if (number < 1 
	    || number > MAX_RELAYS
	    || !list[number-1])
	  {
	     put_error("%s: invalid message number: %s", command, tmp);
	     return;
	  }
	number--;
     }
   
   /* see if we got a number/target */
   while ((tmp = next_arg(args, &args)))
     {
	if (*tmpbuf)
	  my_strmcat(tmpbuf, ",", sizeof(tmpbuf)-1);
	my_strmcat(tmpbuf, tmp, sizeof(tmpbuf)-1);
     }
   
   /* where are we sending it? */
   if (number == -1)
     {
	put_error("%s: no message number specified", command);
	return;
     }
   
   /* no target?  how about a current channel? */
   if (!*tmpbuf)
     {
	if (curchan)
	  {
	     my_strncpy(tmpbuf, curchan, sizeof(tmpbuf)-1);
	     tmpbuf[sizeof(tmpbuf)-1] = '\0';
	  }
	else
	  {
	     put_error("%s: no target specified", command);
	     return;
	  }
     }
   
   /* if that one exists, send it.. */
   if (list[number])
     send_text(tmpbuf, list[number], "PRIVMSG");
}

/*
 * adds a redirect (/relm or /relsm)
 */
void
#ifdef HAVE_STDARG_H
addredirect(int cmd, u_char *format,...)
#else
addredirect(cmd, format, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10)
   int cmd;
   u_char *format, *arg1, *arg2, *arg3, *arg4, *arg5, *arg6, *arg7, *arg8,
    *arg9, *arg10;
#endif
{
   int x, y;
   u_char **list = NULL;
   u_char tmpbuf[1024];
#ifdef HAVE_STDARG_H
   va_list vl;

   va_start(vl, format);
   vsnprintf(tmpbuf, sizeof(tmpbuf)-1, format, vl);
   va_end(vl);
#else
   snprintf(tmpbuf, sizeof(tmpbuf)-1, format, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10);
#endif
   tmpbuf[sizeof(tmpbuf)-1] = '\0';

   if (cmd == REDIRECT_MSG)
     list = redirect_msgs;
   else if (cmd == REDIRECT_SENT_MSG)
     list = redirect_sent_msg;
   else if (cmd == REDIRECT_NOTICE)
     list = redirect_notice;
   else if (cmd == REDIRECT_SENT_NOTICE)
     list = redirect_sent_notice;
   else
     return;
   
   /* shift them up */
   for (x = 9, y = 8; x >= 1; x--, y--)
      dma_strcpy(&list[x], list[y]);
   /* add the new one */
   dma_strcpy(&list[0], tmpbuf);
}

