#include "irc.h"
#include "enemies.h"

#include "channels.h"
#include "dma.h"
#include "ircaux.h"
#include "window.h"
#include "screen.h"
#include "server.h"
#include "lastlog.h"
#include "list.h"
#include "output.h"
#include "ninja.h"
#include "vars.h"
#include "whois.h"
#include "numbers.h"

Enemy *enemy_list = NULL;

/*
 * show the enemy list..
 * accessed via /lists
 */
void
show_enemies(command, args, subargs)
   u_char *command, *args, *subargs;
{
   Enemy *e = enemy_list;
   EChan *chan;
   int count = 0;
   u_char tbuf1[2048], tbuf2[512];

   if (!e)
     {
	put_info("Your enemy list is empty.");
	return;
     }
   put_info("Masks%sChannels(level, date, reason) ...", strfill(' ', 31));
   for (e = enemy_list; e; e = e->next)
     {
	snprintf(tbuf1, sizeof(tbuf1)-1, "%-38.38s", e->host);
	tbuf1[sizeof(tbuf1)-1] = '\0';
	for (chan = e->channels; chan; chan = chan->next)
	  {
	     snprintf(tbuf2, sizeof(tbuf2)-1, " %s(%d, %s, %s)", chan->channel, chan->level,
		      ninja_date(&chan->date), chan->reason);
	     tbuf2[sizeof(tbuf2)-1] = '\0';
	     strmcat(tbuf1, tbuf2, sizeof(tbuf1)-1);
	  }
	put_info("%s", tbuf1);
	count++;
     }
   put_info("End of e list, %d counted.", count);
}

/*
 * check to see if the person whom joined is an enemy..
 */
void
check_enemy(u_char *channel, u_char *user, u_char *nick, int delay)
{
   Enemy *e;
   EChan *EChan;
   u_char reason[1024], *p = user;
   
   /* no power?! */
   if (!is_chanop(channel, get_server_nickname(from_server)))
      return;
   /* are they a bad guy? */
   if ((e = get_enemy(p)) != NULL)
     {
	/* permabanned from here? */
	if ((EChan = get_enemy_chan(e, channel, 0)) != NULL)
	  {
	     snprintf(reason, sizeof(reason)-1, "ENEMY(%s): %s",
		      ninja_date(&EChan->date), EChan->reason);
	     reason[sizeof(reason)-1] = '\0';
	     if (EChan->level > 1)
	       send_to_server("MODE %s +b %s", channel, e->host);
	     send_to_server("KICK %s %s :%s", channel, nick, reason);
	  }
     }
}

/*
 * LOAD the enemy list, partially re-written by Senor Pato on
 * 03/05/98 at 2:23AM
 */
int
load_enemies(quiet)
   int quiet;
{
   Enemy *e = NULL;
   EChan *newchan;
   u_char file[] = NINJA_ENEMY_FILE;
   FILE *infile;
   u_char *thEChan = NULL, *themode, *reason, *ptr, *lineptr, *shdateptr;
   int position = 0, count = 0;
   u_char tmpbuf1[(BIG_BUFFER_SIZE / 2) + 1];

   ptr = expand_twiddle(file);
   infile = fopen(ptr, "r");
   dma_Free(&ptr);
   if (infile == (FILE *) NULL)
     {
	// put_info("Unable to load your enemy list from \"%s\"..", file);
	return 0;
     }
   while (fgets(tmpbuf1, BIG_BUFFER_SIZE / 2, infile))
     {
	tmpbuf1[strlen(tmpbuf1) - 1] = '\0';
	/* comment character.. */
	if (*tmpbuf1 == '#')
	   continue;
	count++;
	e = (Enemy *) dma_Malloc(sizeof(Enemy));
	e->host = NULL;
	e->channels = NULL;
	position = 0;
	lineptr = tmpbuf1;
	while ((ptr = my_strsep(&lineptr, ":")))
	  {
	     if (position == 0)
	       {
		  dma_strcpy(&e->host, ptr);
		  position++;
		  add_to_list((List **) & enemy_list, (List *) e);
	       }
	     else if ((*ptr == '&' || *ptr == '#' || *ptr == '*') && match("*+*", ptr))
	       {
		  dma_strcpy(&thEChan, ptr);
		  themode = strrchr(ptr, '+');
		  shdateptr = strchr(ptr, ' ');
		  reason = strrchr(ptr, ',');
		  reason++;
		  thEChan[strlen(ptr) - strlen(themode)] = '\0';
		  if ((newchan = (EChan *) remove_from_list((List **)&e->channels, thEChan)) != NULL)
		    {
		       dma_Free(&(newchan->channel));
		       dma_Free(&(newchan->reason));
		       dma_Free(&newchan);
		    }
		  newchan = (EChan *) dma_Malloc(sizeof(EChan));
		  newchan->channel = NULL;
		  dma_strcpy(&newchan->channel, thEChan);
		  dma_Free(&thEChan);
		  newchan->level = atoi(themode);
		  newchan->reason = NULL;
		  dma_strcpy(&newchan->reason, reason);
		  newchan->date = atol(shdateptr);
		  add_to_list((List **) & (e->channels), (List *) newchan);
	       }
	     position++;
	  }
     }
   fclose(infile);
   return count;
}

/*
 * SAVE the enemy list file, partially re-written by Senor Pato
 * on 03/05/98 at 2:24AM
 */
int
save_enemies(quiet)
   int quiet;
{
   u_char file[] = NINJA_ENEMY_FILE;
   FILE *outfile;
   u_char *ptr, tb1[1024], tb2[256];
   int count = 0;
   Enemy *e;
   EChan *chan;

   ptr = expand_twiddle(file);
   outfile = fopen(ptr, "w");
   dma_Free(&ptr);
   if (outfile == NULL)
     {
	put_info("Unable to write to \"%s\", aborting Enemy save..", file);
	return 0;
     }
   for (e = enemy_list; e; e = e->next)
     {
	chan = e->channels;
	snprintf(tb1, sizeof(tb1)-1, "%s%s", e->host, e->channels ? ":" : "");
	tb1[sizeof(tb1)-1] = '\0';
	for (chan = e->channels; chan; chan = chan->next)
	  {
	     snprintf(tb2, sizeof(tb2)-1, "%s+%d %lu,%s%s", chan->channel, chan->level,
		      (unsigned long)chan->date, chan->reason, chan->next ? ":" : "");
	     tb2[sizeof(tb2)-1] = '\0';
	     strmcat(tb1, tb2, sizeof(tb1)-1);
	  }
	fprintf(outfile, "%s\n", tb1);
	count++;
     }
   if (!quiet && count > 0)
      put_info("Saved %d enem%s...", count, Y_PLURAL(count));
   fclose(outfile);
   return count;
}

Enemy *
get_enemy(u_char *hostmask)
{
   Enemy *e;

   if (!enemy_list)
      return NULL;
   if ((e = (Enemy *) find_in_list((List **)&enemy_list, hostmask, 1)) != NULL)
      return e;
   return NULL;
}

EChan *
get_enemy_chan(Enemy *e, u_char *channel, int level)
{
   EChan *chan;

   if ((chan = (EChan *) find_in_list((List **)&e->channels, channel, 0)) != NULL)
     {
      if ((chan->level == level) || (level == 0))
	 return chan;
      else
	 return NULL;
     }
   for (chan = e->channels; chan; chan = chan->next)
      if (*chan->channel == '*')
	 if ((chan->level == level) || (level == 0))
	    return chan;
   return NULL;
}

void
add_enemy_chan(u_char *command, u_char *args, u_char *subargs)
{
   EChan *new;
   Enemy *tmp;
   u_char *hostmask = NULL, *channel = NULL, *level = NULL;
   time_t currtime = time(NULL);
   int remove = 0;
   int lev;

   if (command && *command && !strcmp(command, "REMSHITCHAN"))
      remove = 1;
   hostmask = next_arg(args, &args);
   channel = next_arg(args, &args);
   level = next_arg(args, &args);
   if (!(channel && *channel))
     {
	if (!remove)
	   usage(command, "<hostmask> <channel> <level> [<reason>]");
	else
	   usage(command, "<hostmask> <channel>");
	return;
     }
   if (my_stricmp(channel, "*"))
      channel = make_chan(channel);
   if ((tmp = get_enemy(hostmask)) == NULL)
     {
	put_info("Hostmask %s not found in enemy list.", hostmask);
	return;
     }
   if ((new = (EChan *) remove_from_list((List **) & (tmp->channels), channel)) != NULL)
     {
	dma_Free(&(new->channel));
	dma_Free(&(new->reason));
	dma_Free(&new);
	if (remove)
	  {
	     put_info("Removed channel %s from hostmask %s", channel, hostmask);
	     (void)save_enemies(1);
	     return;
	  }
     }
   new = (EChan *) dma_Malloc(sizeof(EChan));
   new->channel = NULL;
   dma_strcpy(&new->channel, channel);
   if (!isdigit(*level))
      lev = 2;
   else
     {
	lev = atoi(level);
	if (lev > 3 || lev < 1)
	   lev = 2;
     }
   new->level = lev;
   new->date = currtime;
   new->reason = NULL;
   if (args != NULL)
      dma_strcpy(&new->reason, args);
   else
      dma_strcpy(&new->reason, "OUT");
   add_to_list((List **) & (tmp->channels), (List *) new);
   put_info("Added %s (level %c) to hostmask %s", channel, *level, hostmask);
   (void)save_enemies(1);
}

void
add_enemy(u_char *command, u_char *args, u_char *subargs)
{
   Enemy *tmp;
   EChan *new, *ttmp;
   u_char *hostmask = NULL, *channel = NULL, *level = NULL, *reason = NULL,
     tmpcallbuf[256];
   int lev, remove = 0;

   if (command && *command && !strcmp(command, "REMSHIT"))
      remove = 1;
   if (args && *args)
     {
	hostmask = next_arg(args, &args);
	channel = next_arg(args, &args);
	level = next_arg(args, &args);
	reason = args;
	if (remove == 0)
	  {
	     if (!(level && *level))
	       {
		  usage(command, "<hostmask> <channel> <level> [<reason>]");
		  return;
	       }
	     if (!match("*!*@*", hostmask))
	       {
		  put_info("hostmask must match *!*@* format");
		  return;
	       }
	     if (!isdigit(*level))
		lev = 2;
	     else
	       {
		  lev = atoi(level);
		  if (lev > 3 || lev < 1)
		     lev = 2;
	       }
	     if (my_stricmp(channel, "*"))
		channel = make_chan(channel);
	     if ((tmp = get_enemy(hostmask)) != NULL)
	       {
		  put_info("%s is already in your enemy list", hostmask);
		  return;
	       }
	     tmp = (Enemy *) dma_Malloc(sizeof(Enemy));
	     tmp->host = NULL;
	     tmp->channels = NULL;
	     dma_strcpy(&tmp->host, hostmask);
	     add_to_list((List **) & enemy_list, (List *) tmp);
	     put_info("Added %s to your enemy list", tmp->host);
	     if (reason != NULL)
		snprintf(tmpcallbuf, sizeof(tmpcallbuf), "%s %s %d %s", hostmask,
			 channel, lev, reason);
	     else
		snprintf(tmpcallbuf, sizeof(tmpcallbuf), "%s %s %d", hostmask, channel,
			 lev);
	     add_enemy_chan("ADDSHITCHAN", tmpcallbuf, NULL);
	  }
	else
	  {
	     if ((tmp = get_enemy(hostmask)) != NULL)
	       {
		  if ((tmp = (Enemy *) remove_from_list((List **) & enemy_list, tmp->host)) != NULL)
		    {
		       put_info("Removed %s from enemy list", tmp->host);
		       dma_Free(&tmp->host);
		       for (new = tmp->channels; new; new = ttmp)
			 {
			    ttmp = new->next;
			    dma_Free(&new->reason);
			    dma_Free(&new->channel);
			    dma_Free(&new);
			 }
		    }
	       }
	     dma_Free((u_char **)&tmp);
	     (void)save_enemies(1);
	  }
     }
   else if (remove)
      usage(command, "<hostmask>");
   else
      usage(command, "<hostmask> <channel> <level> [<reason>]");
}
