/*
 * newmail: a program to tell you if you have new mail in a number of mailboxes
 *
 * 30/9/1999
 *
 * TODO: Parse .pinerc to get names and paths of inboxes.
 *
 * Copyright 1999 Andrew Stevenson. All Rights Reserved.
 *
 * HISTORY:
 *	1.0		Initial release
 *	1.0.1	Changed way messages a generated and fixed typo for negative
 *			message
 */

#include <sys/stat.h>

#include <err.h>
#include <dirent.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <sysexits.h>
#include <unistd.h>

#include "newmail.h"

static void usage();
static void version();

unsigned char	no_mail_message,
				individual_messages,
				use_exit_val;

int main(int argc, char *argv[]) {
	unsigned char warn_non_regular,
				  found_mail;
	unsigned int ui;
	int c;
	struct dirent *curr_file;
	struct stat file_info;
	DIR *curr_dir;

	warn_non_regular = no_mail_message = individual_messages = found_mail = use_exit_val = 0;

	while ((c = getopt(argc, argv, "deirv")) != -1) {
		switch (c) {
		case 'd':
			no_mail_message = 1;
			break;
		case 'e':
			use_exit_val = 1;
			break;
		case 'i':
			individual_messages = 1;
			break;
		case 'r':
			warn_non_regular = 1;
			break;
		case 'v':
			version();
			break;
		case '?':
		default:
			usage();
			break;
		}
	}
	argc -= optind;
	argv += optind;

	if (argc < 1 || (use_exit_val && (no_mail_message || individual_messages))) {
		usage();
	}

	for (ui = 0; ui < argc; ++ui) {
		/* determine argument type */
		if (stat(argv[ui], &file_info) == -1) {
			warn("couldn't stat %s, continuing", argv[ui]);
			continue;
		}
		if (S_ISREG(file_info.st_mode)) {
			/* if its a regular file assume its a mailbox and check it */
			check_mailbox(argv[ui], &file_info, &found_mail);
		} else if (S_ISDIR(file_info.st_mode)) {
			/* if its a directory we want to look inside it. We chdir to
			   save building up pathnames */
			if (chdir(argv[ui]) == -1) {
				err(EX_NOINPUT, "chdir to %s failed", argv[ui]);
			}
			if ((curr_dir = opendir(".")) == NULL) {
				err(EX_NOINPUT, "opendir failed for %s", argv[ui]);
			}

			while ((curr_file = readdir(curr_dir)) != NULL) {
				/* we have a file, skip if it starts with a . */
				if ((curr_file->d_name)[0] == '.') {
					continue;
				}
				/* see if its a regular file */
				if (S_ISREG(DTTOIF(curr_file->d_type))) {
					/* stat it for check_mailbox */
					if (stat(curr_file->d_name, &file_info) == -1) {
						err(EX_NOINPUT, "couldn't stat %s/%s", argv[ui], curr_file->d_name);
					}
					check_mailbox(curr_file->d_name, &file_info, &found_mail);
				} else if (warn_non_regular) {
					/* just check its not a directory before warning */
					if (! S_ISDIR(DTTOIF(curr_file->d_type))) {
						warnx("Found %s/%s: Not a regular file", argv[ui], curr_file->d_name);
					}
				} /* end of is dirent a regular file */
			} /* end of for every file in dir while loop */

			/* we must have read everything in this directory now */
			if (closedir(curr_dir) == -1) {
				err(EX_IOERR, "couldn't close directory %s", argv[ui]);
			}
		} /* end of is a directory */
	} /* end of for every argument for loop */

	if (! individual_messages && ! use_exit_val) {
		if (found_mail) {
			printf(GROUP, AFFIRM);
		} else if (no_mail_message) {
			printf(GROUP, NEGATE);
		}
	}

	if ((! use_exit_val) || (use_exit_val && !found_mail)) {
		return(0);
	} else {
		return(1);
	}
}

void check_mailbox(const char *const fn, const struct stat *const file_info, unsigned char *const found_mail) {

	if (file_info->st_mtime > file_info->st_atime) {
		*found_mail = 1;
		if (individual_messages && ! use_exit_val) {
			printf(INDIVIDUAL, AFFIRM, fn);
		}
	} else if (no_mail_message && individual_messages && ! use_exit_val) {
		printf(INDIVIDUAL, NEGATE, fn);
	}
}

static void usage() {

	fprintf(stderr,"usage: newmail [-deirv] directory|file directory|file ...

where:
	d: Prints messages when you do not have new mail
	e: Exits with 0 on no new mail, 1 on new mail, > 1 on error
		(normally exits with a value from sysexits(3)). May not be used
		with -d or -i.
	i: Prints a message for each mailbox that has new mail (and those
		that don't if -d is also used)
	r: Warns if it finds a non regular file
	v: Print version number and exit
	
directory|file are, respectively, directories to look in for mailboxes
and mailboxes.\n");

	exit(EX_USAGE);
}

static void version() {

	printf(VERSION_STRING);

	exit(EX_OK);
}
