/* $NetBSD$ */

/*-
 * Copyright (C) 2006 Jared D. McNeill <jmcneill@invisible.ca>.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Jared D. McNeill.
 * 4. Neither the name of the author nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>

#include "porting.h"

__RCSID("$NetBSD$");

#include <sys/types.h>

#include <netinet/in.h>

#include <pthread.h>
#include <stdlib.h>
#include <syslog.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>

#include "conf.h"
#include "transmit.h"

typedef struct transmit_args {
	mtftpd_conf_t ta_mc;
	mtftpd_file_t *ta_mf;
} transmit_args_t;

static void *
transmit_thread(void *opaque)
{
	transmit_args_t *ta;
	struct sockaddr_in sin;
	mtftpd_file_t *mf;
	mtftpd_conf_t *mc;
	int fd;

	ta = (transmit_args_t *)opaque;
	mf = ta->ta_mf;
	mc = &ta->ta_mc;

	pthread_mutex_lock(&mf->mf_lock);
	fd = open(mf->mf_filepath, O_RDONLY);
	if (mf->mf_fd < 0) {
		syslog(LOG_ERR, "unable to open %s", mf->mf_filepath);
		mf->mf_running = 0;
		pthread_mutex_unlock(&mf->mf_lock);
		free(ta);
		return NULL;
	}
	pthread_mutex_unlock(&mf->mf_lock);

	sin.sin_family = AF_INET;
	sin.sin_port = htons(1758); /* tftp-mcast */
	sin.sin_addr.s_addr = mf->mf_inetaddr.s_addr;

	syslog(LOG_INFO, "send thread starting up for %s", mf->mf_filename);

	for (;;) {
		ssize_t len;
		int pktnum = 0;

		lseek(fd, 0, SEEK_SET);

		pthread_mutex_lock(&mf->mf_lock);
		if (mf->mf_sendcnt > mf->mf_retry) {
			mf->mf_running = 0;
			pthread_mutex_unlock(&mf->mf_lock);
			break;
		}
		mf->mf_sendcnt++;
		pthread_mutex_unlock(&mf->mf_lock);

		do {
			len = read(fd, mf->mf_buf + 4, mc->mc_blksize);
			if (len > 0) {
				ssize_t sz;

				if (pktnum % mf->mf_burst == 0 &&
				    mf->mf_packetdelay > 0)
					usleep(mf->mf_packetdelay);

				++pktnum;
				mf->mf_buf[0] = 0x00;
				mf->mf_buf[1] = 0x03; /* data packet */
				mf->mf_buf[2] = htons(pktnum) & 0xff;
				mf->mf_buf[3] = htons(pktnum) >> 8;
				sz = sendto(mf->mf_sock, mf->mf_buf,
				    mc->mc_blksize + 4, 0,
				    (struct sockaddr *)&sin,
				    sizeof(struct sockaddr_in));
				if (sz == -1)
					syslog(LOG_WARNING,
					    "sendto failed: %d\n", errno);

			}
		} while (len > 0);
	}

	close(fd);

	syslog(LOG_INFO, "send thread shutting down for %s", mf->mf_filename);
	free(ta);
	return NULL;
}

void
transmit_start(mtftpd_conf_t *mc, mtftpd_file_t *mf)
{
	int rv;

	rv = pthread_mutex_lock(&mf->mf_lock);
	if (rv) {
		syslog(LOG_ERR, "pthread_mutex_lock(...) failed: %d", rv);
		return;
	}

	/* reset send count */
	mf->mf_sendcnt = 0;

	/* if we're not running, start up a new thread */
	if (mf->mf_running == 0) {
		transmit_args_t *ta;
		ta = malloc(sizeof(transmit_args_t));
		if (ta == NULL) {
			syslog(LOG_ERR, "out of memory");
			goto cleanup;
		}

		ta->ta_mc = *mc;
		ta->ta_mf = mf;

		mf->mf_running = 1;
		rv = pthread_create(&mf->mf_thread, NULL, transmit_thread, ta);
		if (rv) {
			syslog(LOG_ERR, "pthread_create(...) failed: %s",
			    strerror(rv));
			goto cleanup;
		}
	}

cleanup:
	rv = pthread_mutex_unlock(&mf->mf_lock);
	if (rv) {
		syslog(LOG_ERR, "pthread_mutex_unlock(...) failed: %d", rv);
		return;
	}

	return;
}
