/*
 * mod_throttle.c for Apache 1.3
 *
 * Copyright 1999, 2000 by Anthony Howe.  All rights reserved.
 *
 *
 * LICENSE
 * -------
 *
 * This source distribution is made freely available and there is no charge
 * for its use, provided you retain this notice, disclaimers, author's
 * copyright, and credits.
 *
 *
 * DISCLAIMER
 * ----------
 *
 * THIS SOFTWARE IS PROVIDE "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO WAY SHALL THE
 * AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *
 * CREDITS
 * -------
 *
 * Original design for mod_throttle/1.0 goes to Mark Lovell
 * <mlovell@bigrock.com>.
 *
 * Elements of the critical & shared memory code, as of version 2.2,
 * originally derived from the Apache Web Server source code.
 *
 * Thank You to Lu Vo <LVo@SUPERB.NET> for providing a Solaris POSIX
 * compliant machine to test on; to Travis Doherty <travis@terradelta.net>
 * for a FreeBSD machine for testing and several suggestions.
 */

/***********************************************************************
 *** Pick one that best suits your system.
 ***********************************************************************/

/*
 * Special features
 */
#define THROTTLE_CLIENT_IP
#define THROTTLE_REMOTE_USER

/*
 * Choice of API to be used for serialization (semaphores & mutexes).
 */
#undef USE_POSIX_SERIALIZATION
#undef USE_FCNTL_SERIALIZATION
#undef USE_FLOCK_SERIALIZATION
#define USE_SYSTEM_V_SERIALIZATION

/*
 * Choice of API to be used for shared memory.
 */
#undef USE_POSIX_SHARED_MEMORY
#define USE_SYSTEM_V_SHARED_MEMORY

/*
 * Define if your system supports setpwent(), getpwent(), and endpwent().
 */
#define HAVE_GETPWENT		1

#ifndef HAVE_GETPWENT
#ifndef DEFAULT_PASSWD_FILE
#define DEFAULT_PASSWD_FILE	"/etc/passwd"
#endif
#endif

/*
 * throttle-status refresh time in seconds.
 */
#ifndef DEFAULT_REFRESH
#define DEFAULT_REFRESH		SECONDS_PER_MINUTE
#endif

/*
 * Policy period in seconds used with some policies.
 */
#ifndef DEFAULT_PERIOD
#define DEFAULT_PERIOD		SECONDS_PER_MONTH
#endif

/*
 * Default maximum tolerated delay in seconds for those
 * policies implementing throttle delays.
 */
#ifndef DEFAULT_MAXDELAY
#define DEFAULT_MAXDELAY	30
#endif

/*
 * throttle-status indicator levels expressed as an integer percentage.
 */
#ifndef WARN_GREEN
#define WARN_GREEN		50
#endif

#ifndef WARN_YELLOW
#define WARN_YELLOW		75
#endif

#ifndef WARN_RED
#define WARN_RED		90
#endif

#ifndef WARN_CRITICAL
#define WARN_CRITICAL		100
#endif

/***********************************************************************
 *** No configuration below this point.
 ***********************************************************************/

#define MODULE			"mod_throttle"
#define AUTHOR			"achowe@snert.com"
#define VERSION			"3.1.2"
#define NDEBUG

#define SECONDS_PER_MINUTE	60
#define SECONDS_PER_HOUR	3600
#define SECONDS_PER_DAY		86400L
#define SECONDS_PER_WEEK	(SECONDS_PER_DAY*7)
#define SECONDS_PER_MONTH	(SECONDS_PER_DAY*30)

#include <sys/types.h>
#include <errno.h>
#include <limits.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <fcntl.h>
#include <sys/mman.h>

#include "httpd.h"
#include "http_core.h"
#include "http_config.h"
#include "http_conf_globals.h"
#include "http_log.h"
#include "http_protocol.h"
#include "http_request.h"
#include "ap_config.h"

#define UNSET			-1
#define MERGE(p, c)		(c == UNSET ? p : c)
#define MERGE_PTR(p, c)		((void *) c == (void *) 0 ? p : c)

#define PUSH(top,obj)		((obj)->next = (top), (top) = (obj))
#define POP(top,var)		((var) = (top), (top) = (top)->next)

module MODULE_VAR_EXPORT throttle_module;

typedef struct config t_config;
typedef struct throttle t_throttle;

typedef void (*t_adjust)(request_rec *, t_config *);
typedef int (*t_apply)(request_rec *, t_config *);
typedef int (*t_percent)(t_config *);

typedef struct policy {
	const char *name;
	t_apply apply;
	t_adjust adjust;
	t_percent percent;
} t_policy;

/*
 * Throttle state information to be stored in shared memory.
 */
struct throttle {
	time_t start;			/* Period start time. */
	time_t last;			/* Time of last request. */
	unsigned int delay;		/* Last delay applied. */
	unsigned long volume;		/* Running total of KBytes sent. */
	unsigned long refused;		/* Total requests refused. */
	unsigned long requests;		/* Total requests receieved. */

	unsigned int active;		/* Concurrent requests. */
};

/*
 * Per user, server, virtual host, directory information.
 */
struct config {
	uid_t uid;
	long limit;			/* Policy limit. */
	long period;			/* Policy period. */
	t_policy *policy;		/* Policy functions. */
	const char *name;		/* User, directory, host name. */
	server_rec *server;		/* Associated server. */
	struct config *next;		/* Link list of configurations. */
	struct throttle *track;		/* This throttle's shared data. */
};

static t_config dummy_config;
static t_throttle dummy_throttle;

static const char true[] = "true";
static const char dummy[] = "Dummy";
static const char stype[] = "Server";
static const char dtype[] = "Directory";
static const char text_html[] = "text/html";
static const char text_plain[] = "text/plain";
static const char file_lock[] = "logs/mod_throttle.lock";
static const char file_runtime[] = "logs/mod_throttle.runtime";

static const char x_is_subrequest[] = "x-is-subrequest";
static const char request_handler[] = "request-handler";
static const char is_file_request[] = "is-file-request";
static const char is_throttle_handler[] = "is-throttle-handler";
static const char request_not_counted[] = "request-not-counted";
static const char volume_not_counted[] = "volume-not-counted";
static const char request_content_type[] = "request-content-type";

static const char view_status[] = "status";
static const char throttle_me_str[] = "throttle-me";
static const char throttle_info_str[] = "throttle-info";
static const char throttle_status_str[] = "throttle-status";

#define ALERT_LEVELS 	(sizeof alert_names / sizeof *alert_names)
static char *alert_names[] = { "green", "yellow", "red", "critical" };

static server_rec *main_server;
static unsigned int refresh = DEFAULT_REFRESH;
static unsigned int max_delay = DEFAULT_MAXDELAY;
static const char *content_type = text_html;
static const char *runtime_file = file_runtime;
static const char *lock_file = file_lock;
static unsigned int alert[ALERT_LEVELS] = {
	WARN_GREEN, WARN_YELLOW, WARN_RED, WARN_CRITICAL
};

#if defined(THROTTLE_CLIENT_IP) || defined(THROTTLE_REMOTE_USER)
typedef struct visitor t_visitor;
typedef struct visitor_list t_visitors;
#endif

#ifdef THROTTLE_CLIENT_IP
static t_config client_ip_config;
static t_visitors *client_ip_pool;
static long client_ip_size = 0;
static const char view_client_ip[] = "client-ip";
static const char throttle_client_ip_str[] = "throttle-client-ip";

static t_visitor *get_client_ip(t_visitors *, struct in_addr);
static void reset_client_ip(t_visitors *, const char *, time_t);
#endif

#ifdef THROTTLE_REMOTE_USER
static t_config remote_user_config;
static t_visitors *remote_user_pool;
static long remote_user_size = 0;
static const char view_remote_user[] = "remote-user";
static const char throttle_remote_user_str[] = "throttle-remote-user";

static t_visitor *get_remote_user(t_visitors *, char *);
static void reset_remote_user(t_visitors *, const char *, time_t);
#endif

/*
 * Policy Table & Functions
 */
static int policy_over_limit(request_rec *, t_config *, unsigned long);
static int policy_none(request_rec *, t_config *);
static int policy_concurrent(request_rec *, t_config *);
static int policy_document(request_rec *, t_config *);
static int policy_idle(request_rec *, t_config *);
static int policy_original(request_rec *, t_config *);
static int policy_random(request_rec *, t_config *);
static int policy_request(request_rec *, t_config *);
static int policy_speed(request_rec *, t_config *);
static int policy_volume(request_rec *, t_config *);

static void adjust_reset(request_rec *, t_config *);
static void adjust_original(request_rec *, t_config *);
static void adjust_speed(request_rec *, t_config *);

static int percent_none(t_config *);
static int percent_concurrent(t_config *);
static int percent_document(t_config *);
static int percent_idle(t_config *);
static int percent_random(t_config *);
static int percent_request(t_config *);
static int percent_volume(t_config *);

static t_policy policy_table[] = {
	{ "None", policy_none, adjust_reset, percent_none },
	{ "Concurrent", policy_concurrent, adjust_reset, percent_concurrent },
	{ "Document", policy_document, adjust_reset, percent_document },
	{ "Idle", policy_idle, adjust_reset, percent_idle },
	{ "Original", policy_original, adjust_original, percent_volume },
	{ "Random", policy_random, adjust_reset, percent_random },
	{ "Request", policy_request, adjust_reset, percent_request },
	{ "Speed", policy_speed, adjust_speed, percent_volume },
	{ "Volume", policy_volume, adjust_reset, percent_volume },
	{ (const char *) 0, (t_apply) 0, (t_adjust) 0, (t_percent) 0 }
};

/*
 *
 */
static int cmd_preserve(struct pool *, const char *);
static int cmd_restore(struct pool *, const char *);

/*
 * Semaphore interface wrappers.
 */
typedef struct critical t_critical;
static void critical_child_init(server_rec *, pool *);
static t_critical *critical_create(struct pool *);
static int critical_acquire(t_critical *);
static int critical_release(t_critical *);

static t_critical *critical;

/*
 * Shared memory pool interface.
 */
typedef struct sm_pool t_sm_pool;
static void *sm_pool_alloc(t_sm_pool *, size_t);
static t_sm_pool *sm_pool_create(pool *, size_t);

/***********************************************************************
 *** Support Routines
 ***********************************************************************/

static int
isfalse(char *s)
{
	if (ap_strcasecmp_match(s, "disable") == 0)
		return 1;
	if (ap_strcasecmp_match(s, "false") == 0)
		return 1;
	if (ap_strcasecmp_match(s, "no") == 0)
		return 1;
	if (ap_strcasecmp_match(s, "reset") == 0)
		return 1;
	if (ap_strcasecmp_match(s, "bogus") == 0)
		return 1;
	if (ap_strcasecmp_match(s, "off") == 0)
		return 1;
	if (ap_strcasecmp_match(s, "0") == 0)
		return 1;

	return 0;
}

/*
 * Local version of ap_uname2id() that doesn't kill the server
 * when the user is not found.
 */
static uid_t
uname2id(const char *user)
{
	if (user == (const char *) 0)
		return (uid_t) -1;

#if defined(WIN32) || defined(NETWARE)
	return (uid_t) 1;
#else
{
	struct passwd *pw;

	if ((pw = getpwnam(user)) == (struct passwd *) 0)
		return (uid_t) -1;

	return pw->pw_uid;
}
#endif
}

static const char *
byte_size(pool *p, long size)
{
	const char *units;

	/* When the size value is not multiple of 1024, display
	 * this number in Kbyte units only, since it might be
	 * representing something else unrelated to Kbytes.
	 */
	if (size % 1024 != 0) {
		units = "";
	} else if ((size /= 1024) % 1024 != 0) {
		units = "M";
	} else {
		size /= 1024;
		units = "G";
	}

	return ap_psprintf(p, "%ld%s", size, units);
}

static const char *
time_period(pool *p, long period)
{
	int units;

	if (period % 60 != 0 || period < 60) {
		units = 's';
	} else if ((period /= 60) < 60) {
		units = 'm';
	} else if ((period /= 60) < 24) {
		units = 'h';
	} else if ((period /= 24) < 7) {
		units = 'd';
	} else {
		period /= 7;
		units = 'w';
	}

	return ap_psprintf(p, "%ld%c", period, units);
}

static const char *
elapsed_time(pool * p, unsigned long s)
{
	int h, m;
	unsigned long d;

	d = s / SECONDS_PER_DAY;
	s %= SECONDS_PER_DAY;

	h = s / SECONDS_PER_HOUR;
	s %= SECONDS_PER_HOUR;

	m = s / SECONDS_PER_MINUTE;
	s %= SECONDS_PER_MINUTE;

	if (0 < d)
		return ap_psprintf(p, "%lu+%02d:%02d.%02d", d, h, m, (int) s);
	if (0 < h)
		return ap_psprintf(p, "%d:%02d.%02d", h, m, (int) s);
	if (0 < m)
		return ap_psprintf(p, "%d.%02d", m, (int) s);

	return ap_psprintf(p, "%d", (int) s);
}

/***********************************************************************
 *** Critical Section Routines
 ***********************************************************************/

/*
 * Define one of the following:
 *
 *	USE_POSIX_SERIALIZATION
 * 	USE_FCNTL_SERIALIZATION
 * 	USE_FLOCK_SERIALIZATION
 * 	USE_SYSTEM_V_SERIALIZATION
 */
#if defined(USE_POSIX_SERIALIZATION)

#include <semaphore.h>

struct critical {
	sem_t sem;
};

static void
critical_child_init(server_rec *s, pool *p)
{
	/* Do nothing. */
}

static void
critical_cleanup(void *obj)
{
	errno = 0;
	do {
		(void) sem_destroy(& ((t_critical *) obj)->sem);
	} while (errno == EBUSY);
}

static t_critical *
critical_create(struct pool *p)
{
	t_critical *mp = (t_critical *) ap_palloc(p, sizeof *mp);

	if (sem_init(&mp->sem, 1, 1) < 0) {
		perror("critical_create(): cannot initialise unnamed semaphore");
		exit(APEXIT_INIT);
	}

	ap_register_cleanup(p, mp, critical_cleanup, ap_null_cleanup);

	return mp;
}

static int
critical_acquire(t_critical *mp)
{
	int rc;

	for (errno = 0; (rc = sem_wait(&mp->sem)) < 0; ) {
		if (errno != EINTR) {
			/*** We really should kill the server here. ***/
			perror("critical_acquire() failed");
			ap_start_shutdown();
			return -1;
		}
	}

	return rc;
}

static int
critical_release(t_critical *mp)
{
	int rc;

	for (errno = 0; (rc = sem_post(&mp->sem)) < 0; ) {
		if (errno != EINTR) {
			/*** We really should kill the server here. ***/
			perror("critical_release() failed");
			ap_start_shutdown();
			return -1;
		}
	}

	return rc;
}

#elif defined(USE_FCNTL_SERIALIZATION)

struct critical {
	int fd;
	const char *filename;
	struct flock lock_it;
	struct flock unlock_it;
};

static void
critical_child_init(server_rec *s, pool *p)
{
	/* Do nothing. */
}

static t_critical *
critical_create(struct pool *p)
{
	t_critical *mp = (t_critical *) ap_palloc(p, sizeof *mp);

	/* Set up the lock file name. */
	mp->filename = ap_psprintf(
		p, "%s.%lu", ap_server_root_relative(p, (char *) lock_file),
		(unsigned long) getpid()
	);

	mp->lock_it.l_whence = SEEK_SET;	/* from current point */
	mp->lock_it.l_start = 0;		/* -"- */
	mp->lock_it.l_len = 0;			/* until end of file */
	mp->lock_it.l_type = F_WRLCK;		/* set exclusive/write lock */
	mp->lock_it.l_pid = 0;			/* pid not actually interesting */
	mp->unlock_it.l_whence = SEEK_SET;	/* from current point */
	mp->unlock_it.l_start = 0;		/* -"- */
	mp->unlock_it.l_len = 0;		/* until end of file */
	mp->unlock_it.l_type = F_UNLCK;		/* set exclusive/write lock */
	mp->unlock_it.l_pid = 0;		/* pid not actually interesting */

	mp->fd = ap_popenf(p, mp->filename, O_CREAT|O_WRONLY|O_EXCL, 0600);

	if (mp->fd < 0) {
		perror("critical_create(): cannot create lock file");
		exit(APEXIT_INIT);
	}

	(void) unlink(mp->filename);

	return mp;
}

static int
critical_acquire(t_critical *mp)
{
	int rc;

	for (errno = 0; (rc = fcntl(mp->fd, F_SETLKW, &mp->lock_it)) < 0; ) {
		if (errno != EINTR) {
			/*** We really should kill the server here. ***/
			perror("critical_acquire() failed");
			ap_start_shutdown();
			return -1;
		}
	}

	return rc;
}

static int
critical_release(t_critical *mp)
{
	int rc;

	for (errno = 0; (rc = fcntl(mp->fd, F_SETLKW, &mp->unlock_it)) < 0; ) {
		if (errno != EINTR) {
			/*** We really should kill the server here. ***/
			perror("critical_release() failed");
			ap_start_shutdown();
			return -1;
		}
	}

	return rc;
}

#elif defined(USE_FLOCK_SERIALIZATION)

struct critical {
	int fd;
	const char *filename;
};

static void
critical_child_init(server_rec *s, pool *p)
{
	/* Attach child process to inherited global critical. */
	critical->fd = ap_popenf(p, critical->filename, O_WRONLY, 0600);

	if (critical->fd < 0) {
		ap_log_error(
			APLOG_MARK, APLOG_EMERG, s,
			"Child cannot open lock file: %s",
			critical->filename
		);

		ap_start_shutdown();
	}
}

static void
critical_cleanup(void *obj)
{
	(void) unlink(((t_critical *) obj)->filename);
}

static t_critical *
critical_create(struct pool *p)
{
	t_critical *mp = (t_critical *) ap_palloc(p, sizeof *mp);

	/* Set up the lock file name. */
	mp->filename = ap_psprintf(
		p, "%s.%lu", ap_server_root_relative(p, (char *) lock_file),
		(unsigned long) getpid()
	);
	(void) unlink(mp->filename);

	/* Apache's lock file code creates the lock file owned by root
	 * with 0600 permissions and each child opens it as root before
	 * the change of uid & gid.
	 *
	 * We don't have this luxury, since the child_init() code executes
	 * after the change in the child server's uid and gid. So chown()
	 * the lock file in advance to belong to the child server uid.
	 */
	mp->fd = ap_popenf(p, mp->filename, O_CREAT|O_WRONLY|O_EXCL, 0600);

	if (mp->fd < 0) {
		perror("critical_create(): cannot create lock file");
		exit(APEXIT_INIT);
	}

	if (chown(mp->filename, ap_user_id, ap_group_id) < 0) {
		perror("critical_create(): cannot set lock file ownership");
		exit(APEXIT_INIT);
	}

	ap_register_cleanup(p, mp, critical_cleanup, ap_null_cleanup);

	return mp;
}

static int
critical_acquire(t_critical *mp)
{
	int rc;

	for (errno = 0; (rc = flock(mp->fd, LOCK_EX)) < 0; ) {
		if (errno != EINTR) {
			/*** We really should kill the server here. ***/
			perror("critical_acquire() failed");
			ap_start_shutdown();
			return -1;
		}
	}

	return rc;
}

static int
critical_release(t_critical *mp)
{
	if (flock(mp->fd, LOCK_UN) < 0) {
		/*** We really should kill the server here. ***/
		perror("critical_release() failed");
		ap_start_shutdown();
		return -1;
	}

	return 0;
}

#elif defined(USE_SYSTEM_V_SERIALIZATION)

#include <sys/ipc.h>
#include <sys/sem.h>

#if (defined(__GNU_LIBRARY__) && (!defined(_SEM_SEMUN_UNDEFINED))) || defined(__FreeBSD__) || defined(__NetBSD__)
/* union semun is defined by including <sys/sem.h> */
#else
/* X/OPEN says we have to define it ourselves (twits). */
union semun {
	int val;			/* value for SETVAL */
	struct semid_ds *buf;		/* buffer for IPC_STAT, IPC_SET */
	unsigned short int *array;	/* array for GETALL, SETALL */
	struct seminfo *__buf;		/* buffer for IPC_INFO */
};
#endif

struct critical {
	int id;
	struct sembuf on;
	struct sembuf off;
};

static void
critical_child_init(server_rec *s, pool *p)
{
	/* Do nothing. */
}

static void
critical_cleanup(void *obj)
{
	union semun ick;
	t_critical *mp = (t_critical *) obj;

	if (0 <= mp->id) {
		ick.val = 0;
		(void) semctl(mp->id, 0, IPC_RMID, ick);
	}
}

static t_critical *
critical_create(struct pool *p)
{
	union semun ick;
	t_critical *mp = (t_critical *) ap_palloc(p, sizeof *mp);

	if ((mp->id = semget(IPC_PRIVATE, 1, IPC_CREAT|0600)) < 0) {
		perror("critical_create(): semget() failed");
		exit(APEXIT_INIT);
	}

	ick.val = 1;
	if (semctl(mp->id, 0, SETVAL, ick) < 0) {
		perror("critical_create(): semctl(SETVAL) failed");
		exit(APEXIT_INIT);
	}

	if (getuid() == 0) {
		struct semid_ds buf;

		buf.sem_perm.uid = ap_user_id;
		buf.sem_perm.gid = ap_group_id;
		buf.sem_perm.mode = 0600;
		ick.buf = &buf;

		if (semctl(mp->id, 0, IPC_SET, ick) < 0) {
			perror("critical_create(): semctl(IPC_SET) failed");
			exit(APEXIT_INIT);
		}
	}

	ap_register_cleanup(p, (void *) mp, critical_cleanup, ap_null_cleanup);

	mp->on.sem_num = 0;
	mp->on.sem_op = -1;
	mp->on.sem_flg = SEM_UNDO;
	mp->off.sem_num = 0;
	mp->off.sem_op = 1;
	mp->off.sem_flg = SEM_UNDO;

	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_INFO, main_server,
		"allocated semaphore #%d", mp->id
	);

	return mp;
}

static int
critical_acquire(t_critical *mp)
{
	for (errno = 0; semop(mp->id, &mp->on, 1) < 0; ) {
		if (errno != EINTR) {
			/*** We really should kill the server here. ***/
			perror("critical_acquire() failed");

			/* Neither of these calls appear to shutdown the
			 * server and its children; exit(APEXIT_CHILDFATAL),
			 * appears to kill only the parent process.
			 */
			ap_start_shutdown();
			return -1;
		}
	}

	return 0;
}

static int
critical_release(t_critical *mp)
{
	for (errno = 0; semop(mp->id, &mp->off, 1) < 0; ) {
		if (errno != EINTR) {
			/*** We really should kill the server here. ***/
			perror("critical_release() failed");

			/* Neither of these calls appear to shutdown the
			 * server and its children; exit(APEXIT_CHILDFATAL),
			 * appears to kill only the parent process.
			 */
			ap_start_shutdown();
			return -1;
		}
	}

	return 0;
}

#else
/*
 * Default, no serialization!
 */
typedef int t_critical;

t_critical *critical;

static void
critical_child_init(server_rec *s, pool *p)
{
	/* Do nothing. */
}

static t_critical *
critical_create(struct pool *p)
{
	/* No critical defined. */
	return (t_critical *) -1;
}

static int
critical_acquire(t_critical *mp)
{
	return 0;
}

static int
critical_release(t_critical *mp)
{
	return 0;
}

#endif

/***********************************************************************
 *** Shared Memory Routines
 ***********************************************************************/

struct sm_pool {
	void *pool;
	size_t max;
	size_t length;
	t_critical critical;
};

static void *
sm_pool_alloc(t_sm_pool *sm, size_t size)
{
	void *mem;

	if (sm->max < sm->length + size)
		return (void *) 0;

	mem = sm->pool + sm->length;
	sm->length += size;

	return mem;
}

/*
 * Define one of the following:
 *
 *	USE_POSIX_SHARED_MEMORY
 * 	USE_SYSTEM_V_SHARED_MEMORY
 */
#if defined(USE_POSIX_SHARED_MEMORY)

#ifdef USE_SHM_OPEN
static void
sm_pool_cleanup(void *data)
{
	shm_unlink("/" MODULE);
}
#endif

static t_sm_pool *
sm_pool_create(pool *p, size_t size)
{
	int fd;
	t_sm_pool *sm = (t_sm_pool *) ap_pcalloc(p, sizeof *sm);

	ap_block_alarms();
	errno = 0;

#ifdef USE_SHM_OPEN
/* For some reason, using shm_open() and mmap() on Solaris 5.7,
 * causes a SIGBUS when the shared memory address returned from
 * mmap() is read or written, despite the fact that we don't get
 * any errors from shm_open() and mmap().
 */
	if ((fd = shm_open("/" MODULE, O_RDWR|O_CREAT, S_IRUSR|S_IWUSR)) < 0) {
		ap_log_error(
			APLOG_MARK, APLOG_EMERG, main_server,
			"Failed to create a shared memory object."
		);
		exit(APEXIT_INIT);
	}
#else
/* Since POSIX mmap() works with file objects and most systems
 * will probably come with a /dev/zero, we can use this instead,
 * which works happily on Solaris 5.7.
 */
	if ((fd = open("/dev/zero", O_RDWR, 0600)) < 0) {
		ap_log_error(
			APLOG_MARK, APLOG_EMERG, main_server,
			"Failed to open shared opbject."
		);
		exit(APEXIT_INIT);
	}
#endif

	sm->pool = mmap((void *) 0, size, PROT_READ|PROT_WRITE, MAP_SHARED, fd, (off_t) 0);
	if (sm->pool == MAP_FAILED) {
		shm_unlink("/" MODULE);
		ap_log_error(
			APLOG_MARK, APLOG_EMERG, main_server,
			"Failed to map the shared object."
		);
		exit(APEXIT_INIT);
	}

	sm->max = size;
	sm->length = 0;
	(void) close(fd);
	(void) memset(sm->pool, 0, size);

#ifdef USE_SHM_OPEN
	ap_register_cleanup(p, NULL, sm_pool_cleanup, ap_null_cleanup);
#endif

	ap_unblock_alarms();

	return sm;
}

#elif defined(USE_SYSTEM_V_SHARED_MEMORY)

#include <sys/ipc.h>
#include <sys/shm.h>

static void
sm_pool_cleanup(void *sm)
{
	(void) shmdt(sm);
}

static t_sm_pool *
sm_pool_create(pool *p, size_t size)
{
	int id;
#ifdef MOVEBREAK
	char *obrk;
#endif
	struct shmid_ds shmbuf;
	t_sm_pool *sm = (t_sm_pool *) ap_pcalloc(p, sizeof *sm);

	if ((id = shmget(IPC_PRIVATE, size, IPC_CREAT|SHM_R|SHM_W)) < 0) {
		ap_log_error(
			APLOG_MARK, APLOG_EMERG, main_server,
			"Failed to allocated shared memory."
		);
		exit(APEXIT_INIT);
	}

	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_INFO, main_server,
		"Created shared memory segment #%d", id
	);

#ifdef MOVEBREAK
	/*
	 * Some SysV systems place the shared segment WAY too close
	 * to the dynamic memory break point (sbrk(0)). This severely
	 * limits the use of malloc/sbrk in the program since sbrk will
	 * refuse to move past that point.
	 *
	 * To get around this, we move the break point "way up there",
	 * attach the segment and then move break back down. Ugly
	 */
	if ((obrk = sbrk(MOVEBREAK)) == (char *) -1) {
		ap_log_error(
			APLOG_MARK, APLOG_ERR, main_server,
			"sbrk() could not move break"
		);
	}
#endif
	sm->max = size;
	sm->length = 0;

	if ((sm->pool = shmat(id, (char *) 0, 0)) == (void *) -1) {
		ap_log_error(
			APLOG_MARK, APLOG_EMERG, main_server,
			"shmat() could not attach segment #%d", id
		);
	} else if (shmctl(id, IPC_STAT, &shmbuf) < 0) {
		ap_log_error(
			APLOG_MARK, APLOG_EMERG, main_server,
			"shmctl() could not stat segment #%d", id
		);
	} else {
		shmbuf.shm_perm.uid = ap_user_id;
		shmbuf.shm_perm.gid = ap_group_id;

		if (shmctl(id, IPC_SET, &shmbuf) != 0) {
			ap_log_error(
				APLOG_MARK, APLOG_ERR, main_server,
				"shmctl() could not set segment #%d", id
			);
		}
	}

	/* We must avoid leaving segments in the kernel's (small) tables. */
	if (shmctl(id, IPC_RMID, NULL) != 0) {
		ap_log_error(
			APLOG_MARK, APLOG_WARNING, main_server,
			"shmctl() could not remove shared memory segment #%d",
			id
		);
	}

	if (sm->pool == (void *) -1)
		exit(APEXIT_INIT);

	(void) memset(sm->pool, 0, size);

#ifdef MOVEBREAK
	if (obrk == (char *) -1)
		return sm;			/* nothing else to do */

	if (sbrk(-(MOVEBREAK)) == (char *) -1) {
		ap_log_error(
			APLOG_MARK, APLOG_ERR, main_server,
			"sbrk() could not move break back"
		);
	}
#endif

	/* For restarts, detach the shared memory segment, then we
	 * safely reallocate a new segment during configuration phase.
	 */
	ap_register_cleanup(p, sm->pool, sm_pool_cleanup, ap_null_cleanup);

	return sm;
}

#else
/*
 * Default, assume threads where all memory is shared!
 */
static t_sm_pool *
sm_pool_create(pool *p, size_t size)
{
	t_sm_pool *sm = (t_sm_pool *) ap_pcalloc(p, sizeof *sm);

	sm->pool = ap_pcalloc(p, size);
	sm->max = size;
	sm->length = 0;

	return sm;
}
#endif

#if defined(THROTTLE_CLIENT_IP) || defined(THROTTLE_REMOTE_USER)
/***********************************************************************
 *** Visitor Pool Routines
 ***********************************************************************/

#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

struct visitor {
	time_t start;			/* Period start time. */
	time_t last;			/* Time of last request. */
	unsigned int delay;		/* Last delay applied. */
	unsigned long volume;		/* Running total of KBytes sent. */
	unsigned long refused;		/* Total requests refused. */
	unsigned long requests;		/* Total requests receieved. */

	union {
		char user[16];		/* Authenticated remote user. */
		struct in_addr ip;	/* IP of last connection. */

	} remote;

	struct visitor *next;
};

struct visitor_list {
	int used;
	struct visitor *head;
	struct visitor base[1];
};

static t_visitor dummy_visitor;

#ifdef THROTTLE_CLIENT_IP
/*
 * NOTE this is a critical section.
 */
static t_visitor *
get_client_ip(t_visitors *vp, struct in_addr ip)
{
	t_visitor *v, **prev = &vp->head;

	if (vp->head == (t_visitor *) 0)
		return &dummy_visitor;

	for (v = vp->head; v->next != (t_visitor *) 0; v = v->next) {
		if (v->remote.ip.s_addr == 0 || v->remote.ip.s_addr == ip.s_addr)
			break;
		prev = &v->next;
	}

	/* Move the entry to the front of the list. */
	*prev = v->next;
	PUSH(vp->head, v);

	/* The last entry in the list might match. */
	if (v->remote.ip.s_addr != ip.s_addr) {
		if (v->remote.ip.s_addr == 0) vp->used++;
		v->volume = v->refused = v->requests = v->delay = 0;
		v->start = v->last = time((time_t *) 0) - 1;
		v->remote.ip = ip;
	}

	return v;
}

static void
reset_client_ip(t_visitors *vp, const char *args, time_t when)
{
	int all;
	t_visitor *v;
	struct in_addr ip;

	ip.s_addr = inet_addr(args);
	all = (args[0] == '*' && args[1] == '\0');

	(void) critical_acquire(critical);

	for (v = vp->head; v != (t_visitor *) 0; v = v->next) {
		if (all || v->remote.ip.s_addr == ip.s_addr) {
			v->volume = v->refused = v->requests = v->delay = 0;
			v->start = v->last = when - 1;
			if (!all) break;
		}
	}

	(void) critical_release(critical);
}
#endif

#ifdef THROTTLE_REMOTE_USER
/*
 * NOTE this is a critical section.
 */
static t_visitor *
get_remote_user(t_visitors *vp, char *user)
{
	t_visitor *v, **prev = &vp->head;

	if (vp->head == (t_visitor *) 0 || user == (char *) 0)
		return (t_visitor *) 0;

	for (v = vp->head; v->next != (t_visitor *) 0; v = v->next) {
		if (*v->remote.user == '\0' || ap_strcasecmp_match(v->remote.user, user) == 0)
			break;
		prev = &v->next;
	}

	/* Move the entry to the front of the list. */
	*prev = v->next;
	PUSH(vp->head, v);

	/* The last entry in the list might match. */
	if (ap_strcasecmp_match(v->remote.user, user) != 0) {
		if (*v->remote.user == '\0') vp->used++;
		(void) ap_cpystrn(v->remote.user, user, sizeof v->remote.user);
		v->volume = v->refused = v->requests = v->delay = 0;
		v->start = v->last = time((time_t *) 0) - 1;
	}

	(void) critical_release(critical);

	return v;
}

static void
reset_remote_user(t_visitors *vp, const char *args, time_t when)
{
	int all;
	t_visitor *v;

	all = strchr(args, '*') != (char *) 0;

	(void) critical_acquire(critical);

	for (v = vp->head; v != (t_visitor *) 0; v = v->next) {
		if (ap_strcasecmp_match(v->remote.user, args) == 0) {
			v->volume = v->refused = v->requests = v->delay = 0;
			v->start = v->last = when - 1;
			if (!all) break;
		}
	}

	(void) critical_release(critical);
}
#endif

#endif /* defined(THROTTLE_CLIENT_IP) || defined(THROTTLE_REMOTE_USER) */

/***********************************************************************
 ***  Create, Merge, and Initialise Routines
 ***********************************************************************/

/* Link list/stack of the assorted things that we throttle. */
static t_config *config_stack = (t_config *) 0;
static t_config *stack_top = (t_config *) 0;
static unsigned int config_count = 0;
static unsigned int stack_count = 0;

static void *
create_dir_config(struct pool *p, char *dir)
{
	t_config *config = (t_config *) ap_pcalloc(p, sizeof *config);

	config->name = ap_pstrdup(p, dir);
	config->uid = (uid_t) UNSET;
	config->period = UNSET;
	config->limit = UNSET;

	/* Push this new entry onto the stack. */
	config->next = config_stack;
	config_stack = config;

	++config_count;

#ifndef NDEBUG
	fprintf(
		stderr, "create_dir_config(%s) %lx\n",
		dir == (char *) 0 ? "(null)" : dir, config
	);
#endif
	return (void *) config;
}

static void *
merge_dir_config(struct pool *p, void *dad, void *kid)
{
	t_config *child = (t_config *) kid;
	t_config *parent = (t_config *) dad;

	if (child->name == (const char *) 0)
		child->name = child->server->server_hostname;

	/* If a value in the child is unset, then inherit the setting of
	 * the parent, which may or may not be the default, otherwise use
	 * the child's value.
	 */
	child->uid = MERGE(parent->uid, child->uid);
	child->limit = MERGE(parent->limit, child->limit);
	child->period = MERGE(parent->period, child->period);
	child->policy = MERGE_PTR(parent->policy, child->policy);

	/* We don't bother merging the server & info fields, since every
	 * entry will have its own. See init_module().
	 */

#ifndef NDEBUG
	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, main_server,
		"Merge parent=%s, child=%s, policy=%s\n",
		parent->name, child->name, child->policy->name
	);
#endif

	return (void *) child;
}

static void
cleanup_module(void *data)
{
	(void) cmd_preserve((struct pool *) data, (const char *) 0);
}

static void
init_module(server_rec *server, struct pool *p)
{
	int i;
	size_t size;
	void *sm_pool;
	t_config *config = (t_config *) ap_get_module_config(
		server->lookup_defaults, &throttle_module
	);

#ifndef NDEBUG
	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, server,
		"init_module server=%s; config_count=%u",
		server->server_hostname, config_count
	);
#endif

	/* For error logging. */
	main_server = server;

	/* Dummy configuration used when no configuration specified. */
	dummy_config.policy = &policy_table[0];
	dummy_config.track = &dummy_throttle;
	dummy_config.period = LONG_MAX;
	dummy_config.limit = LONG_MAX;
	dummy_config.server = server;
	dummy_config.name = dummy;

	/* Our initial server configuration is unset, so at this point
	 * we set our defaults. These will be overridden by directives
	 * below. It will be this parent server data that will be later
	 * used to merge with or inherit from.
	 */
	config->server = server;
	config->name = server->server_hostname;

	if (config->policy == (t_policy *) 0) {
		config->policy = &policy_table[0];
		if (config->period == UNSET) {
			/* When unset, default to 30 days for policy none,
			 * which is a reasonable period for the accumulation
			 * of data.
			 */
			config->period = SECONDS_PER_MONTH;
		}
	}

	if (config->period == UNSET)
		config->period = DEFAULT_PERIOD;

	if (config->limit == UNSET)
		config->limit = 0;

	/* The first pass of the configuration which occurs prior to
	 * init_module() being called, will generate our count.
	 */
	size = config_count * sizeof(t_throttle);

#ifdef THROTTLE_CLIENT_IP
	size += sizeof(t_visitors) + client_ip_size * sizeof(t_visitor);
#endif
#ifdef THROTTLE_REMOTE_USER
	size += sizeof(t_visitors) + remote_user_size * sizeof(t_visitor);
#endif

	sm_pool = sm_pool_create(p, size);

	/* Allocate server, virtual host, directory, location, and
	 * local user throttles.
	 */
	for (config = config_stack; config != (t_config *) 0; config = config->next) {
		config->track = (t_throttle *) sm_pool_alloc(sm_pool, sizeof (t_throttle));
		if (config->track != (t_throttle *) 0) {
			config->track->start =
			config->track->last = time((time_t *) 0)-1;
		}
	}

	/* Remember the stack head for in a runtime variable, and reset
	 * the configuration stack for the next restart.
	 */
	stack_top = config_stack;
	config_stack = (t_config *) 0;

	stack_count = config_count;
	config_count = 0;

#ifdef THROTTLE_CLIENT_IP
{
	t_visitor *v;
	long n = client_ip_size;

	client_ip_pool = sm_pool_alloc(
		sm_pool, sizeof(t_visitors) + n * sizeof(t_visitor)
	);

	/* Link up the list of client-ip entries. */
	for (v = client_ip_pool->base; 0 <= n; --n, ++v) {
		PUSH(client_ip_pool->head, v);
	}

	client_ip_config.server = server;

	/* We use this field to identify regular throttle configurations
	 * vs. the global configuration used to throttle client-ip.
	 */
	client_ip_config.name = throttle_client_ip_str;
	client_ip_config.uid = UNSET;

	if (client_ip_config.policy == (t_policy *) 0)
		client_ip_config.policy = &policy_table[0];

	if (client_ip_config.period <= 0)
		client_ip_config.period = SECONDS_PER_HOUR;
}
#endif
#ifdef THROTTLE_REMOTE_USER
{
	t_visitor *v;
	long n = remote_user_size;

	remote_user_pool = sm_pool_alloc(
		sm_pool, sizeof(t_visitors) + n * sizeof(t_visitor)
	);

	/* Link up the list of remote-user entries. */
	for (v = remote_user_pool->base; 0 <= n; --n, ++v) {
		PUSH(remote_user_pool->head, v);
	}

	remote_user_config.server = server;

	/* We use this field to identify regular throttle configurations
	 * vs. the global configuration used to throttle client-ip.
	 */
	remote_user_config.name = throttle_remote_user_str;
	remote_user_config.track = &dummy_throttle;
	remote_user_config.uid = UNSET;

	if (remote_user_config.policy == (t_policy *) 0)
		remote_user_config.policy = &policy_table[0];

	if (remote_user_config.period <= 0)
		remote_user_config.period = SECONDS_PER_HOUR;
}
#endif

	critical = critical_create(p);
	srand((unsigned int) time((time_t *) 0));
        ap_add_version_component("mod_throttle/"VERSION);

	if (cmd_restore(p, (const char *) 0) < 0) {
		ap_log_error(
			APLOG_MARK, APLOG_ERR, server,
			"restore from \"%s\" failed", runtime_file
		);
		exit(APEXIT_INIT);
	}

	/* This must be last since cleanups are done in reverse order
	 * and we want to preserve the data before we release our
	 * shared memory and semaphores.
	 */
	ap_register_cleanup(p, p, cleanup_module, ap_null_cleanup);
}

static void
child_init(server_rec *s, pool *p)
{
	critical_child_init(s, p);
}

/***********************************************************************
 ***  Special Runtime Commands
 ***********************************************************************/

static int
cmd_preserve(struct pool *p, const char *args)
{
	FILE *fp;
	const char *fn;
	t_config *config;

	/* Preserve the runtime data in a file in order to save state
	 * information across shutdowns and restarts.
	 */
	fn = ap_server_root_relative(p, (char *) runtime_file);
	if ((fp = ap_pfopen(p, fn, "w")) == (FILE *) 0)
		return -1;

	/* Write out the number of section entries. */
	(void) fprintf(fp, "throttle=%lu\n", stack_count);

	for (config = stack_top; config != (t_config *) 0; config = config->next) {
		if (config->track == (t_throttle *) 0)
			continue;

		(void) fprintf(
			fp, "%s\t%lx\t%lx\t%u\t%lu\t%lu\t%lu\n",
			config->name,
			config->track->start,
			config->track->last,
			config->track->delay,
			config->track->volume,
			config->track->refused,
			config->track->requests
		);
	}

#if defined(THROTTLE_CLIENT_IP) || defined(THROTTLE_REMOTE_USER)
{
	t_visitor *v;

#ifdef THROTTLE_CLIENT_IP
	(void) fprintf(fp, "client-ip=%lu\n", client_ip_pool->used);

	for (v = client_ip_pool->head; v != (t_visitor *) 0; v = v->next) {
		if (v->remote.ip.s_addr == 0)
			break;

		(void) fprintf(
			fp, "%s\t%lx\t%lx\t%u\t%lu\t%lu\t%lu\n",
			inet_ntoa(v->remote.ip),
			v->start,
			v->last,
			v->delay,
			v->volume,
			v->refused,
			v->requests
		);
	}
#endif
#ifdef THROTTLE_REMOTE_USER
	(void) fprintf(fp, "remote-user=%lu\n", remote_user_pool->used);

	for (v = remote_user_pool->head; v != (t_visitor *) 0; v = v->next) {
		if (*v->remote.user == '\0')
			break;

		(void) fprintf(
			fp, "%s\t%lx\t%lx\t%u\t%lu\t%lu\t%lu\n",
			v->remote.user,
			v->start,
			v->last,
			v->delay,
			v->volume,
			v->refused,
			v->requests
		);
	}
#endif
}
#endif /* defined(THROTTLE_CLIENT_IP) || defined(THROTTLE_REMOTE_USER) */

	if (ap_pfclose(p, fp) < 0)
		return -1;

	/* Make sure that when we shutdown or restart, the file ownership
	 * of the runtime file is that of the child server process. This
	 * allows the child process to perform preserve & restore commands.
	 */
	if (getuid() == 0)
		return chown(fn, ap_user_id, ap_group_id);

	return 0;
}

/*
 * Restore the runtime data previoisly preserveded in a file prior to a
 * shutdown or restart.
 */
static int
cmd_restore(struct pool *p, const char *args)
{
	int rc;
	FILE *fp;
	t_config *config;
	unsigned long count;
	const char *fn, *fmt, *buf;

	p = ap_make_sub_pool(p);
	fn = ap_server_root_relative(p, (char *) runtime_file);
	if ((fp = ap_pfopen(p, fn, "r")) == (FILE *) 0) {
		ap_destroy_pool(p);
		return 0;
	}

	/* Create a buffer large enough to hold a pathname and terminating
	 * null byte. Then to avoid possible buffer overrun create the
	 * fscanf() format string specifying the buffer's byte length.
	 */
	buf = ap_pcalloc(p, PATH_MAX+1);
	fmt = ap_psprintf(p, "%%%lds", (long) PATH_MAX);

	if (fscanf(fp, "throttle=%lu ", &count) != 1)
		return 1;

	/* Read the name at the start of this line. */
	while (0 < count-- && fscanf(fp, fmt, buf) == 1) {
		/* Find named configuration. */
		for (config = stack_top; config != (t_config *) 0; config = config->next) {
			if (ap_strcasecmp_match(buf, config->name) == 0)
				break;
		}

		/* Has the configuration been removed? */
		if (config == (t_config *) 0 || config->track == (t_throttle *) 0) {
			/* Skip remainder of the line. */
			(void) fscanf(fp, "%*[^\n]");
			continue;
		}

		/* Restore the configuration's previous running data. */
		rc = fscanf(
			fp, "%lx %lx %u %lu %lu %lu ",
			&config->track->start,
			&config->track->last,
			&config->track->delay,
			&config->track->volume,
			&config->track->refused,
			&config->track->requests
		);

		if (rc != 6)
			break;
	}

#ifdef THROTTLE_CLIENT_IP
	if (fscanf(fp, "client-ip=%lu ", &count) != 1)
		return 1;

	if (client_ip_size < count)
		count = client_ip_size;

	(void) critical_acquire(critical);

	/* Read the name at the start of this line. */
	while (0 < count-- && fscanf(fp, fmt, buf) == 1) {
		t_visitor *v;
		struct in_addr ip;

		ip.s_addr = inet_addr(buf);
		v = get_client_ip(client_ip_pool, ip);

		rc = fscanf(
			fp, "%lx %lx %u %lu %lu %lu ",
			&v->start,
			&v->last,
			&v->delay,
			&v->volume,
			&v->refused,
			&v->requests
		);

		if (rc != 6)
			break;
	}

	(void) critical_release(critical);
#endif
#ifdef THROTTLE_REMOTE_USER
	if (fscanf(fp, "remote-user=%lu ", &count) != 1)
		return 1;

	if (remote_user_size < count)
		count = remote_user_size;

	(void) critical_acquire(critical);

	/* Read the name at the start of this line. */
	while (0 < count-- && fscanf(fp, fmt, buf) == 1) {
		t_visitor *v;

		v = get_remote_user(remote_user_pool, (char *) buf);
		if (v == (t_visitor *) 0)
			continue;

		rc = fscanf(
			fp, "%lx %lx %u %lu %lu %lu ",
			&v->start,
			&v->last,
			&v->delay,
			&v->volume,
			&v->refused,
			&v->requests
		);

		if (rc != 6)
			break;
	}

	(void) critical_release(critical);
#endif

	(void) ap_pfclose(p, fp);
	ap_destroy_pool(p);

	return 0;
}

/***********************************************************************
 ***  Directive Routines
 ***********************************************************************/

static const char *
set_policy3(pool *p, t_config *config, char *policy, char *limit, char *period)
{
	t_policy *pp;

	for (pp = policy_table; pp->name != (const char *) 0; ++pp) {
		if (ap_strcasecmp_match(policy, pp->name) == 0)
			break;
	}

	if (pp->name == (const char *) 0)
		return "Invalid policy.";

	config->policy = pp;

	config->limit = strtol(limit, (char **) &limit, 10);
	if (config->limit < 0)
		config->limit = 0;

	/* The limit value will normally, but not necessarily, be an
	 * expression in kilo-bytes, mega-bytes, or giga-bytes. Even
	 * when the limit doesn't refer to kilo-byte units, treat it
	 * as such for the qualifiers. Ignore anything larger since
	 * it will overflow unsigned long on 32-bit systems.
	 */
	switch (ap_toupper(*limit)) {
	default:
		return "Invalid units for limit.";
	case 'G':
		config->limit *= 1024;
	case 'M':
		config->limit *= 1024;
	case 'K': case '\0':
		break;
	}

	if (config->policy->apply == policy_random && 100 < config->limit)
		return "Random policy requires a percentage limit (0..100).";

	if (period != (char *) 0 && ap_isdigit(*period)) {
		config->period = strtol(period, (char **) &period, 10);

		if (config->period <= 0)
			config->period = 1;

		/* The period value can be expressed in terms of seconds,
		 * minutes, hours, days, and weeks.
		 */
		switch (ap_tolower(*period)) {
		default:
			return "Invalid units for period.";
		case 'w':
			config->period *= 7;
		case 'd':
			config->period *= 24;
		case 'h':
			config->period *= 60;
		case 'm':
			config->period *= 60;
		case 's': case '\0':
			break;
		}
	} else if (config->policy->apply == policy_none) {
		config->period = SECONDS_PER_MONTH;
	} else {
		config->period = DEFAULT_PERIOD;
	}

	return (const char *) 0;
}

static const char *
set_policy(pool *p, t_config *config, const char *args)
{
	char *policy, *limit, *period;

	if (config == (t_config *) 0)
		return (const char *) 0;

	/* First argument is required. */
	if ((policy = ap_getword_white(p, &args)) == (char *) 0)
		return "Policy not specified.";

	/* Second argument required for everything but "off". */
	if ((limit = ap_getword_white(p, &args)) == (char *) 0)
		return "Limit not specified.";

	/* Third argument optional. */
	period = ap_getword_white(p, &args);

	return set_policy3(p, config, policy, limit, period);
}

/*
 * ThrottlePolicy {policy} [limit [period]]
 *
 * Context: server
 */
static const char *
throttle_policy(cmd_parms *cmd, void *dconfig, const char *args)
{
	t_config *config = (t_config *) dconfig;

	if (config == (t_config *) 0)
		return (const char *) 0;

	config->server = cmd->server;
#ifndef NDEBUG
{
	const char *what, *where;

	if (cmd->path == (char *) 0 || *cmd->path == '\0') {
		where = config->server->server_hostname;
		what = stype;
	} else {
		where = cmd->path;
		what = dtype;
	}

	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, cmd->server,
		"%s %s: ThrottlePolicy %s", what, where, args
	);
}
#endif
	return set_policy(cmd->temp_pool, config, args);
}

static const char *
throttle_single_user(cmd_parms *cmd, const char *user, const char *args)
{
	uid_t uid;
	t_config *config;

	/* Skip users who are not defined in the system user database. */
	if ((uid = uname2id(user)) == (uid_t) -1)
		return (const char *) 0;

	/* Has this user been previously defined? */
	for (config = config_stack; config != (t_config *) 0; config = config->next) {
		if (config->uid == uid)
			break;
	}

	if (config == (t_config *) 0)
		config = create_dir_config(cmd->pool, (char *) user);

	config->server = cmd->server;
	config->uid = uid;

#ifndef NDEBUG
	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, cmd->server,
		"ThrottleUser %s %s", user, args
	);
#endif
	return set_policy(cmd->temp_pool, config, args);
}

/*
 * ThrottleUser {user} {policy} [limit [period]]
 * ThrottleUser    *   {policy} [limit [period]]
 * ThrottleUser /path  {policy} [limit [period]]
 *
 * Context: server
 */
static const char *
throttle_user(cmd_parms *cmd, void *dconfig, const char *args)
{
	const char *user, *buf, *fmt, *err;

	if ((user = ap_getword_white(cmd->temp_pool, &args)) == (char *) 0)
		return "User ID, *, or pathname not specified.";

	/* Create a throttle for all the users that appear in the
	 * system user database.
	 */
	if (user[0] == '*' && user[1] == '\0') {
#ifdef HAVE_GETPWENT
		struct passwd *pw;

		setpwent();
		while ((pw = getpwent()) != (struct passwd *) 0) {
			err = throttle_single_user(cmd, pw->pw_name, args);
			if (err != (const char *) 0)
				break;
		}
		endpwent();

		return err;
#else
		user = DEFAULT_PASSWD_FILE;
#endif
	}

	/* Create a throttle for all the users that appear in the first
	 * field of each line of the file. The field separator is a colon
	 * (:). This format is common to both /etc/passwd and .htpasswd
	 * files.
	 */
	if (ap_os_is_path_absolute(user)) {
		FILE *fp;

		if ((fp = ap_pfopen(cmd->temp_pool, user, "r")) == (FILE *) 0)
			return "ThrottleUser file not found.";

		/* Create a buffer large enough to hold a line and terminating
		 * null byte. Then to avoid possible buffer overrun create the
		 * fscanf() format string specifying the buffer's byte length.
		 */
		buf = ap_pcalloc(cmd->temp_pool, BUFSIZ);
		fmt = ap_psprintf(cmd->temp_pool, "%%%ld[^:]%%*[^\n] ", (long) BUFSIZ-1);

		/* Read the user name at the start of this line. */
		while (fscanf(fp, fmt, buf) == 1) {
			err = throttle_single_user(cmd, buf, args);
			if (err != (const char *) 0)
				break;
		}

		(void) ap_pfclose(cmd->temp_pool, fp);

		return err;
	}

	return throttle_single_user(cmd, user, args);
}

#ifdef THROTTLE_CLIENT_IP
/*
 * ThrottleClientIP {size} {policy} [limit [period]]
 *
 * Context: server
 */
static const char *
throttle_client_ip(cmd_parms *cmd, void *dconfig, const char *args)
{
	client_ip_size = strtol(args, (char **) &args, 10);

#ifndef NDEBUG
	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, cmd->server,
		"ThrottleClientIP %s", args
	);
#endif

	if (client_ip_size == 0 && !ap_isspace(*args))
		return "Client IP pool size not specified.";

	/* Remove one from the count, since the visitor list structure
	 * declares one entry already for us.
	 */
	client_ip_size--;

	while (ap_isspace(*args)) ++args;

	return set_policy(cmd->temp_pool, &client_ip_config, args);
}
#endif

#ifdef THROTTLE_REMOTE_USER
/*
 * ThrottleRemoteUser {size} {policy} [limit [period]]
 *
 * Context: server
 */
static const char *
throttle_remote_user(cmd_parms *cmd, void *dconfig, const char *args)
{
	remote_user_size = strtol(args, (char **) &args, 10);

#ifndef NDEBUG
	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, cmd->server,
		"ThrottleRemoteUser %s", args
	);
#endif

	if (remote_user_size == 0 && !ap_isspace(*args))
		return "Remote user pool size not specified.";

	/* Remove one from the count, since the visitor list structure
	 * declares one entry already for us.
	 */
	remote_user_size--;

	while (ap_isspace(*args)) ++args;

	return set_policy(cmd->temp_pool, &remote_user_config, args);
}
#endif

/*
 * ThrottleRuntimeFile {filename}
 *
 * Context: server
 */
static const char *
throttle_runtime_file(cmd_parms *cmd, void *dconfig, char *filename)
{
	runtime_file = ap_pstrdup(cmd->pool, filename);

	return (const char *) 0;
}

/*
 * ThrottleLockFile {filename}
 *
 * Context: server
 */
static const char *
throttle_lock_file(cmd_parms *cmd, void *dconfig, char *filename)
{
	lock_file = ap_pstrdup(cmd->pool, filename);

	return (const char *) 0;
}

/*
 * ThrottleMaxDelay {seconds}
 *
 * Context: server
 */
static const char *
throttle_max_delay(cmd_parms *cmd, void *dconfig, char *seconds)
{
	long secs = strtol(seconds, (char **) 0, 10);

	if (secs < 0)
		max_delay = 0;
	else if (UINT_MAX < secs)
		max_delay = UINT_MAX;
	else
		max_delay = (unsigned int) secs;

	return (const char *) 0;
}

/*
 * ThrottleRefresh {seconds}
 *
 * Context: server
 */
static const char *
throttle_refresh(cmd_parms *cmd, void *dconfig, char *seconds)
{
	long secs = strtol(seconds, (char **) 0, 10);

	if (secs < 0)
		refresh = 0;
	else if (UINT_MAX < secs)
		refresh = UINT_MAX;
	else
		refresh = (unsigned int) secs;

	return (const char *) 0;
}

/*
 * ThrottleContentType {mime_type}
 *
 * where mime_type is either "text/html" or "text/plain"
 *
 * Context: server
 */
static const char *
throttle_content_type(cmd_parms *cmd, void *dconfig, char *mime_type)
{
	if (ap_strcasecmp_match(mime_type, text_html) == 0)
		content_type = text_html;
	else if (ap_strcasecmp_match(mime_type, text_plain) == 0)
		content_type = text_plain;
	else
		return "Supported content-types are: text/html, text/plain";

	return (const char *) 0;
}

/*
 * ThrottleIndicator {green | yellow | red | critical} {percentage}
 *
 * Context: server
 */
static const char *
throttle_indicator(cmd_parms *cmd, void *dconfig, char *indicator, char *num)
{
	int i;

	for (i = 0; i < ALERT_LEVELS; ++i) {
		if (ap_strcasecmp_match(indicator, alert_names[i]) == 0) {
			alert[i] = strtol(num, (char **) 0, 10);
			return (const char *) 0;
		}
	}

	return "Invalid indicator";
}

/***********************************************************************
 ***  Policy Handlers
 ***********************************************************************/

/*
 * Reset the period and collected data.
 *
 * NOTE that this function is a *critical section* and should be wrapped
 * by critical_acquire() & critical_release(). I use to do it within this
 * function, which is called within a loop sometimes, but moved it out
 * a level or two to improve speed (even just a little).
 */
static void
reset_info(t_config *config, time_t when)
{
	if (config->track == (t_throttle *) 0)
		return;

	/* Avoid possible divide by zero errors when we calculate a
	 * time difference and compute something per second.
	 */
	config->track->start = config->track->last = when - 1;
	config->track->requests = 0;
	config->track->refused = 0;
	config->track->volume = 0;
	config->track->delay = 0;
}

/*
 * Reset a throttle by its name be it user, directory, or host.
 * If the name is "*", then all the throttles will be reset.
 */
static int
reset_info_match(const char *args, time_t when)
{
	t_config *config;

	(void) critical_acquire(critical);

	for (config = stack_top; config != (t_config *) 0; config = config->next) {
		if (ap_strcasecmp_match(config->name, args) == 0) {
			reset_info(config, when);

			if (args[0] != '*' || args[1] != '\0')
				break;
		}
	}

	(void) critical_release(critical);

	return 0;
}

static int
busy_signal(request_rec *r, t_config *config)
{
	(void) critical_acquire(critical);
	config->track->refused++;
	(void) critical_release(critical);

	return HTTP_SERVICE_UNAVAILABLE;
}

/*
 * Check value against the limit and refuse the connection if exceeded.
 */
static int
policy_over_limit(request_rec *r, t_config *config, unsigned long value)
{
#ifndef NDEBUG
	ap_log_rerror(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r,
		"(%ld) policy_over_limit server=%s, request=%s",
		(long) getpid(), ap_get_server_name(r), r->the_request
	);
#endif

	if (config->limit < value) {
		ap_log_rerror(
			APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING, r,
			"%s policy %s limit of %ld exceeded, %ld",
			config->name, config->policy->name, config->limit, value
		);
		return busy_signal(r, config);
	}

	return DECLINED;
}

/*
 * NOTE is a critical section. See access_handler().
 */
static void
adjust_reset(request_rec *r, t_config *config)
{
	reset_info(config, r->request_time);
}

/*
 * ThrottlePolicy none {ignored} {period}
 *
 * No policy enforced, just monitoring. The period is used only to
 * periodically reset the counters.
 */
static int
policy_none(request_rec *r, t_config *config)
{
	return policy_over_limit(r, config, 0);
}

/*
 * Return 0 percent.
 */
static int
percent_none(t_config *config)
{
	/* Do nothing. */
	return 0;
}

/*
 * ThrottlePolicy concurrent {limit} {period}
 *
 * Impose a limit on the number of concurrent requests at any one time.
 * The period is used only to periodically reset the counters.
 *
 * TEST COMMENTS:
 *
 * As a policy, this one doesn't appear to work so well, possibly because
 * the requests from one client may happen sequentially; when refused, a
 * client may make a certain number of attempts before giving up; a client
 * may only maintain a limited number of requests / connection at any one
 * time which is less than the limit.
 *
 * To test this properly, browsers (IE, NS, Opera, wget) on multiple machines
 * must make requests at the same time. The number of refused requests should
 * increase.
 */
static int
policy_concurrent(request_rec *r, t_config *config)
{
#ifdef THROTTLE_CLIENT_IP
	if (config->name == throttle_client_ip_str)
		return DECLINED;
#endif
#ifdef THROTTLE_REMOTE_USER
	if (config->name == throttle_remote_user_str)
		return DECLINED;
#endif

	return policy_over_limit(r, config, config->track->active);
}

/*
 * Return percentage of requests relative to our limit.
 */
static int
percent_concurrent(t_config *config)
{
	if (config->limit <= 0)
		return 0;

	return config->track->active * 100 / config->limit;
}

/*
 * ThrottlePolicy request {limit} {period}
 *
 * Impose a limit on the number of requests per period. When this limit is
 * exceeded all further requests are refused, until the end of the period at
 * which time the period and request count are reset.
 */
static int
policy_request(request_rec *r, t_config *config)
{
	return policy_over_limit(r, config, config->track->requests);
}

/*
 * Return percentage of requests relative to our limit.
 */
static int
percent_request(t_config *config)
{
	if (config->limit <= 0)
		return 0;

	return config->track->requests * 100 / config->limit;
}

/*
 * ThrottlePolicy document {limit} {period}
 *
 * Excluding requests for HTML page elements such as images and style sheets,
 * impose a limit on the number of requests per period. When this limit is
 * exceeded all further requests are refused, until the end of the period at
 * which time the period and request count are reset.
 */
static int
policy_document(request_rec *r, t_config *config)
{
	const char **pe;
	request_rec *sub;
	static const char *page_elements[] = {
		"image/*", "audio/*", "text/css", "text/*script", (char *) 0
	};

#ifndef NDEBUG
	ap_log_rerror(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r,
		"(%ld) policy_document server=%s, request=%s",
		(long) getpid(), ap_get_server_name(r), r->the_request
	);
#endif

	/* Is it a page element request? */
	ap_table_setn(r->headers_in, x_is_subrequest, true);
	sub = ap_sub_req_lookup_uri(r->uri, r);
	if (sub->content_type != (const char *) 0) {
		for (pe = page_elements; *pe != (const char *) 0; ++pe) {
			if (ap_strcasecmp_match(sub->content_type, *pe) == 0) {
				ap_table_setn(r->notes, request_not_counted, true);
				break;
			}
		}
	}
	ap_destroy_sub_req(sub);
	ap_table_unset(r->headers_in, x_is_subrequest);

	/* Have we exceeded our request limit?. */
	if (0 < config->limit && config->limit < config->track->requests) {
		ap_log_rerror(
			APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING, r,
			"%s limit of %ld documents exceeded",
			config->name, config->limit
		);
		return busy_signal(r, config);
	}

	return DECLINED;
}

/*
 * Return percentage of active requests relative to our limit.
 */
static int
percent_document(t_config *config)
{
	if (config->limit <= 0)
		return 0;

	return config->track->requests * 100 / config->limit;
}

/*
 * ThrottlePolicy random {percent} {period}
 *
 * Randomly accept a percentage of the requests. If the percentage
 * is zero (0), then every request is refused; if the percentage
 * is 100, then all requests are accepted. The period is used only
 * to periodically reset the counters.
 */
static int
policy_random(request_rec *r, t_config *config)
{
	return policy_over_limit(r, config, rand() % 100);
}

/*
 * Return percentage of requests accepted.
 */
static int
percent_random(t_config *config)
{
	if (config->track->requests <= 0)
		return 0;

	return 100 - config->track->refused * 100 / config->track->requests;
}

/*
 * ThrottlePolicy volume {kbytes} {period}
 *
 * Impose a limit on the volume (kbytes sent) per period. When this limit is
 * exceeded all further requests are refused, until the end of the period at
 * which time the period and volume count are reset.
 */
static int
policy_volume(request_rec *r, t_config *config)
{
	return policy_over_limit(r, config, config->track->volume);
}

/*
 * Return percentage of volume relative to our limit.
 */
static int
percent_volume(t_config *config)
{
	if (config->limit <= 0)
		return 0;

	return config->track->volume * 100 / config->limit;
}

/*
 * ThrottlePolicy idle {minimum} {period}
 *
 * Impose a mimimum idle time between requests. When the miminum is not
 * reached, then the request incurs a calculated delay penalty or is
 * refused.
 *
 * First, whenever the elapsed time exceeds the period length, then the
 * counters are reset.
 *
 * Second, if the idle time between requests exceeds the minimum, then the
 * the request proceeds without delay. Otherwise the request is delayed
 * between one and ThrottleMaxDelay seconds. If the delay would exceed
 * ThrottleMaxDelay, then we refuse the request entirely to avoid occupying
 * servers unnecessarily.
 *
 * The delay is computed as the minimum less the idle time between requests.
 */
static int
policy_idle(request_rec *r, t_config *config)
{
	unsigned long idle;

#ifndef NDEBUG
	ap_log_rerror(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r,
		"(%ld) policy_idle server=%s, request=%s",
		(long) getpid(), ap_get_server_name(r), r->the_request
	);
#endif

	idle = r->request_time - config->track->last;

	if (idle < config->limit) {
		(void) critical_acquire(critical);
		config->track->delay = config->limit - idle;
		(void) critical_release(critical);

		if (0 < max_delay && max_delay < config->track->delay) {
#ifdef POLICY_DELAY_ONLY
			(void) critical_acquire(critical);
			config->track->delay = max_delay;
			(void) critical_release(critical);
#else
			ap_log_rerror(
				APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING, r,
				"%s delay=%lu too large",
				config->name, config->track->delay
			);
			return busy_signal(r, config);
#endif
		}

		sleep(config->track->delay);
	} else if (0 < config->track->delay) {
		(void) critical_acquire(critical);
		config->track->delay = 0;
		(void) critical_release(critical);
	}

	return DECLINED;
}

static int
percent_idle(t_config *config)
{
	return config->track->delay * 100 / config->limit;
}

/*
 * ThrottlePolicy speed {kbytes} {period}
 *
 * Impose a limit on the volume (kbytes sent) per period, which when exceeded
 * the request incurs a calculated delay penalty or is refused.
 *
 * First, whenever the elapsed time exceeds the period length, then the limit
 * (allowance) is deducted from the volume, which cannot be a negative result;
 * also the period length is deducted from the elapse time.
 *
 * Second, if the volume is below the limit, in which case the request
 * proceeds without delay. Otherwise the request is delayed between one and
 * ThrottleMaxDelay seconds. If the delay would exceed ThrottleMaxDelay, then
 * we refuse the request entirely to avoid occupying servers unnecessarily.
 *
 * The delay is currently computed as one plus the integer result of the
 * volume times 10 divided by the limit.
 */
static int
policy_speed(request_rec *r, t_config *config)
{
#ifndef NDEBUG
	ap_log_rerror(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r,
		"(%ld) policy_speed server=%s, request=%s",
		(long) getpid(), ap_get_server_name(r), r->the_request
	);
#endif

	/* Second, have we exceeded our limit (allowance)? */
	if (0 < config->limit && config->limit < config->track->volume) {
		/* Compute the delay required to maintain the speed limit.
		 * We keep track of it only for the throttle status, since
		 * the field is there for policy_original().
		 */
		(void) critical_acquire(critical);
		config->track->delay = (config->track->volume * 10) / config->limit + 1;
		(void) critical_release(critical);

		/* When we exeed our upper bounds for an acceptable
		 * delay, then refuse the request instead. When
		 * this starts happening too often either our max. delay
		 * is set too low or our speed limit is too low.
		 */
		if (0 < max_delay && max_delay < config->track->delay) {
#ifdef POLICY_DELAY_ONLY
			(void) critical_acquire(critical);
			config->track->delay = max_delay;
			(void) critical_release(critical);
#else
			ap_log_rerror(
				APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING, r,
				"%s delay=%lu too large",
				config->name, config->track->delay
			);
			return busy_signal(r, config);
#endif
		}

		sleep(config->track->delay);
	} else if (0 < config->track->delay) {
		/* We have fallen below the limit and must update the delay
		 * for the status display.
		 */
		(void) critical_acquire(critical);
		config->track->delay = 0;
		(void) critical_release(critical);
	}

	return DECLINED;
}

/*
 * NOTE is a critical section. See access_handler().
 */
static void
adjust_speed(request_rec *r, t_config *config)
{
	/* Reset these every period. */
	config->track->delay = 0;
	config->track->refused = 0;
	config->track->requests = 0;

	/* Deduct limit from the volume for this period. We keep
	 * the excess and count it towards the next period.
	 */
	if (config->limit < config->track->volume)
		config->track->volume -= config->limit;
	else
		config->track->volume = 0;

	/* Advance to the next period. */
	config->track->start += config->period;
}

/*
 * ThrottlePolicy original {KBps} {period}
 *
 * Original mod_throttle 2.0 heuristic.
 *
 * Impose a limit on the volume (kbytes sent) per period, which when exceeded
 * the request incurs a counter-based delay penalty or is refused.
 *
 * First, whenever the elapsed time exceeds the period length, then the volume
 * and elapsed time are halved.
 *
 * Second, if the volume is below the limit, then the delay counter is
 * decreased by one second if it is not yet zero. Otherwise, when the limit
 * is exeeded, the delay counter is increased by one second. The delay can be
 * between zero and ThrottleMaxDelay seconds, after which the request will
 * be refused to avoid occupying servers unnecessarily.
 */
static int
policy_original(request_rec *r, t_config *config)
{
	long slack;

#ifndef NDEBUG
	ap_log_rerror(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r,
		"(%ld) policy_original server=%s, request=%s",
		(long) getpid(), ap_get_server_name(r), r->the_request
	);
#endif

	/* The original policy had a slack time applied so that the
	 * delay would not kick in so quickly after a restart. We
	 * have dropped ThrottleSlack in favour of using 1/5 of the
	 * period for the slack, because ThrottleSlack was a global
	 * and the period can now vary, its too confusing to try
	 * and find a suitable value that is less than all the period.
	 */
	slack = config->period / 5;

	/* When we are within our limit, decrease the delay; otherwise
	 * increase the delay upto but not exceeding max_delay seconds
	 * when max_delay is > 0.
	 */
	(void) critical_acquire(critical);

	if (config->track->volume <= config->limit) {
		if (0 < config->track->delay)
			--config->track->delay;
	} else if (config->track->delay <= max_delay || max_delay <= 0) {
		++config->track->delay;
	}

	(void) critical_release(critical);

	/* Are we monitoring only? */
	if (config->limit <= 0)
		return DECLINED;

#ifndef POLICY_DELAY_ONLY
	/* Allow the delay to adjusted (hopefully downwards) before
	 * considering to refuse the connection.
	 */
	if (0 < max_delay && max_delay < config->track->delay) {
		return busy_signal(r, config);
	}
#endif

	/* Muhahaha - throttle the buggers! */
	if (0 < config->track->delay)
		sleep(config->track->delay);

	return DECLINED;
}

/*
 * NOTE is a critical section. See access_handler().
 */
static void
adjust_original(request_rec *r, t_config *config)
{
	config->track->start += (r->request_time - config->track->start) / 2;
	config->track->volume /= 2;
}

/***********************************************************************
 ***  Request Phase Handlers
 ***********************************************************************/

/*
 * Return true if this a simple file request, as oppose to a directory,
 * dynamic document, CGI, etc.
 */
static int
is_request_for_file(request_rec *r)
{
	const char *handler = ap_table_get(r->notes, request_handler);
	const char *mime = ap_table_get(r->notes, request_content_type);

	/* Is it something other than a regular file? */
	if (r->finfo.st_mode == 0 || !S_ISREG(r->finfo.st_mode))
		return 0;

	/* Do we have a mime type? */
	if (mime == (const char *) 0)
		return 0;

	/* Do we have a special handler for it? */
	if (handler != (const char *) 0)
		return 0;

	/* Some special mime type that is a parsed file? */
	if (ap_strcmp_match(mime, "application/x-httpd-*") == 0)
		return 0;

	/* Assume its a simple file. */
	return 1;
}

/*
 * Determine who our content-response handler is and parse the query string.
 */
static int
uri_handler(request_rec *r)
{
	int not_for_us;
	request_rec *sub;
	char *arg, *key, *value;

	if (!ap_is_initial_req(r))
		return DECLINED;

	/* Find out what handler is going to be called. */
	ap_table_setn(r->headers_in, x_is_subrequest, true);
	sub = ap_sub_req_lookup_uri(r->uri, r);

	not_for_us = sub->handler == (char *) 0
		|| ap_strcmp_match(sub->handler, "throttle-*") != 0;
	ap_table_set(r->notes, request_handler, sub->handler);
	ap_table_set(r->notes, request_content_type, sub->content_type);

	if (is_request_for_file(sub))
		ap_table_setn(r->notes, is_file_request, true);

	ap_destroy_sub_req(sub);
	ap_table_unset(r->headers_in, x_is_subrequest);

	/* Check for arguments only when its for our handler. */
	if (not_for_us)
		return DECLINED;

	/* Remember this fact for later phases. */
	ap_table_setn(r->notes, is_throttle_handler, true);

	/* Parse the query string into the notes table. */
	if (r->args != (char *) 0) {
		for (arg = r->args; *arg != '\0'; ) {
			value = ap_getword_nc(r->pool, &arg, '&');
			if (value == (char *) 0)
				break;

			key = ap_getword_nc(r->pool, &value, '=');
			if (key == (char *) 0 || ap_unescape_url(key) != OK)
				continue;

			if (ap_unescape_url(value) != OK)
				continue;

			ap_table_setn(r->notes, key, value);
		}
	}

	return OK;
}

#ifdef THROTTLE_CLIENT_IP
/*
 * Throttle based on the client IP address' usage.
 */
static int
access_handler(request_rec *r)
{
	/* Bypass access check on subrequest we make. */
	if (ap_table_get(r->headers_in, x_is_subrequest) == true)
		return OK;

	/* Avoid throttling status requests, but subject them to
	 * other access controls.
	 */
	if (ap_table_get(r->notes, is_throttle_handler) == true)
		return DECLINED;

	if (client_ip_size <= 0 || !ap_is_initial_req(r))
		return DECLINED;

#ifndef NDEBUG
	ap_log_rerror(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r,
		"(%ld) access_handler ip=%s, request=%s",
		(long) getpid(), r->remote_ip, r->the_request
	);
#endif

	(void) critical_acquire(critical);

	/* Lookup any throttle information for this client IP address for
	 * which we will apply the global policy for client connections.
	 */
	client_ip_config.track = (t_throttle *) get_client_ip(
		client_ip_pool, r->connection->remote_addr.sin_addr
	);

	/* Is it time for the period adjustment? */
	if (client_ip_config.period <= (r->request_time - client_ip_config.track->start)) {
		(*client_ip_config.policy->adjust)(r, &client_ip_config);
	}

	/* Add in the request size now if we can. */
	if (ap_table_get(r->notes, is_file_request) == true) {
		unsigned long kbytes = (r->finfo.st_size + 512) / 1024;
		ap_table_setn(r->notes, volume_not_counted, true);
		client_ip_config.track->volume += kbytes;
	}

	(void) critical_release(critical);

	return (*client_ip_config.policy->apply)(r, &client_ip_config);
}
#else
/*
 * Bypass access checks on subrequest generated by us.
 */
static int
access_handler(request_rec *r)
{
	if (ap_table_get(r->headers_in, x_is_subrequest) == true)
		return OK;

	return DECLINED;
}
#endif

/*
 * Bypass authentication checks on subrequest generated by us.
 */
static int
authentication_handler(request_rec *r)
{
	if (ap_table_get(r->headers_in, x_is_subrequest) == true)
		return OK;

	return DECLINED;
}

#ifdef THROTTLE_REMOTE_USER
/*
 * Throttle based on the supplied remote user name
 */
static int
authorization_handler(request_rec *r)
{
	t_visitor *remote_user;

	/* Bypass authorization check on subrequest we make. */
	if (ap_table_get(r->headers_in, x_is_subrequest) == true)
		return OK;

	/* Avoid throttling status requests, but subject them to
	 * other authorization checks.
	 */
	if (ap_table_get(r->notes, is_throttle_handler) == true)
		return DECLINED;

	if (remote_user_size <= 0 || !ap_is_initial_req(r))
		return DECLINED;

#ifndef NDEBUG
	ap_log_rerror(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r,
		"(%ld) authorization_handler user=%s request=%s",
		(long) getpid(), r->connection->user, r->the_request
	);
#endif

	(void) critical_acquire(critical);

	remote_user = get_remote_user(remote_user_pool, r->connection->user);
	if (remote_user == (t_visitor *) 0)
		return DECLINED;

	remote_user_config.track = (t_throttle *) remote_user;

	/* Is it time for the period adjustment? */
	if (remote_user_config.period <= (r->request_time - remote_user_config.track->start)) {
		(*remote_user_config.policy->adjust)(r, &remote_user_config);
	}

	/* Add in the request size now if we can. */
	if (ap_table_get(r->notes, is_file_request) == true) {
		unsigned long kbytes = (r->finfo.st_size + 512) / 1024;
		ap_table_setn(r->notes, volume_not_counted, true);
		remote_user_config.track->volume += kbytes;
	}

	(void) critical_release(critical);

	return (*remote_user_config.policy->apply)(r, &remote_user_config);
}
#else
/*
 * Bypass authorization checks on subrequest generated by us.
 */
static int
authorization_handler(request_rec *r)
{
	if (ap_table_get(r->headers_in, x_is_subrequest) == true)
		return OK;

	return DECLINED;
}
#endif

static t_config *
get_config_by_name(char *name)
{
	t_config *config;

	for (config = stack_top; config != (t_config *) 0; config = config->next) {
		if (ap_strcasecmp_match(name, config->name) == 0)
			return config;
	}

	return &dummy_config;
}

static t_config *
get_config_by_uid(uid_t uid)
{
	t_config *config;

	for (config = stack_top; config != (t_config *) 0; config = config->next) {
		if (uid == config->uid)
			return config;
	}

	return &dummy_config;
}

static t_config *
get_config_by_dir(request_rec *r)
{
	t_config *config = (t_config *) ap_get_module_config(
		r->per_dir_config, &throttle_module
	);

	if (config == (t_config *) 0)
		return &dummy_config;

	return config;
}

/*
 * Specify content-type of throttle status page requests or apply policies
 * specified by ThrottleUser and ThrottlePolicy for all other requests.
 */
static int
mime_handler(request_rec *r)
{
	int rc;
	const char *arg;
	t_config *user, *host;

	/* Only specify the content-type of our throttle status pages. */
	if (ap_table_get(r->notes, is_throttle_handler) == true) {
		/* On a previous subrequest we found out what our
		 * handler would be from mod_mime. Restore that handler
		 * here, since mod_mime will be bypassed.
		 */
		r->handler = ap_table_get(r->notes, request_handler);

		/* Determine the response's content-type. */
		arg = ap_table_get(r->notes, "content-type");
		if (arg == (const char *) 0)
			r->content_type = content_type;
		else if (arg == text_plain)
			r->content_type = text_plain;
		else
			r->content_type = text_html;

		return OK;
	}

	if (!ap_is_initial_req(r))
		return DECLINED;

	/* Get configurations for ThrottleUser and/or ThrottlePolicy. */
	user = get_config_by_uid(r->finfo.st_uid);
	host = get_config_by_dir(r);

	/* Count concurrent requests & perform adjustments. */
	(void) critical_acquire(critical);

	user->track->active++;
	if (user->period < (r->request_time - user->track->start))
		(*user->policy->adjust)(r, user);

	host->track->active++;
	if (host->period < (r->request_time - host->track->start))
		(*host->policy->adjust)(r, host);

	/* Add in the request size now if we can. */
	if (ap_table_get(r->notes, is_file_request) == true) {
		unsigned long kbytes = (r->finfo.st_size + 512) / 1024;
		ap_table_setn(r->notes, volume_not_counted, true);
		user->track->volume += kbytes;
		host->track->volume += kbytes;
	}

	(void) critical_release(critical);

	/* Apply ThrottleUser and/or ThrottlePolicy. */
	rc = (*user->policy->apply)(r, user);
	if (ap_is_HTTP_ERROR(rc))
		return rc;

	return (*host->policy->apply)(r, host);
}

static int
fixup_handler(request_rec *r)
{
	unsigned int rsec;
	char *key, *value, *arg, *view;

	if (!ap_is_initial_req(r))
		return DECLINED;

	if (ap_table_get(r->notes, is_throttle_handler) != true)
		return DECLINED;

	/* Response's refresh time. */
	if ((arg = (char *) ap_table_get(r->notes, "refresh")) == (char *) 0)
		rsec = refresh;
	else
		rsec = (unsigned int) strtol(arg, (char **) 0, 10);

	arg = ap_psprintf(r->pool, "%u", rsec);
	ap_table_setn(r->notes, "refresh", arg);
	if (0 < rsec)
		ap_table_setn(r->headers_out, "Refresh", arg);

	/* The throttle-me handler is intend for induhvidual users,
	 * who are not allowed to preform any commands what so ever.
	 */
	if (ap_strcmp_match(r->handler, throttle_me_str) == 0)
		return OK;

	/* The view is specified when one display handler links to another
	 * without being able to change the URI used to build the URL. We
	 * cannot change the URI, because we don't know if the handler
	 * has been declared in httpd.conf nor the location assigned to it.
	 */
	if ((arg = (char *) ap_table_get(r->notes, "view")) != (char *) 0) {
		if (ap_strcasecmp_match(arg, view_status) == 0)
			r->handler = throttle_status_str;
#ifdef THROTTLE_CLIENT_IP
		else if (ap_strcasecmp_match(arg, view_client_ip) == 0)
			r->handler = throttle_client_ip_str;
#endif
#ifdef THROTTLE_REMOTE_USER
		else if (ap_strcasecmp_match(arg, view_remote_user) == 0)
			r->handler = throttle_remote_user_str;
#endif
	}

#ifdef THROTTLE_CLIENT_IP
	if (ap_strcmp_match(r->handler, throttle_client_ip_str) == 0)
		view = (char *) view_client_ip;
	else
#endif
#ifdef THROTTLE_REMOTE_USER
	if (ap_strcmp_match(r->handler, throttle_remote_user_str) == 0)
		view = (char *) view_remote_user;
	else
#endif
		view = (char *) view_status;

	/* Any commands to process? */
	if ((value = (char *) ap_table_get(r->notes, "command")) == (char *) 0)
		return OK;

	/* Lookup command. */
	key = ap_getword_nc(r->pool, &value, ',');

	if (ap_strcasecmp_match(key, "preserve") == 0) {
		(void) cmd_preserve(r->pool, value);
	} else if (ap_strcasecmp_match(key, "restore") == 0) {
		(void) cmd_restore(r->pool, value);
	} else if (ap_strcasecmp_match(key, "reset") == 0) {
#ifdef THROTTLE_CLIENT_IP
		if (view == view_client_ip)
			reset_client_ip(client_ip_pool, value, r->request_time);
		else
#endif
#ifdef THROTTLE_REMOTE_USER
		if (view == view_remote_user)
			reset_remote_user(remote_user_pool, value, r->request_time);
		else
#endif
			reset_info_match(value, r->request_time);
	} else if (ap_strcasecmp_match(key, "restart") == 0) {
		/* Restart the parent & children. */
	} else if (ap_strcasecmp_match(key, "shutdown") == 0) {
		/* Terminate the parent & children cleanly. */
	}

	/* Once we have processed the command, we have to do a redirection
	 * that the client browser doesn't repeatedly refresh and redo the
	 * the command.
	 */
	arg = ap_psprintf(
		r->pool, "%s?content-type=%s&refresh=%u&view=%s",
		r->uri, r->content_type, rsec, view
	);
	r->content_type = text_html;
	arg = ap_construct_url(r->pool, arg, r);
	ap_table_setn(r->headers_out, "Location", arg);
	ap_table_setn(r->notes, volume_not_counted, true);
	ap_table_setn(r->notes, request_not_counted, true);

	return HTTP_MOVED_PERMANENTLY;
}

/*
 * Track the total number of kbytes sent.
 */
int
log_handler(request_rec *r)
{
#ifdef THROTTLE_CLIENT_IP
	t_visitor *client_ip;
#endif
#ifdef THROTTLE_REMOTE_USER
	t_visitor *remote_user;
#endif
	unsigned long kbytes;
	t_config *user, *host;

#ifdef NDEBUG
	/* Don't count the throttle status pages in the stats. */
	if (ap_table_get(r->notes, is_throttle_handler) == true)
		return DECLINED;
#endif

	if (!ap_is_initial_req(r))
		return DECLINED;

	/* In the case of internal redirects, from directory to index.html,
	 * the final sub-request should be the actual number of kbytes sent.
	 */
	while (r->next != (request_rec *) 0)
		r = r->next;

#ifndef NDEBUG
	ap_log_rerror(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r,
		"(%ld) log_handler bytes_sent=%lu, request=%s",
		(long) getpid(), r->bytes_sent, r->the_request
	);
#endif

	kbytes = (r->bytes_sent + 512) / 1024;

	/* We have four levels of throttles to update: client-ip,
	 * remote-user, local-user, server/location.
	 */
	user = get_config_by_uid(r->finfo.st_uid);

	host = get_config_by_dir(r);

	(void) critical_acquire(critical);

#ifdef THROTTLE_CLIENT_IP
	client_ip = get_client_ip(client_ip_pool, r->connection->remote_addr.sin_addr);
#endif
#ifdef THROTTLE_REMOTE_USER
	remote_user = get_remote_user(remote_user_pool, r->connection->user);
	if (remote_user == (t_visitor *) 0)
		remote_user = &dummy_visitor;
#endif

	/* Policies can use notes to control whether or not to count
	 * bytes_sent and/or requests received. Intended for certain
	 * policies that exclude certain types of requests from the
	 * stats.
	 */

	if (ap_table_get(r->notes, volume_not_counted) != true) {
		host->track->volume += kbytes;
		user->track->volume += kbytes;
#ifdef THROTTLE_REMOTE_USER
		remote_user->volume += kbytes;
#endif
#ifdef THROTTLE_CLIENT_IP
		client_ip->volume += kbytes;
#endif
	}

	if (ap_table_get(r->notes, request_not_counted) != true) {
		host->track->requests++;
		user->track->requests++;
#ifdef THROTTLE_REMOTE_USER
		remote_user->requests++;
#endif
#ifdef THROTTLE_CLIENT_IP
		client_ip->requests++;
#endif
	}

	host->track->active--;
	host->track->last = r->request_time;

	user->track->active--;
	user->track->last = r->request_time;

#ifdef THROTTLE_REMOTE_USER
	remote_user->last = r->request_time;
#endif
#ifdef THROTTLE_CLIENT_IP
	client_ip->last = r->request_time;
#endif

	(void) critical_release(critical);

	return DECLINED;
}

/***********************************************************************
 ***  Content Handlers
 ***********************************************************************/

static const char header[] =
	"<html>\n<head>\n<title>%s - %s</title>\n"
	"<style type=\"text/css\">\n"
	".small { font-family: sans-serif; font-size: 8pt }\n"
	".normal, th { font-family: sans-serif; font-size: 10pt }\n"
	".big, h2 { font-family: sans-serif; font-size: 14pt }\n"
	".green { color: #00dd00; font-family: sans-serif; font-size: 10pt; font-weight: bold }\n"
	".yellow { color: #ff9900; font-family: sans-serif; font-size: 10pt; font-weight: bold }\n"
	".red { color: #cc0000; font-family: sans-serif; font-size: 10pt; font-weight: bold }\n"
	".critical { color: #ff0000; font-family: sans-serif; font-size: 10pt; font-weight: bold }\n"
	"</style>\n</head>\n"
	"<body bgcolor=\"#ffffff\" text=\"#000000\" class=\"normal\">\n"
	"<center>\n"
;
static const char footer[] =
	"<p class=\"small\">" MODULE "/" VERSION
	"<br>Copyright 1999, 2000 by <a href=\"mailto:"
	AUTHOR "?subject=" MODULE "/" VERSION
	"\">Anthony Howe</a>.  All rights reserved."
	"</p>\n</center>\n</body>\n</html>\n"
;

static void
status_html_header(request_rec *r)
{
	const char *url;

	url = ap_psprintf(
		r->pool, "%s?content-type=text/html&refresh=%s",
		r->uri, ap_table_get(r->notes, "refresh")
	);
	url = ap_construct_url(r->pool, url, r);

	ap_rprintf(r, header, ap_get_server_name(r), "Throttle Status");

	ap_rprintf(
		r,
		"<table width=\"100%\">\n<tr valign=\"middle\">\n"
		"<th align=\"left\"><h2>%s</h2></th>\n"
		"<th><h2>Server Uptime&nbsp;&nbsp;&nbsp;%s</h2></th>\n"
		"<th align=\"right\"><h2>Throttle Status</h2></th>\n"
		"</tr>\n</table>\n"

		"<table cellpadding=\"0\" cellspacing=\"0\" width=\"100%\">\n"
		"<tr valign=\"bottom\">\n"
			"\t<td colspan=\"2\" class=\"small\">",
		ap_get_server_name(r),
		elapsed_time(r->pool, r->request_time - ap_restart_time)
	);

	/* Only the full status display gets the command links. */
	if (ap_strcmp_match(r->handler, throttle_me_str) != 0) {
		ap_rprintf(
			r,
			"<a href=\"%s&view=status&command=reset,*\">Reset All</a>&nbsp;&nbsp;&nbsp;"
			"<a href=\"%s&view=status&command=preserve\">Preserve</a>&nbsp;&nbsp;&nbsp;"
			"<a href=\"%s&view=status&command=restore\">Restore</a>&nbsp;&nbsp;&nbsp;",
			 url, url, url
		);
#ifdef THROTTLE_CLIENT_IP
		if (0 < client_ip_size)
			ap_rprintf(
				r,
				"<a href=\"%s&view=client-ip\">Client-IP</a>&nbsp;&nbsp;&nbsp;",
				url
			);
#endif
#ifdef THROTTLE_REMOTE_USER
		if (0 < remote_user_size)
			ap_rprintf(
				r,
				"<a href=\"%s&view=remote-user\">Remote-User</a>&nbsp;&nbsp;&nbsp;",
				url
			);
#endif
	}

	ap_rprintf(
		r,
			"</td>\n"
			"\t<th>%%</th>\n"

			"\t<th>Hits</th>\n"
			"\t<th>Refused</th>\n"
			"\t<th>KBytes<br>sent</th>\n"
			"\t<th>KBytes<br>per hit</th>\n"
			"\t<th>Delay<br>(&lt;=%d)</th>\n"

			"\t<th>Policy</th>\n"
			"\t<th>Limit</th>\n"
			"\t<th>Period</th>\n"
			"\t<th>Period<br>Elapsed</th>\n"
			"\t<th>Idle<br>Time</th>\n"
		"</tr>\n",
		max_delay
	);
}

static void
status_html_line(request_rec *r, t_config *config, int row)
{
	int i, percent;
	const char *url, *level;

	/* Compute the policy specific percentage of the limit. */
	percent = (*config->policy->percent)(config);

	for (i = 0; i < ALERT_LEVELS-1; ++i) {
		if (percent < alert[i])
			break;
	}
	level = alert_names[i];

	/* Start table row, alternating background colour. */
	ap_rprintf(
		r, "<tr align=\"right\"%s>\n",
		(row & 1) ? " bgcolor=\"#eeeeff\"" : ""
	);

	/* Display row number. */
	if (ap_strcmp_match(r->handler, throttle_me_str) == 0) {
		ap_rprintf(r, "<td class=\"normal\">%d.&nbsp;</td>\n", row);
	} else {
		url = ap_psprintf(
			r->pool, "%s?content-type=text/html&refresh=%s&view=status&command=reset,%s",
			r->uri, ap_table_get(r->notes, "refresh"), config->name
		);
		url = ap_construct_url(r->pool, url, r);
		ap_rprintf(r, "<td class=\"normal\"><a href=\"%s\">%d.</a>&nbsp;</td>\n", url, row);
	}

	/* Display server, directory, or user being throttled. */
	ap_rprintf(r, "<td align=\"left\" class=\"normal\">");
	if (config->uid == (uid_t) -2 || ap_os_is_path_absolute(config->name)) {
		ap_rprintf(r, config->name);
	} else if (config->uid != UNSET) {
		url = ap_psprintf(r->pool, "/~%s/", config->name);
		url = ap_construct_url(r->pool, url, r);
		ap_rprintf(r, "<a href=\"%s\">%s</a>", url, config->name);
	} else {
		ap_rprintf(
			r, "<a href=\"http://%s:%d/\">%s</a>",
			config->name, config->server->port, config->name
		);
	}
	ap_rprintf(r, "</td>\n");

	/* Display stats. */
	ap_rprintf(
		r,
		"<td class=\"%s\">%u</td>\n"

		"<td class=\"%s\">%lu</td>\n"
		"<td class=\"%s\">%lu</td>\n"
		"<td class=\"%s\">%lu</td>\n"
		"<td class=\"%s\">%lu</td>\n"
		"<td class=\"%s\">%u</td>\n"

		"<td class=\"%s\">%s</td>\n"
		"<td class=\"%s\">%s</td>\n"
		"<td class=\"%s\">%s</td>\n"
		"<td class=\"%s\">%s</td>\n"
		"<td class=\"%s\">%s</td>\n",
		level, percent,

		level, config->track->requests,
		level, config->track->refused,
		level, config->track->volume,
		level, 0 < config->track->requests
			? config->track->volume / config->track->requests
			: 0,
		level, config->track->delay,

		level, config->policy->name,
		level, byte_size(r->pool, config->limit),
		level, time_period(r->pool, config->period),
		level, elapsed_time(r->pool, r->request_time - config->track->start),
		level, elapsed_time(
			r->pool,
			config->track->last < r->request_time
				? r->request_time - config->track->last
				: 0
		)
	);

	/* End table row. */
	ap_rprintf(r, "</tr>\n");
}

static void
status_html_footer(request_rec *r)
{
	ap_rprintf(r, "</table>\n");
	ap_rprintf(r, footer);
}

static void
status_text_line(request_rec *r, t_config *config, int row)
{
	ap_rprintf(
		r,
		"%u. %s"
		"\t%u"

		"\t%lu"
		"\t%lu"
		"\t%lu"
		"\t%lu"
		"\t%u"

		"\t%s"
		"\t%s"
		"\t%s"
		"\t%s"
		"\t%s"
		"\r\n",
		row, config->name,
		config->policy->percent(config),

		config->track->requests,
		config->track->refused,
		config->track->volume,
		0 < config->track->requests
			? config->track->volume / config->track->requests
			: 0,
		config->track->delay,

		config->policy->name,
		byte_size(r->pool, config->limit),
		time_period(r->pool, config->period),
		elapsed_time(
			r->pool, r->request_time - config->track->start
		),
		elapsed_time(
			r->pool,
			config->track->last < r->request_time
				? r->request_time - config->track->last
				: 0
		)
	);
}

static void
status_footer_text(request_rec *r)
{
	/* Do nothing. */
}

static int
server_status(request_rec *r)
{
	int rc, row;
	const char *arg;
	t_config *config;

	if (!ap_is_initial_req(r))
		return DECLINED;

#ifndef NDEBUG
	ap_log_rerror(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r,
		"(%ld) server_status server=%s, request=%s",
		(long) getpid(), ap_get_server_name(r), r->the_request
	);
#endif

	if ((rc = ap_discard_request_body(r)) != OK)
		return rc;

	ap_send_http_header(r);

	if (r->header_only)
		return OK;

	if (r->content_type != text_plain)
		status_html_header(r);

	row = 1;

	for (config = stack_top; config != (t_config *) 0; config = config->next, ++row) {
		if (config->track == (t_throttle *) 0)
			continue;

		if (r->content_type == text_plain)
			status_text_line(r, config, row);
		else
			status_html_line(r, config, row);
	}

	if (r->content_type != text_plain)
		status_html_footer(r);

	return OK;
}

/*
 * Generate an individual throttle status for either a ~user
 * or the server under which the request was made. For example
 * if httpd.conf is configured with:
 *
 *	<Location /throttle-me>
 *	SetHandler throttle-me
 *	</Location>
 *
 * Then a URL of the form "http://my.domain.com/throttle-me" will
 * display the throttle status of "my.domain.com", if it has a
 * ThrottlePolicy or return HTTP_NOT_FOUND.
 */
static int
me_status(request_rec *r)
{
	int rc;
	t_config *config;

	if (!ap_is_initial_req(r))
		return DECLINED;

#ifndef NDEBUG
	ap_log_rerror(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r,
		"(%ld) me_status server=%s, request=%s",
		(long) getpid(), ap_get_server_name(r), r->the_request
	);
#endif

	if ((rc = ap_discard_request_body(r)) != OK)
		return rc;

	if (r->uri[0] == '/' && r->uri[1] == '~' && ap_isalnum(r->uri[2])) {
		/* Lookup user throttle. */
		uid_t uid;
		const char *user, *remainder;

		remainder = r->uri + 2;
		user = ap_getword(r->pool, &remainder, '/');

		if ((uid = uname2id(user)) == (uid_t) -1)
			return HTTP_NOT_FOUND;

		config = get_config_by_uid(uid);
	} else {
		/* Lookup server throttle. */
		for (config = stack_top; config != (t_config *) 0; config = config->next) {
			if (config->name == r->server->server_hostname)
				break;
		}

		if (config == (t_config *) 0)
			return HTTP_NOT_FOUND;
	}

	ap_send_http_header(r);

	if (r->header_only)
		return OK;

	if (r->content_type == text_plain) {
		status_text_line(r, config, 1);
	} else {
		status_html_header(r);
		status_html_line(r, config, 1);
		status_html_footer(r);
	}

	return OK;
}

#if defined(THROTTLE_CLIENT_IP) || defined(THROTTLE_REMOTE_USER)

static void
general_html_header(request_rec *r)
{
	t_config *config;
	const char *url, *title, *view;

	url = ap_psprintf(
		r->pool, "%s?content-type=text/html&refresh=%s",
		r->uri, ap_table_get(r->notes, "refresh")
	);
	url = ap_construct_url(r->pool, url, r);

#ifdef THROTTLE_CLIENT_IP
	if (r->handler == throttle_client_ip_str) {
		config = &client_ip_config;
		view = view_client_ip;
	}
#endif
#ifdef THROTTLE_REMOTE_USER
	if (r->handler == throttle_remote_user_str) {
		config = &remote_user_config;
		view = view_remote_user;
	}
#endif

	ap_rprintf(r, header, ap_get_server_name(r), view);

	ap_rprintf(
		r,
		"<table width=\"100%\">\n<tr valign=\"middle\">\n"
		"<th align=\"left\"><h2>%s</h2></th>\n"
		"<th><h2>Policy: %s&nbsp;&nbsp;&nbsp;&nbsp;Limit: %s&nbsp;&nbsp;&nbsp;&nbsp;Period: %s</h2></th>\n"
		"<th align=\"right\"><h2>%s</h2></th>\n"
		"</tr>\n</table>\n"

		"<table cellpadding=\"0\" cellspacing=\"0\" width=\"100%\">\n"
		"<tr valign=\"bottom\">\n"
			"\t<td colspan=\"2\" class=\"small\">",
		ap_get_server_name(r),
		config->policy->name,
		byte_size(r->pool, config->limit),
		time_period(r->pool, config->period),
		view
	);

	/* Only the full status display gets the command links. */
	ap_rprintf(
		r,
		"<a href=\"%s&view=%s&command=reset,*\">Reset All</a>&nbsp;&nbsp;&nbsp;"
		"<a href=\"%s&view=status\">Status</a>&nbsp;&nbsp;&nbsp;",
		url, view, url
	);

#if defined(THROTTLE_CLIENT_IP) && defined(THROTTLE_REMOTE_USER)
	if (r->handler == throttle_client_ip_str) {
		ap_rprintf(
			r,
			"<a href=\"%s&view=remote-user\">Remote-User</a>&nbsp;&nbsp;&nbsp;",
			url
		);
	}
	if (r->handler == throttle_remote_user_str) {
		ap_rprintf(
			r,
			"<a href=\"%s&view=client-ip\">Client-IP</a>&nbsp;&nbsp;&nbsp;",
			url
		);
	}
#endif

	ap_rprintf(
		r,
			"</td>\n"
			"\t<th>%%</th>\n"
			"\t<th>Requests</th>\n"
			"\t<th>Refused</th>\n"
			"\t<th>KBytes<br>Sent</th>\n"
			"\t<th>KBytes<br>per hit</th>\n"
			"\t<th>Delay<br>(&lt;=%d)</th>\n"
			"\t<th>Period<br>Elapsed</th>\n"
			"\t<th>Idle<br>Time</th>\n"
		"</tr>\n",
		max_delay
	);
}

static void
general_html_line(request_rec *r, t_visitor *v, int row)
{
	int i, percent;
	t_config *config;
	const char *url, *visitor, *view, *level;

#ifdef THROTTLE_CLIENT_IP
	if (r->handler == throttle_client_ip_str) {
		config = &client_ip_config;
		visitor = inet_ntoa(v->remote.ip);
		view = view_client_ip;
	}
#endif
#ifdef THROTTLE_REMOTE_USER
	if (r->handler == throttle_remote_user_str) {
		config = &remote_user_config;
		visitor = v->remote.user;
		view = view_remote_user;
	}
#endif

	/* Compute the policy specific percentage of the limit. */
	config->track = (t_throttle *) v;
	percent = (*config->policy->percent)(config);

	for (i = 0; i < ALERT_LEVELS-1; ++i) {
		if (percent < alert[i])
			break;
	}
	level = alert_names[i];

	/* Start table row, alternating background colour. */
	ap_rprintf(
		r, "<tr align=\"right\"%s>\n",
		(row & 1) ? " bgcolor=\"#eeeeff\"" : ""
	);

	/* Display row number. */
	url = ap_psprintf(
		r->pool, "%s?content-type=text/html&refresh=%s&view=%s&command=reset,%s",
		r->uri, ap_table_get(r->notes, "refresh"), view, visitor
	);
	url = ap_construct_url(r->pool, url, r);
	ap_rprintf(r, "<td class=\"normal\"><a href=\"%s\">%d.</a>&nbsp;</td>\n", url, row);

	ap_rprintf(r, "<td align=\"left\" class=\"normal\">%s</td>\n", visitor);

	/* Display stats. */
	ap_rprintf(
		r,
		"<td class=\"%s\">%u</td>\n"

		"<td class=\"%s\">%lu</td>\n"
		"<td class=\"%s\">%lu</td>\n"
		"<td class=\"%s\">%lu</td>\n"
		"<td class=\"%s\">%lu</td>\n"
		"<td class=\"%s\">%u</td>\n"

		"<td class=\"%s\">%s</td>\n"
		"<td class=\"%s\">%s</td>\n",
		level, percent,

		level, v->requests,
		level, v->refused,
		level, v->volume,
		level, 0 < v->requests ? v->volume / v->requests : 0,
		level, v->delay,

		level, elapsed_time(r->pool, r->request_time - v->start),
		level, elapsed_time(r->pool, r->request_time - v->last)
	);
}

static void
general_text_line(request_rec *r, t_visitor *v, int row)
{
	t_config *config;
	const char *visitor;

#ifdef THROTTLE_CLIENT_IP
	if (r->handler == throttle_client_ip_str) {
		visitor = inet_ntoa(v->remote.ip);
		config = &client_ip_config;
	}
#endif
#ifdef THROTTLE_REMOTE_USER
	if (r->handler == throttle_remote_user_str) {
		visitor = v->remote.user;
		config = &remote_user_config;
	}
#endif
	config->track = (t_throttle *) v;

	ap_rprintf(
		r,
		"%u. %s"
		"\t%u"

		"\t%lu"
		"\t%lu"
		"\t%lu"
		"\t%u"

		"\t%s"
		"\t%s"
		"\r\n",
		row,
		visitor,
		config->policy->percent(config),

		v->requests,
		v->refused,
		v->volume,
		v->delay,

		elapsed_time(r->pool, r->request_time - v->start),
		elapsed_time(r->pool, r->request_time - v->last)
	);
}

static int
visitor_status(request_rec *r)
{
	int rc, row = 0;
	t_visitor *v;
	t_visitors *vp;

	if (!ap_is_initial_req(r))
		return DECLINED;

	if ((rc = ap_discard_request_body(r)) != OK)
		return rc;

	ap_send_http_header(r);

	if (r->header_only)
		return OK;

#ifdef THROTTLE_CLIENT_IP
	if (r->handler == throttle_client_ip_str) {
		vp = client_ip_pool;
	}
#endif
#ifdef THROTTLE_REMOTE_USER
	if (r->handler == throttle_remote_user_str) {
		vp = remote_user_pool;
	}
#endif

	if (r->content_type == text_plain) {
		for (v = vp->head; v != (t_visitor *) 0; v = v->next) {
			if (vp->used <= row)
				break;
			general_text_line(r, v, ++row);
		}
	} else {
		general_html_header(r);
		for (v = vp->head; v != (t_visitor *) 0; v = v->next) {
			if (vp->used <= row)
				break;
			general_html_line(r, v, ++row);
		}
		status_html_footer(r);
	}

	return OK;
}
#endif

static handler_rec content_handlers[] = {
	{ throttle_me_str, me_status },
	{ throttle_info_str, server_status },
	{ throttle_status_str, server_status },
#ifdef THROTTLE_CLIENT_IP
	{ throttle_client_ip_str, visitor_status },
#endif
#ifdef THROTTLE_REMOTE_USER
	{ throttle_remote_user_str, visitor_status },
#endif
	{ NULL }
};

static command_rec command_table[] = {
#ifdef THROTTLE_CLIENT_IP
	{ "ThrottleClientIP", throttle_client_ip, NULL, RSRC_CONF, RAW_ARGS,
	"Specify global throttle pool size, policy, limit, and period for all client IP connections." },
#endif
	{ "ThrottleContentType", throttle_content_type, NULL, RSRC_CONF, TAKE1,
	"Content-Type of throttle display, or use /throttle-status?content-type=text/plain" },

	{ "ThrottleIndicator", throttle_indicator, NULL, RSRC_CONF, TAKE2,
	"Set % threshold for green, yellow, red, and critical alerts." },

	{ "ThrottleLockFile", throttle_lock_file, NULL, RSRC_CONF, TAKE1,
	"The lock file used with fcntl() or flock() serialization. Must be stored on a local disk." },

	{ "ThrottleMaxDelay", throttle_max_delay, NULL, RSRC_CONF, TAKE1,
	"Set max. delay threshold in seconds before refusing connections." },

	{ "ThrottlePolicy", throttle_policy, NULL, RSRC_CONF|ACCESS_CONF, RAW_ARGS,
	"Select policy, limit, and period." },

	{ "ThrottleRefresh", throttle_refresh, NULL, RSRC_CONF, TAKE1,
	"Refresh time in seconds, or use /throttle-status?refresh=sec" },

#ifdef THROTTLE_REMOTE_USER
	{ "ThrottleRemoteUser", throttle_remote_user, NULL, RSRC_CONF, RAW_ARGS,
	"Specify global throttle pool size, policy, limit, and period for all authenticated remote users." },
#endif
	{ "ThrottleRuntimeFile", throttle_runtime_file, NULL, RSRC_CONF, TAKE1,
	"The runtime data file used across shutdowns and restarts." },

	{ "ThrottleUser", throttle_user, NULL, RSRC_CONF, RAW_ARGS,
	"Specify user to throttle according to policy, limit, and period." },

	{ NULL }
};

module MODULE_VAR_EXPORT throttle_module = {
	STANDARD_MODULE_STUFF,
	init_module,		/* module initializer                  */
	create_dir_config,	/* create per-dir    config structures */
	merge_dir_config,	/* merge  per-dir    config structures */
	NULL,			/* create per-server config structures */
	NULL,			/* merge  per-server config structures */
	command_table,		/* table of config file commands       */
	content_handlers,	/* [#8] MIME-typed-dispatched handlers */
	uri_handler,		/* [#1] URI to filename translation    */
	authentication_handler,	/* [#4] validate user id from request  */
	authorization_handler,	/* [#5] check if the user is ok _here_ */
	access_handler,		/* [#3] check access by host address   */
	mime_handler,		/* [#6] determine MIME type            */
	fixup_handler,		/* [#7] pre-run fixups                 */
	log_handler,		/* [#9] log a transaction              */
#if MODULE_MAGIC_NUMBER >= 19970103
	NULL,			/* [#2] header parser                  */
#endif
#if MODULE_MAGIC_NUMBER >= 19970719
	child_init,		/* child_init			       */
#endif
#if MODULE_MAGIC_NUMBER >= 19970728
	NULL,			/* child_exit			       */
#endif
#if MODULE_MAGIC_NUMBER >= 19970902
	NULL			/* [#0] post read-request              */
#endif
};
