/*************************************************************************************************
 * A command line utility of LZMA
 *************************************************************************************************/


#include <lzmalib.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#undef TRUE
#define TRUE           1
#undef FALSE
#define FALSE          0


/* global variables */
const char *progname;                    /* program name */


/* function prototypes */
int main(int argc, char **argv);
void usage(void);
char *readfile(const char *file, int *sp);


/* main routine */
int main(int argc, char **argv){
  int i, bsiz, rsiz, dec;
  char *file, *buf, *res;
  progname = argv[0];
  dec = FALSE;
  file = NULL;
  for(i = 1; i < argc; i++){
    if(!file && argv[i][0] == '-'){
      if(!strcmp(argv[i], "-d")){
        dec = TRUE;
      } else {
        usage();
      }
    } else if(!file){
      file = argv[i];
    } else {
      usage();
    }
  }
  buf = NULL;
  if(!(buf = readfile(file, &bsiz))){
    fprintf(stderr, "%s: %s: cannot open\n", progname, file ? file : "(stdin)");
    return 1;
  }
  if(dec){
    if(!(res = lzma_decompress(buf, bsiz, &rsiz))){
      fprintf(stderr, "%s: decompression failed\n", progname);
      free(buf);
      return 1;
    }
    for(i = 0; i < rsiz; i++){
      putchar(res[i]);
    }
    lzma_free(res);
  } else {
    if(!(res = lzma_compress(buf, bsiz, &rsiz))){
      fprintf(stderr, "%s: compression failed\n", progname);
      free(buf);
      return 1;
    }
    for(i = 0; i < rsiz; i++){
      putchar(res[i]);
    }
    lzma_free(res);
  }
  free(buf);
  return 0;
}


/* print the usage and exit */
void usage(void){
  fprintf(stderr, "%s: LZMA encoder and decoder\n", progname);
  fprintf(stderr, "\n");
  fprintf(stderr, "usage:\n");
  fprintf(stderr, "  %s [-d] [file]\n", progname);
  fprintf(stderr, "\n");
  exit(1);
}


/* read the standard input */
char *readfile(const char *file, int *sp){
  FILE *ifp;
  char *buf;
  int i, blen, c;
  if(file){
    if(!(ifp = fopen(file, "rb"))) return NULL;
  } else {
    ifp = stdin;
  }
  blen = 256;
  if(!(buf = malloc(blen))){
    if(ifp != stdin) fclose(ifp);
    return NULL;
  }
  for(i = 0; (c = fgetc(ifp)) != EOF; i++){
    if(i >= blen - 1 && !(buf = realloc(buf, blen *= 2))){
      if(ifp != stdin) fclose(ifp);
      return NULL;
    }
    buf[i] = c;
  }
  buf[i] = '\0';
  if(sp) *sp = i;
  if(ifp != stdin) fclose(ifp);
  return buf;
}



/* END OF FILE */
