/*
 * Copyright (c) 2006-2010 Citrix Systems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */


#include <stddef.h>
#include <stdlib.h>

#include "xen_internal.h"
#include <xen/api/xen_common.h>
#include <xen/api/xen_secret.h>


XEN_FREE(xen_secret)
XEN_SET_ALLOC_FREE(xen_secret)
XEN_ALLOC(xen_secret_record)
XEN_SET_ALLOC_FREE(xen_secret_record)
XEN_ALLOC(xen_secret_record_opt)
XEN_RECORD_OPT_FREE(xen_secret)
XEN_SET_ALLOC_FREE(xen_secret_record_opt)


static const struct_member xen_secret_record_struct_members[] =
    {
        { .key = "uuid",
          .type = &abstract_type_string,
          .offset = offsetof(xen_secret_record, uuid) },
        { .key = "value",
          .type = &abstract_type_string,
          .offset = offsetof(xen_secret_record, value) }
    };

const abstract_type xen_secret_record_abstract_type_ =
    {
       .typename = STRUCT,
       .struct_size = sizeof(xen_secret_record),
       .member_count =
           sizeof(xen_secret_record_struct_members) / sizeof(struct_member),
       .members = xen_secret_record_struct_members
    };


void
xen_secret_record_free(xen_secret_record *record)
{
    if (record == NULL)
    {
        return;
    }
    free(record->handle);
    free(record->uuid);
    free(record->value);
    free(record);
}


bool
xen_secret_get_record(xen_session *session, xen_secret_record **result, xen_secret secret)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = secret }
        };

    abstract_type result_type = xen_secret_record_abstract_type_;

    *result = NULL;
    XEN_CALL_("secret.get_record");

    if (session->ok)
    {
       (*result)->handle = xen_strdup_((*result)->uuid);
    }

    return session->ok;
}


bool
xen_secret_get_by_uuid(xen_session *session, xen_secret *result, char *uuid)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = uuid }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("secret.get_by_uuid");
    return session->ok;
}


bool
xen_secret_create(xen_session *session, xen_secret *result, xen_secret_record *record)
{
    abstract_value param_values[] =
        {
            { .type = &xen_secret_record_abstract_type_,
              .u.struct_val = record }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("secret.create");
    return session->ok;
}


bool
xen_secret_destroy(xen_session *session, xen_secret secret)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = secret }
        };

    xen_call_(session, "secret.destroy", param_values, 1, NULL, NULL);
    return session->ok;
}


bool
xen_secret_get_value(xen_session *session, char **result, xen_secret secret)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = secret }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("secret.get_value");
    return session->ok;
}


bool
xen_secret_set_value(xen_session *session, xen_secret secret, char *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = secret },
            { .type = &abstract_type_string,
              .u.string_val = value }
        };

    xen_call_(session, "secret.set_value", param_values, 2, NULL, NULL);
    return session->ok;
}


bool
xen_secret_introduce(xen_session *session, xen_secret *result, char *uuid, char *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = uuid },
            { .type = &abstract_type_string,
              .u.string_val = value }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("secret.introduce");
    return session->ok;
}


bool
xen_secret_get_all(xen_session *session, struct xen_secret_set **result)
{

    abstract_type result_type = abstract_type_string_set;

    *result = NULL;
    xen_call_(session, "secret.get_all", NULL, 0, &result_type, result);
    return session->ok;
}


bool
xen_secret_get_uuid(xen_session *session, char **result, xen_secret secret)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = secret }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("secret.get_uuid");
    return session->ok;
}
