/*
 * Copyright (c) 2006-2010 Citrix Systems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */


#include <stddef.h>
#include <stdlib.h>

#include "xen_internal.h"
#include <xen/api/xen_bond.h>
#include <xen/api/xen_common.h>
#include <xen/api/xen_network.h>
#include <xen/api/xen_pif.h>
#include <xen/api/xen_string_string_map.h>


XEN_FREE(xen_bond)
XEN_SET_ALLOC_FREE(xen_bond)
XEN_ALLOC(xen_bond_record)
XEN_SET_ALLOC_FREE(xen_bond_record)
XEN_ALLOC(xen_bond_record_opt)
XEN_RECORD_OPT_FREE(xen_bond)
XEN_SET_ALLOC_FREE(xen_bond_record_opt)


static const struct_member xen_bond_record_struct_members[] =
    {
        { .key = "uuid",
          .type = &abstract_type_string,
          .offset = offsetof(xen_bond_record, uuid) },
        { .key = "master",
          .type = &abstract_type_ref,
          .offset = offsetof(xen_bond_record, master) },
        { .key = "slaves",
          .type = &abstract_type_ref_set,
          .offset = offsetof(xen_bond_record, slaves) },
        { .key = "other_config",
          .type = &abstract_type_string_string_map,
          .offset = offsetof(xen_bond_record, other_config) }
    };

const abstract_type xen_bond_record_abstract_type_ =
    {
       .typename = STRUCT,
       .struct_size = sizeof(xen_bond_record),
       .member_count =
           sizeof(xen_bond_record_struct_members) / sizeof(struct_member),
       .members = xen_bond_record_struct_members
    };


void
xen_bond_record_free(xen_bond_record *record)
{
    if (record == NULL)
    {
        return;
    }
    free(record->handle);
    free(record->uuid);
    xen_pif_record_opt_free(record->master);
    xen_pif_record_opt_set_free(record->slaves);
    xen_string_string_map_free(record->other_config);
    free(record);
}


bool
xen_bond_get_record(xen_session *session, xen_bond_record **result, xen_bond bond)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = bond }
        };

    abstract_type result_type = xen_bond_record_abstract_type_;

    *result = NULL;
    XEN_CALL_("Bond.get_record");

    if (session->ok)
    {
       (*result)->handle = xen_strdup_((*result)->uuid);
    }

    return session->ok;
}


bool
xen_bond_get_by_uuid(xen_session *session, xen_bond *result, char *uuid)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = uuid }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("Bond.get_by_uuid");
    return session->ok;
}


bool
xen_bond_get_master(xen_session *session, xen_pif *result, xen_bond bond)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = bond }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("Bond.get_master");
    return session->ok;
}


bool
xen_bond_get_slaves(xen_session *session, struct xen_pif_set **result, xen_bond bond)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = bond }
        };

    abstract_type result_type = abstract_type_string_set;

    *result = NULL;
    XEN_CALL_("Bond.get_slaves");
    return session->ok;
}


bool
xen_bond_get_other_config(xen_session *session, xen_string_string_map **result, xen_bond bond)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = bond }
        };

    abstract_type result_type = abstract_type_string_string_map;

    *result = NULL;
    XEN_CALL_("Bond.get_other_config");
    return session->ok;
}


bool
xen_bond_set_other_config(xen_session *session, xen_bond bond, xen_string_string_map *other_config)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = bond },
            { .type = &abstract_type_string_string_map,
              .u.set_val = (arbitrary_set *)other_config }
        };

    xen_call_(session, "Bond.set_other_config", param_values, 2, NULL, NULL);
    return session->ok;
}


bool
xen_bond_add_to_other_config(xen_session *session, xen_bond bond, char *key, char *value)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = bond },
            { .type = &abstract_type_string,
              .u.string_val = key },
            { .type = &abstract_type_string,
              .u.string_val = value }
        };

    xen_call_(session, "Bond.add_to_other_config", param_values, 3, NULL, NULL);
    return session->ok;
}


bool
xen_bond_remove_from_other_config(xen_session *session, xen_bond bond, char *key)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = bond },
            { .type = &abstract_type_string,
              .u.string_val = key }
        };

    xen_call_(session, "Bond.remove_from_other_config", param_values, 2, NULL, NULL);
    return session->ok;
}


bool
xen_bond_create(xen_session *session, xen_bond *result, xen_network network, struct xen_pif_set *members, char *mac)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = network },
            { .type = &abstract_type_string_set,
              .u.set_val = (arbitrary_set *)members },
            { .type = &abstract_type_string,
              .u.string_val = mac }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("Bond.create");
    return session->ok;
}


bool
xen_bond_destroy(xen_session *session, xen_bond self)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self }
        };

    xen_call_(session, "Bond.destroy", param_values, 1, NULL, NULL);
    return session->ok;
}


bool
xen_bond_get_all(xen_session *session, struct xen_bond_set **result)
{

    abstract_type result_type = abstract_type_string_set;

    *result = NULL;
    xen_call_(session, "Bond.get_all", NULL, 0, &result_type, result);
    return session->ok;
}


bool
xen_bond_get_uuid(xen_session *session, char **result, xen_bond bond)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = bond }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("Bond.get_uuid");
    return session->ok;
}
