/*
 * Copyright (c) 2006-2010 Citrix Systems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifndef XEN_ROLE_H
#define XEN_ROLE_H

#include <xen/api/xen_common.h>
#include <xen/api/xen_role_decl.h>
#include <xen/api/xen_string_set.h>


/*
 * The role class.
 * 
 * A set of permissions associated with a subject.
 */


/**
 * Free the given xen_role.  The given handle must have been allocated
 * by this library.
 */
extern void
xen_role_free(xen_role role);


typedef struct xen_role_set
{
    size_t size;
    xen_role *contents[];
} xen_role_set;

/**
 * Allocate a xen_role_set of the given size.
 */
extern xen_role_set *
xen_role_set_alloc(size_t size);

/**
 * Free the given xen_role_set.  The given set must have been allocated
 * by this library.
 */
extern void
xen_role_set_free(xen_role_set *set);


typedef struct xen_role_record
{
    xen_role handle;
    char *uuid;
    char *name_label;
    char *name_description;
    struct xen_role_record_opt_set *subroles;
} xen_role_record;

/**
 * Allocate a xen_role_record.
 */
extern xen_role_record *
xen_role_record_alloc(void);

/**
 * Free the given xen_role_record, and all referenced values.  The
 * given record must have been allocated by this library.
 */
extern void
xen_role_record_free(xen_role_record *record);


typedef struct xen_role_record_opt
{
    bool is_record;
    union
    {
        xen_role handle;
        xen_role_record *record;
    } u;
} xen_role_record_opt;

/**
 * Allocate a xen_role_record_opt.
 */
extern xen_role_record_opt *
xen_role_record_opt_alloc(void);

/**
 * Free the given xen_role_record_opt, and all referenced values.  The
 * given record_opt must have been allocated by this library.
 */
extern void
xen_role_record_opt_free(xen_role_record_opt *record_opt);


typedef struct xen_role_record_set
{
    size_t size;
    xen_role_record *contents[];
} xen_role_record_set;

/**
 * Allocate a xen_role_record_set of the given size.
 */
extern xen_role_record_set *
xen_role_record_set_alloc(size_t size);

/**
 * Free the given xen_role_record_set, and all referenced values.  The
 * given set must have been allocated by this library.
 */
extern void
xen_role_record_set_free(xen_role_record_set *set);



typedef struct xen_role_record_opt_set
{
    size_t size;
    xen_role_record_opt *contents[];
} xen_role_record_opt_set;

/**
 * Allocate a xen_role_record_opt_set of the given size.
 */
extern xen_role_record_opt_set *
xen_role_record_opt_set_alloc(size_t size);

/**
 * Free the given xen_role_record_opt_set, and all referenced values. 
 * The given set must have been allocated by this library.
 */
extern void
xen_role_record_opt_set_free(xen_role_record_opt_set *set);


/**
 * Get a record containing the current state of the given role.
 */
extern bool
xen_role_get_record(xen_session *session, xen_role_record **result, xen_role role);


/**
 * Get a reference to the role instance with the specified UUID.
 */
extern bool
xen_role_get_by_uuid(xen_session *session, xen_role *result, char *uuid);


/**
 * Get all the role instances with the given label.
 */
extern bool
xen_role_get_by_name_label(xen_session *session, struct xen_role_set **result, char *label);


/**
 * Get the uuid field of the given role.
 */
extern bool
xen_role_get_uuid(xen_session *session, char **result, xen_role role);


/**
 * Get the name/label field of the given role.
 */
extern bool
xen_role_get_name_label(xen_session *session, char **result, xen_role role);


/**
 * Get the name/description field of the given role.
 */
extern bool
xen_role_get_name_description(xen_session *session, char **result, xen_role role);


/**
 * Get the subroles field of the given role.
 */
extern bool
xen_role_get_subroles(xen_session *session, struct xen_role_set **result, xen_role role);


/**
 * This call returns a list of permissions given a role.
 */
extern bool
xen_role_get_permissions(xen_session *session, struct xen_role_set **result, xen_role self);


/**
 * This call returns a list of permission names given a role.
 */
extern bool
xen_role_get_permissions_name_label(xen_session *session, struct xen_string_set **result, xen_role self);


/**
 * This call returns a list of roles given a permission.
 */
extern bool
xen_role_get_by_permission(xen_session *session, struct xen_role_set **result, xen_role permission);


/**
 * This call returns a list of roles given a permission name.
 */
extern bool
xen_role_get_by_permission_name_label(xen_session *session, struct xen_role_set **result, char *label);


/**
 * Return a list of all the roles known to the system.
 */
extern bool
xen_role_get_all(xen_session *session, struct xen_role_set **result);


#endif
