/*  Programmable IRC function library 
    Copyright (C) 1999-2000 Jorgen Sigvardsson <jorgen@cs.kau.se>
    $Id: irc_core.c,v 1.6 2000/01/06 14:18:18 jorgen Exp $
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/signal.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>

#include "irc_core.h"
#include "tcp.h"
#include "cmd.h"

#include "macros.h"
#include "macros_priv.h"

/* Local variables */
static GSList*		servers		= NULL;
static errhandler_t	errorHandler	= NULL;
static GTree*		callbacks	= NULL;

/* Local functions prototypes */
static gint handle_connection(srv_h srv);
static void execute_errhandler(srv_h srv, gint errcode);
static void parse_message(srv_h srv, gstring msg);

/* Exported functions */
srv_h
irc_connect(	gcstring	server,
		gushort		port,
		gcstring	nick,
		gcstring	user,
		gcstring	fullname )
{
    srv_h srv;
    int fd;
    char buf[129];

    /* Connect to server */
    if((fd = tcp_connect(server, port)) < 0)
	return NULL;

    srv = (srv_h)g_new(struct _srv, 1);
    srv->fd = fd;
    
    /* Send login information */
    irc_write_cmd(srv, CMD_NICK, nick);
    g_snprintf(buf, 128, "%s %s %s :%s", 
	     user, "js-pc.cs.kau.se", server, fullname);
    irc_write_cmd(srv, CMD_USER, buf);

    return srv;
}

void
irc_disconnect( srv_h server )
{
    close(server->fd);
    g_free(server);
}

void
irc_write_cmd( srv_h server,
	       gcstring cmd,
	       gcstring params )
{
    write(server->fd, cmd, strlen(cmd));
    write(server->fd, " ", 1);
    write(server->fd, params, strlen(params));
    write(server->fd, "\r\n", 2);
}

void
irc_write_raw( srv_h server, 
	       gcstring rawtext )
{
    write(server->fd, rawtext, strlen(rawtext));
    write(server->fd, "\r\n", 2);
}

void 
irc_set_callback( callback_t	cb,
		  gcstring	cmd )
{
    if(callbacks == NULL)
	callbacks = g_tree_new((GCompareFunc)strcmp);
    g_tree_insert(callbacks, (gstring)cmd, cb);
}

void
irc_set_errhandler( errhandler_t	errh ) 

{
    errorHandler = errh;
}

void
irc_add_server_listener( srv_h server )
{
    servers = g_slist_append(servers, server);
}

void
irc_remove_server_listener( srv_h server )
{
    servers = g_slist_remove(servers, server);
}

gint
irc_getfdset( fd_set* set )
{
    gint maxfd;
    GSList* i;
    srv_h srv;
    
    FD_ZERO(set);
    maxfd = -1;
    for(i = servers; i != NULL; i = g_slist_next(i)) {
	srv = (srv_h)i->data;
	FD_SET(srv->fd, set);
	if(srv->fd > maxfd)
	    maxfd = srv->fd;
    }
    return maxfd;
}
    
srv_h
irc_nextsrv( fd_set* set )
{
    GSList* i;
    srv_h srv;
    
    for(i = servers; i != NULL; i = g_slist_next(i)) {
	srv = (srv_h)i->data;
	if(FD_ISSET(srv->fd, set)) {
	    FD_CLR(srv->fd, set);
	    return srv;
	}
    }
    return NULL;
}

gint
irc_handle_server( srv_h server )
{
    return handle_connection(server);
}



/* IRC Main loop - mostly for development purposes.. 
   Serious developers will use the the irc_fetfdset(), irc_nextsrv() and
   irc_context() in combination with Xt/Gtk/Qt socket/event
   handling mechanisms
*/
void
irc_main( void ) 
{
    fd_set set;
    srv_h srv;
    int maxfd;
    
    while(1) {
	/* If we are not listening to any servers anymore, quit. */
	if(servers == NULL) 
	    return;
	maxfd = irc_getfdset(&set);
	
	while(1) {
	    if(select(maxfd + 1, &set, 0, 0, 0) < 0) {
		if(errno != EINTR) {
		    perror("select");
		    return;
		}
	    }
	    else
		break;
	}
	
	while((srv = irc_nextsrv(&set)))
	    irc_handle_server(srv);
    }
}

/* Local functions */
static gint handle_connection(srv_h srv)
{
    gchar buf[MAX_LINE_LEN];
    int curb;
    int bread;
    
    curb = 0;
    while(1) {
	if((bread = read(srv->fd, buf + curb, 1)) < 1) {
	    execute_errhandler(srv, (bread == 0 ? ERR_EOF : ERR_CONN_BROKEN));
	    return -1;
	}
	if(buf[curb] == '\n')
	    break;
	curb++;

	/* Make sure buffer does not exceed capacity */
	if(curb >= MAX_LINE_LEN - 1) {
	    execute_errhandler(srv, ERR_BUFFER_OVERRUN);
	    return -1;
	}
    }
    buf[curb+1] = '\0'; /* NUL-terminate */
    parse_message(srv, buf);
    return 0;
}

static void execute_errhandler(srv_h srv, gint errcode)
{
    ERROR(srv, errcode);
}

static guint calc_params(gstring msg)
{
    guint num;
    
    num = 1; /* There is always one parameter */
    while(*msg && *msg != ':')
    {
	if(*msg == ' ') { num++; while(*msg == ' ') msg++; }
	else msg++;
    }

    return num;
}

static void parse_params(gstring msg, gstring* arr)
{
    while(*msg && *msg != ':') {
	if(*msg == ' ') {
	    while(*msg == ' ') {
		*msg = '\0';
		msg++;
	    }
	} else {
	    *arr = msg;
	    arr++;
	    while(*msg != ' ') msg++;
	}
    }
    if(*msg == ':')
	*arr = msg + 1;
}

static void parse_message(srv_h srv, gstring msg)
{
    gstring arr_params[MAX_PARAMS];
    guint num_params;
    gcstring prefix;
    gcstring cmd;
    gstring i;
    callback_t cb;

    /* If there are no callbacks, then processing the
       message is unnecessary */
    if(callbacks == NULL)
	return;
    
    /* Strip all newline crap at the end of the message */
    for(i = msg + strlen(msg) - 1; *i == '\r' || *i == '\n'; i--)
	*i = '\0';

    /* If message starts with ':' -> it has a prefix */
    if(*msg == ':') {
	prefix = msg + 1;
	/* Wipe out all spaces trailing the prefix */
	for(i = msg + 1; *i != ' '; i++);
	for(; *i == ' '; i++)
	    *i = '\0';
	msg = i;
    } else {
	prefix = NULL;
    }
    cmd = msg;
    for(i = msg; *i != ' ' && *i; i++);
    for(; *i == ' '; i++)
	*i = '\0';

    /* Now we now what command it is.. lets see if
       there is a callback for it. */
    if((cb = g_tree_lookup(callbacks, (gpointer)cmd)) == NULL)
	return;

    msg = i;
    num_params = calc_params(msg);
    parse_params(msg, arr_params);
    cb(srv, prefix, cmd, arr_params, num_params);    
}
