/*  Programmable IRC function library 
    Copyright (C) 1999-2000 Jorgen Sigvardsson <jorgen@cs.kau.se>
    $Id: ctcp.c,v 1.5 2000/01/06 14:18:18 jorgen Exp $
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <string.h>
#include <stdarg.h>

#include "irc_core.h"
#include "ctcp.h"
#include "macros_priv.h"
#include "cmd.h"

/* Local variables */
static GTree* callbacks = NULL;
static gchar buf[MAX_LINE_LEN]; /* Local 'utility buffer' */

/* Local functions */
static gchar* compose( gchar* fmt, ... )
{
    va_list ap;
    
    va_start(ap, fmt);
    g_vsnprintf(buf, MAX_LINE_LEN, fmt, ap);
    va_end(ap);
    return buf;
}

static void
ctcp_write_raw( srv_h srv,
		gcstring to, 
		gboolean isRequest,
		gcstring cmd,
		gcstring args )
{
    gstring ctcp_cmd = compose("%c%s %s%c", '\001', cmd, args, '\001');
    if(isRequest)
	irc_do_privmsg(srv, to, ctcp_cmd);
    else
	irc_do_notice(srv, to, ctcp_cmd);
}

/* Exported functions */
void
ctcp_write_error( srv_h srv,
		  gcstring to, 
		  gcstring cmd,
		  gcstring arg )
{
    gstring err_cmd = compose("%s %s", cmd, arg);
    ctcp_write_raw(srv, to, FALSE, CMD_CTCP_ERRMSG, err_cmd);
}

void
ctcp_set_callback( ctcp_callback_t	cb,
		   gcstring		cmd )
{
    if(callbacks == NULL)
	callbacks = g_tree_new((GCompareFunc)strcmp);
    g_tree_insert(callbacks, (gstring)cmd, cb);
}

void ctcp_dispatch_msg( srv_h srv,
			gcstring from,
			gboolean isRequest,
			gstring line )
{
    gchar* i;
    gstring cmd;
    gstring param;
    ctcp_callback_t cb;

    /* If there are no callbacks, then processing the
       message is unnecessary */
    if(callbacks == NULL)
	return;

    cmd = line + 1;
    for(i = cmd; *i != ' ' && *i != '\001'; i++);
    *i = '\0';
    i++;

    if((cb = g_tree_lookup(callbacks, (gpointer)cmd)) == NULL)
	return;

    param = i;
    /* Remove the last CTCP marker */
    while(*i != '\001' && *i != '\0') i++;
    *i = '\0';

    cb(srv, from, isRequest, cmd, param);
}

void ctcp_write_req( srv_h srv, 
		     gcstring to,
		     gcstring cmd,
		     gcstring args )
{
    ctcp_write_raw(srv, to, TRUE, cmd, args);
}

void ctcp_write_resp( srv_h srv,
		      gcstring to,
		      gcstring cmd,
		      gcstring args )
{
    ctcp_write_raw(srv, to, FALSE, cmd, args);
}

void ctcp_do_action( srv_h srv,
		     gcstring to,
		     gcstring action )
{
    ctcp_write_req(srv, to, CMD_CTCP_ACTION, action);
}

void ctcp_req_version( srv_h srv, 
		       gcstring to )
{
    ctcp_write_req(srv, to, CMD_CTCP_VERSION, "");
}

void ctcp_req_ping( srv_h srv, 
		    gcstring to,
		    gcstring arg )
{
    ctcp_write_req(srv, to, CMD_CTCP_PING, arg);
}

void ctcp_req_cliinfo( srv_h srv, 
		       gcstring to )
{
    ctcp_write_req(srv, to, CMD_CTCP_CLIENTINFO, "");
}

void ctcp_req_userinfo( srv_h srv, 
			gcstring to )
{
    ctcp_write_req(srv, to, CMD_CTCP_USERINFO, "");
}

void ctcp_req_time( srv_h srv, 
		    gcstring to )
{
    ctcp_write_req(srv, to, CMD_CTCP_TIME, "");
}

void ctcp_resp_version( srv_h srv, 
			gcstring to,
			gcstring version )
{
    ctcp_write_resp(srv, to, CMD_CTCP_VERSION, version);
}

void ctcp_resp_ping( srv_h srv, 
		     gcstring to,
		     gcstring arg )
{
    ctcp_write_resp(srv, to, CMD_CTCP_PONG, arg);
}

void ctcp_resp_cliinfo( srv_h srv, 
			gcstring to,
			gcstring info )
{
    ctcp_write_resp(srv, to, CMD_CTCP_CLIENTINFO, info);
}

void ctcp_resp_userinfo( srv_h srv, 
			 gcstring to,
			 gcstring info )
{
    ctcp_write_resp(srv, to, CMD_CTCP_USERINFO, info);
}

void ctcp_resp_time( srv_h srv, 
		     gcstring to,
		     gcstring time )
{
    ctcp_write_resp(srv, to, CMD_CTCP_TIME, time);
}
