/**
 * @node libcomprex/fsnode.c FsNodesystem node structures
 * 
 * $Id: fsnode.c,v 1.5 2003/01/01 06:22:36 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libcomprex/internal.h>
#include <libcomprex/fsnode.h>

CxFsNode *
cxNewFsNode(void)
{
	CxFsNode *node;

	MEM_CHECK(node = (CxFsNode *)malloc(sizeof(CxFsNode)));
	memset(node, 0, sizeof(CxFsNode));

	cxSetFsNodeType(node, CX_FSNODETYPE_UNKNOWN);

	CX_LINK(node);

	return node;
}

void
cxDestroyFsNode(CxFsNode *node)
{
	if (node == NULL || node->refCount == 0)
		return;

	CX_UNLINK(node);

	if (node->refCount > 0)
		return;

	if (node->name != NULL) free(node->name);
	if (node->path != NULL) free(node->path);

	free(node);
}

void
cxSetFsNodeArchive(CxFsNode *node, CxArchive *archive)
{
	if (node == NULL)
		return;

#if 0
	if (node->archive != NULL && node->archive->refCount > 0)
		cxDestroyArchive(node->archive);
#endif

	node->archive = archive;

#if 0
	if (archive != NULL)
	{
		CX_LINK(archive);
	}
#endif
}

void
cxSetFsNodeParent(CxFsNode *node, CxDirectory *parent)
{
	if (node == NULL)
		return;

	node->parentDir = parent;
}

void
cxSetFsNodeName(CxFsNode *node, const char *name)
{
	if (node == NULL)
		return;

	if (node->name != NULL)
		free(node->name);

	node->name = (name == NULL ? NULL : strdup(name));
}

void
cxSetFsNodeType(CxFsNode *node, CxFsNodeType type)
{
	if (node == NULL)
		return;

	node->type = type;
}

void
cxSetFsNodeMode(CxFsNode *node, mode_t mode)
{
	if (node == NULL)
		return;

	node->mode = mode;
}

void
cxSetFsNodeUid(CxFsNode *node, uid_t uid)
{
	if (node == NULL)
		return;

	node->uid = uid;
}

void
cxSetFsNodeGid(CxFsNode *node, gid_t gid)
{
	if (node == NULL)
		return;

	node->gid = gid;
}

void
cxSetFsNodeDate(CxFsNode *node, time_t date)
{
	if (node == NULL)
		return;

	node->date = date;
}

void
cxSetFsNodeLocal(CxFsNode *node, char isLocalFsNode)
{
	if (node == NULL)
		return;
	
	node->isLocal = (isLocalFsNode == 0 ? 0 : 1);
}

CxArchive *
cxGetFsNodeArchive(CxFsNode *node)
{
	if (node == NULL)
		return NULL;

	return node->archive;
}

CxDirectory *
cxGetFsNodeParent(CxFsNode *node)
{
	if (node == NULL)
		return NULL;

	return node->parentDir;
}

const char *
cxGetFsNodeName(CxFsNode *node)
{
	if (node == NULL)
		return NULL;

	return node->name;
}

const char *
cxGetFsNodePath(CxFsNode *node)
{
	int len = 0;
	const char *parentPath, *nodeName;
	char *path;

	if (node == NULL)
		return NULL;

	if ((nodeName = cxGetFsNodeName(node)) == NULL)
		return NULL;

	len += strlen(nodeName);

	if ((parentPath = cxGetFsNodePath(cxGetFsNodeParent(node))) != NULL)
	{
		len += strlen(parentPath);
		
		if (strcmp(parentPath, "/"))
			len++;
	}

	MEM_CHECK(path = (char *)malloc(len + 1));

	if (parentPath != NULL)
	{
		snprintf(path, len + 1, "%s/%s",
				 (!strcmp(parentPath, "/") ? "" : parentPath),
				 nodeName);
	}
	else
	{
		path = strdup(nodeName);
	}

	return path;
}

CxFsNodeType
cxGetFsNodeType(CxFsNode *node)
{
	if (node == NULL)
		return CX_FSNODETYPE_UNKNOWN;

	return node->type;
}

mode_t
cxGetFsNodeMode(CxFsNode *node)
{
	if (node == NULL)
		return 0;
	
	return node->mode;
}

uid_t
cxGetFsNodeUid(CxFsNode *node)
{
	if (node == NULL)
		return 0; /* ??? */

	return node->uid;
}

gid_t
cxGetFsNodeGid(CxFsNode *node)
{
	if (node == NULL)
		return 0; /* ??? */

	return node->gid;
}

time_t
cxGetFsNodeDate(CxFsNode *node)
{
	if (node == NULL)
		return 0;

	return node->date;
}

char
cxIsFsNodeLocal(CxFsNode *node)
{
	if (node == NULL)
		return 0;

	return node->isLocal;
}

CxFsNode *
cxGetNextFsNode(CxFsNode *node)
{
	if (node == NULL)
		return NULL;

	return node->next;
}

CxFsNode *
cxGetPreviousFsNode(CxFsNode *node)
{
	if (node == NULL)
		return NULL;

	return node->prev;
}

