/**
 * @file libcomprex/fsiter.c Filesystem iteration API
 * 
 * $Id: fsiter.c,v 1.8 2003/01/01 06:22:36 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libcomprex/internal.h>

static CxFsNode *
__findFirstNode(CxFsNode *baseNode, CxFsNodeType type)
{
	CxFsNode *node;

	for (node = baseNode; node != NULL; node = node->next)
	{
		if (type == CX_FSNODETYPE_UNKNOWN || cxGetFsNodeType(node) == type)
			return node;

		if (cxGetFsNodeType(node) == CX_FSNODETYPE_DIRECTORY)
		{
			CxFsNode *tempNode = __findFirstNode(node->u.dir->children, type);

			if (tempNode != NULL)
				return tempNode;
		}
	}

	return NULL;
}

static CxFsNode *
__findNextNode(CxFsNode *baseNode, CxFsNodeType type)
{
	CxFsNode *node = NULL;

	while (baseNode != NULL && node == NULL)
	{
		if (cxGetFsNodeType(baseNode) == CX_FSNODETYPE_DIRECTORY)
			node = baseNode->u.dir->children;
		
		if (node == NULL)
			node = cxGetNextFsNode(baseNode);

		/* Descend down the directories. */
		while (node == NULL && baseNode != NULL)
		{
			baseNode = cxGetFsNodeParent(baseNode);

			if (cxGetNextFsNode(baseNode) != NULL)
			{
				baseNode = cxGetNextFsNode(baseNode);
				node = baseNode;
			}
		}

		if (node != NULL &&
			(type == CX_FSNODETYPE_UNKNOWN || cxGetFsNodeType(node) == type))
		{
			return node;
		}
		else
			node = NULL; /* Continue on. */
	}

	return NULL;
}

CxFsIterator *
cxNewFsIterator(CxArchive *archive, CxFsIteratorType type)
{
	CxFsIterator *iterator;

	if (archive == NULL)
		return NULL;

	MEM_CHECK(iterator = (CxFsIterator *)malloc(sizeof(CxFsIterator)));
	memset(iterator, 0, sizeof(CxFsIterator));

	iterator->type    = type;
	iterator->archive = archive;

	return iterator;
}

void
cxDestroyFsIterator(CxFsIterator *iterator)
{
	if (iterator == NULL)
		return;

	free(iterator);
}

CxFsNode *
cxGetFsIterFirst(CxFsIterator *iterator)
{
	CxDirectory *root;

	if (iterator == NULL)
		return NULL;

	root = cxGetArchiveRoot(iterator->archive);

	if (iterator->type == CX_FSITER_DIRS)
	{
		iterator->lastNode = __findFirstNode(root->u.dir->children,
											CX_FSNODETYPE_DIRECTORY);
	}
	else if (iterator->type == CX_FSITER_FILES)
	{
		iterator->lastNode = __findFirstNode(root->u.dir->children,
											CX_FSNODETYPE_FILE);
	}
	else if (iterator->type == CX_FSITER_FILES_DIRS)
	{
		iterator->lastNode = root->u.dir->children;
	}

	return iterator->lastNode;
}

CxFsNode *
cxGetFsIterNext(CxFsIterator *iterator)
{
	if (iterator == NULL)
		return NULL;

	if (iterator->type == CX_FSITER_DIRS)
	{
		iterator->lastNode = __findNextNode(iterator->lastNode,
											CX_FSNODETYPE_DIRECTORY);
	}
	else if (iterator->type == CX_FSITER_FILES)
	{
		iterator->lastNode = __findNextNode(iterator->lastNode,
											CX_FSNODETYPE_FILE);
	}
	else if (iterator->type == CX_FSITER_FILES_DIRS)
	{
		iterator->lastNode = __findNextNode(iterator->lastNode,
											CX_FSNODETYPE_UNKNOWN);
	}

	return iterator->lastNode;
}

CxFsNode *
cxGetFsIterPrev(CxFsIterator *iterator)
{
	if (iterator == NULL)
		return NULL;

	return NULL;
}

