/**
 * @file libcomprex/directory.c Directory structures
 * 
 * $Id: directory.c,v 1.39 2003/01/01 06:22:35 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libcomprex/internal.h>

static CxDirectory *
__getExistingPart(CxDirectory *base, const char *path, char **invalidPart)
{
	CxDirectory *dir = NULL;
	char *newPath;
	const char *p, *p2;

	if (base == NULL || path == NULL || *path == '\0')
		return NULL;

	newPath = cxFixPath(path);
	p = newPath;

	if (invalidPart != NULL)
		*invalidPart = NULL;

	if (*p == '/')
	{
		/* Proceed to the root directory. */
		dir = cxGetArchiveRoot(cxGetDirArchive(base));

		p++;
	}
	else
		dir = base;

	for (; p != NULL && *p != '\0'; p = (p2 == NULL ? NULL : p2 + 1))
	{
		char *dirname;
		int len;

		/* Find the start of the next dir, if any, and NUL-terminate it. */
		p2 = strchr(p, '/');

		if (p2 != NULL)
			len = p2 - p;
		else
		{
			len = strlen(p);
		}

		/* See if we're going down a level. */
		if (!strncmp(p, "..", 2) && (*(p + 2) == '/' || *(p + 2) == '\0'))
		{
			if (cxGetDirParent(dir) != NULL)
				dir = cxGetDirParent(dir);

			continue;
		}

		/* Get this path. */
		MEM_CHECK(dirname = (char *)malloc(len + 1));

		strncpy(dirname, p, len);
		dirname[len] = '\0';

		if (strncmp(dirname, ".", 2))
		{
			CxDirectory *tempDir;

			/* Try to see if this exists. */
			for (tempDir = cxGetFirstSubDir(dir);
				 tempDir != NULL && strcmp(cxGetDirName(tempDir), dirname);
				 tempDir = cxGetNextDir(tempDir))
				;

			if (tempDir == NULL)
			{
				free(dirname);

				break;
			}

			/*
			 * On the next iteration, we'll be working with this
			 * new-found directory.
			 */
			dir = tempDir;
		}

		free(dirname);
	}

	if (invalidPart != NULL && p != NULL && *p != '\0')
		*invalidPart = strdup(p);

	free(newPath);

	return dir;
}

CxDirectory *
cxNewDirectory(void)
{
	CxDirectory *dir;

	dir = (CxDirectory *)cxNewFsNode();

	MEM_CHECK(dir->u.dir = (CxDirectoryData *)malloc(sizeof(CxDirectoryData)));
	memset(dir->u.dir, 0, sizeof(CxDirectoryData));

	cxSetFsNodeType(dir, CX_FSNODETYPE_DIRECTORY);

	return dir;
}

void
cxDestroyDirectory(CxDirectory *dir)
{
	CxFile *curFile, *nextFile;
	CxDirectory *curDir, *nextDir;

	if (dir == NULL || !CX_IS_DIRECTORY(dir) || dir->refCount == 0)
		return;

	if ((dir->refCount - 1) > 0)
		return;

	for (curFile = cxGetFirstFile(dir);
		 curFile != NULL;
		 curFile = nextFile)
	{
		nextFile = cxGetNextFile(curFile);

		cxDestroyFile(curFile);
	}

	for (curDir = cxGetFirstSubDir(dir);
		 curDir != NULL;
		 curDir = nextDir)
	{
		nextDir = cxGetNextDir(curDir);

		cxDestroyDirectory(curDir);
	}

	free(dir->u.dir);

	cxDestroyFsNode(dir);
}

void
cxSetDirArchive(CxDirectory *dir, CxArchive *archive)
{
	cxSetFsNodeArchive(dir, archive);
}

void
cxSetDirParent(CxDirectory *dir, CxDirectory *parent)
{
	cxSetFsNodeParent(dir, parent);
}

void
cxSetDirName(CxDirectory *dir, const char *name)
{
	cxSetFsNodeName(dir, name);
}

void
cxSetDirPhysicalPath(CxDirectory *dir, const char *path)
{
	if (dir == NULL)
		return;

	if (dir->u.dir->physPath != NULL)
		free(dir->u.dir->physPath);

	dir->u.dir->physPath = (path == NULL ? NULL : strdup(path));
}

void
cxSetDirMode(CxDirectory *dir, mode_t mode)
{
	cxSetFsNodeMode(dir, mode);
}

void
cxSetDirUid(CxDirectory *dir, uid_t uid)
{
	cxSetFsNodeUid(dir, uid);
}

void
cxSetDirGid(CxDirectory *dir, gid_t gid)
{
	cxSetFsNodeGid(dir, gid);
}

void
cxSetDirDate(CxDirectory *dir, time_t date)
{
	cxSetFsNodeDate(dir, date);
}

void
cxSetDirLocal(CxDirectory *dir, char isLocal)
{
	cxSetFsNodeLocal(dir, isLocal);
}

		
CxArchive *
cxGetDirArchive(CxDirectory *dir)
{
	return cxGetFsNodeArchive(dir);
}

CxDirectory *
cxGetDirParent(CxDirectory *dir)
{
	return cxGetFsNodeParent(dir);
}

const char *
cxGetDirName(CxDirectory *dir)
{
	return cxGetFsNodeName(dir);
}

const char *
cxGetDirPath(CxDirectory *dir)
{
	return cxGetFsNodePath(dir);
}

const char *
cxGetDirPhysicalPath(CxDirectory *dir)
{
	if (dir == NULL)
		return NULL;

	return dir->u.dir->physPath;
}

mode_t
cxGetDirMode(CxDirectory *dir)
{
	return cxGetFsNodeMode(dir);
}

uid_t
cxGetDirUid(CxDirectory *dir)
{
	return cxGetFsNodeUid(dir);
}

gid_t
cxGetDirGid(CxDirectory *dir)
{
	return cxGetFsNodeGid(dir);
}

time_t
cxGetDirDate(CxDirectory *dir)
{
	return cxGetFsNodeDate(dir);
}

char
cxIsDirLocal(CxDirectory *dir)
{
	return cxIsFsNodeLocal(dir);
}

unsigned int
cxGetFileCount(CxDirectory *dir)
{
	if (dir == NULL || !CX_IS_DIRECTORY(dir))
		return 0;

	return dir->u.dir->fileCount;
}

unsigned int
cxGetSubDirCount(CxDirectory *dir)
{
	if (dir == NULL || !CX_IS_DIRECTORY(dir))
		return 0;

	return dir->u.dir->subdirCount;
}

CxDirectory *
cxGetDirectory(CxDirectory *base, const char *path)
{
	CxDirectory *dir;
	char *remaining;

	if (base == NULL || !CX_IS_DIRECTORY(base) ||
		path == NULL || *path == '\0')
	{
		return NULL;
	}

	dir = __getExistingPart(base, path, &remaining);

	/* Make sure the ENTIRE path exists! */
	if (remaining == NULL)
		return dir;

	free(remaining);

	return NULL;
}

CxFile *
cxGetFile(CxDirectory *base, const char *path)
{
	CxDirectory *dir  = NULL;
	CxFile      *file = NULL;
	char *temp;

	if (base == NULL || !CX_IS_DIRECTORY(base) ||
		path == NULL || *path == '\0')
	{
		return NULL;
	}

	temp = cxGetBasePath(path);

	if (temp != NULL)
	{
		dir = cxGetDirectory(base, temp);
		free(temp);
	}

	if (dir == NULL)
		dir = base;
	
	/* Now we should have the directory. Get the filename. */
	temp = cxGetBaseName(path);

	if (temp == NULL || *temp == '\0')
	{
		/*
		 * No base name. Oops, error on the user's part, most likely.
		 *
		 * Either an out of memory, or a trailing '/'
		 */

		if (temp != NULL)
			free(temp);

		return NULL;
	}

	/* Find the file. */
	for (file = cxGetFirstFile(dir);
		 file != NULL && strcmp(cxGetFileName(file), temp);
		 file = cxGetNextFile(file))
		;

	free(temp);

	return file;
}

CxDirectory *
cxMkDir(CxDirectory *base, const char *path)
{
	CxDirectory *dir = NULL, *newDir;
	char *newPath;
	char *p, *p2;

	if (base == NULL || !CX_IS_DIRECTORY(base) ||
		path == NULL || *path == '\0')
	{
		return NULL;
	}

	dir = __getExistingPart(base, path, &newPath);

	if (newPath == NULL)
		return dir; /* XXX NULL? This means the directory already exists! */

	for (p = newPath;
		 p != NULL && *p != '\0';
		 p = (p2 == NULL ? NULL : p2 + 1))
	{
		/* Find the start of the next dir, if any, and NUL-terminate it. */
		p2 = strchr(p, '/');

		if (p2 != NULL)
			*p2 = '\0';

		/* Create the directory and add it. */
		newDir = cxNewDirectory();
		cxSetDirName(newDir, p);

		cxDirAddSubDir(dir, newDir);

		dir = newDir;
	}

	free(newPath);

	return dir;
}

static void
__dirAddChild(CxDirectory *dir, CxFsNode *node)
{
	if (dir == NULL || node == NULL)
		return;

	if (dir->u.dir->children == NULL)
		dir->u.dir->children = node;

	node->prev = dir->u.dir->lastChild;

	if (dir->u.dir->lastChild != NULL)
		dir->u.dir->lastChild->next = node;

	dir->u.dir->lastChild = node;

	cxSetFsNodeArchive(node, cxGetDirArchive(dir));
	cxSetFsNodeParent(node,  dir);
}

void
cxDirAddFile(CxDirectory *dir, CxFile *file)
{
	CxArchive *archive;

	if (dir == NULL || !CX_IS_DIRECTORY(dir) || file == NULL)
		return;

	__dirAddChild(dir, file);

	archive = cxGetDirArchive(dir);

	dir->u.dir->fileCount++;
	archive->fileCount++;

	cxSetArchiveSize(archive, cxGetArchiveSize(archive) + cxGetFileSize(file));
}

void
cxDirAddSubDir(CxDirectory *dir, CxDirectory *subdir)
{
	if (dir == NULL || subdir == NULL ||
		!CX_IS_DIRECTORY(dir) || !CX_IS_DIRECTORY(subdir))
	{
		return;
	}

	__dirAddChild(dir, subdir);

	dir->u.dir->subdirCount++;
}

static void
__dirRemoveChild(CxDirectory *dir, CxFsNode *node)
{
	if (dir == NULL || node == NULL || cxGetFsNodeParent(node) != dir)
		return;

	if (node->prev == NULL)
		dir->u.dir->children = node->next;
	else
		node->prev->next = node->next;

	if (node->next == NULL)
		dir->u.dir->lastChild = node->prev;
	else
		node->next->prev = node->prev;

	cxSetFsNodeArchive(node, NULL);

	cxDestroyFsNode(node);
}

void
cxDirRemoveFile(CxDirectory *dir, CxFile *file)
{
	CxArchive *archive;

	if (dir == NULL || file == NULL || !CX_IS_DIRECTORY(dir))
		return;

	__dirRemoveChild(dir, file);

	archive = cxGetDirArchive(dir);

	dir->u.dir->fileCount--;
	archive->fileCount--;

	cxSetArchiveSize(archive, cxGetArchiveSize(archive) - cxGetFileSize(file));
}

void
cxDirRemoveSubDir(CxDirectory *dir, CxDirectory *subdir)
{
	if (dir == NULL || subdir == NULL ||
		!CX_IS_DIRECTORY(dir) || !CX_IS_DIRECTORY(subdir))
		return;

	__dirRemoveChild(dir, subdir);

	dir->u.dir->subdirCount--;
}

CxFile *
cxGetFirstFile(CxDirectory *dir)
{
	CxFile *file;

	if (dir == NULL || !CX_IS_DIRECTORY(dir))
		return NULL;

	for (file = dir->u.dir->children;
		 file != NULL && cxGetFsNodeType(file) != CX_FSNODETYPE_FILE;
		 file = file->next)
		;

	return file;
}

CxDirectory *
cxGetFirstSubDir(CxDirectory *dir)
{
	CxDirectory *subdir;

	if (dir == NULL || !CX_IS_DIRECTORY(dir))
		return NULL;

	for (subdir = dir->u.dir->children;
		 subdir != NULL && !CX_IS_DIRECTORY(subdir);
		 subdir = subdir->next)
		;

	return subdir;
}

CxDirectory *
cxGetPreviousDir(CxDirectory *dir)
{
	if (dir == NULL || !CX_IS_DIRECTORY(dir))
		return NULL;

	for (dir = dir->prev;
		 dir != NULL && !CX_IS_DIRECTORY(dir);
		 dir = dir->prev)
		;

	return dir;
}

CxDirectory *
cxGetNextDir(CxDirectory *dir)
{
	if (dir == NULL || !CX_IS_DIRECTORY(dir))
		return NULL;

	for (dir = dir->next;
		 dir != NULL && !CX_IS_DIRECTORY(dir);
		 dir = dir->next)
		;

	return dir;
}

